#!/bin/bash
set -e

TEAM_ID="AP4CTZ4AQ4"
BUNDLE_ID="com.pdcompany.yarnai"

cd "${1:-.}"
echo "=== YarnAI iOS Build (Fully Automated) ==="

# Step 1: Remove and recreate iOS project fresh (auto-select certificate 1)
echo ">>> Creating fresh iOS project..."
rm -rf ios
yes 1 | flutter create . --platforms=ios --org com.pdcompany --project-name yarnai 2>/dev/null || true

# Step 2: Restore app icons
echo ">>> Restoring app icons..."
cp -f tooling/app_icons/*.png ios/Runner/Assets.xcassets/AppIcon.appiconset/ 2>/dev/null || true
cp -f tooling/app_icons/Contents.json ios/Runner/Assets.xcassets/AppIcon.appiconset/ 2>/dev/null || true

# Step 3: Add privacy usage descriptions to Info.plist
echo ">>> Adding privacy usage descriptions..."
/usr/libexec/PlistBuddy -c "Add :NSMicrophoneUsageDescription string 'YarnAI uses your microphone so you can speak your questions and have conversations with the AI assistant.'" ios/Runner/Info.plist 2>/dev/null || /usr/libexec/PlistBuddy -c "Set :NSMicrophoneUsageDescription 'YarnAI uses your microphone so you can speak your questions and have conversations with the AI assistant.'" ios/Runner/Info.plist
/usr/libexec/PlistBuddy -c "Add :NSCameraUsageDescription string 'YarnAI uses your camera to take photos of letters and documents so the app can help you understand them.'" ios/Runner/Info.plist 2>/dev/null || /usr/libexec/PlistBuddy -c "Set :NSCameraUsageDescription 'YarnAI uses your camera to take photos of letters and documents so the app can help you understand them.'" ios/Runner/Info.plist
/usr/libexec/PlistBuddy -c "Add :NSSpeechRecognitionUsageDescription string 'YarnAI uses speech recognition to turn your voice into text so you can talk naturally with the app.'" ios/Runner/Info.plist 2>/dev/null || /usr/libexec/PlistBuddy -c "Set :NSSpeechRecognitionUsageDescription 'YarnAI uses speech recognition to turn your voice into text so you can talk naturally with the app.'" ios/Runner/Info.plist
/usr/libexec/PlistBuddy -c "Add :ITSAppUsesNonExemptEncryption bool false" ios/Runner/Info.plist 2>/dev/null || /usr/libexec/PlistBuddy -c "Set :ITSAppUsesNonExemptEncryption false" ios/Runner/Info.plist
/usr/libexec/PlistBuddy -c "Add :LSRequiresIPhoneOS bool true" ios/Runner/Info.plist 2>/dev/null || true
/usr/libexec/PlistBuddy -c "Add :UIRequiredDeviceCapabilities array" ios/Runner/Info.plist 2>/dev/null || true
/usr/libexec/PlistBuddy -c "Add :UIRequiredDeviceCapabilities:0 string armv7" ios/Runner/Info.plist 2>/dev/null || true
echo "   Privacy strings added"

# Step 3b: Set display name to YarnAI
/usr/libexec/PlistBuddy -c "Add :CFBundleDisplayName string 'YarnAI'" ios/Runner/Info.plist 2>/dev/null || /usr/libexec/PlistBuddy -c "Set :CFBundleDisplayName 'YarnAI'" ios/Runner/Info.plist
/usr/libexec/PlistBuddy -c "Set :CFBundleName 'YarnAI'" ios/Runner/Info.plist 2>/dev/null || true
echo "   Display name set to YarnAI"

# Step 3c: Add Privacy Manifest (required by Apple since iOS 17)
echo ">>> Adding Privacy Manifest..."
cp -f tooling/PrivacyInfo.xcprivacy ios/Runner/PrivacyInfo.xcprivacy 2>/dev/null || true
echo "   Privacy manifest added"

# Step 4: Fix Podfile - set iOS 13.0 minimum
echo ">>> Fixing Podfile for iOS 13.0..."
cat > ios/Podfile << 'PODFILE'
platform :ios, '13.0'

ENV['COCOAPODS_DISABLE_STATS'] = 'true'

project 'Runner', {
  'Debug' => :debug,
  'Profile' => :release,
  'Release' => :release,
}

def flutter_root
  generated_xcode_build_settings_path = File.expand_path(File.join('..', 'Flutter', 'Generated.xcconfig'), __FILE__)
  unless File.exist?(generated_xcode_build_settings_path)
    raise "#{generated_xcode_build_settings_path} must exist. If you're running pod install manually, make sure flutter pub get is executed first"
  end
  File.foreach(generated_xcode_build_settings_path) do |line|
    matches = line.match(/FLUTTER_ROOT\=(.*)/)
    return matches[1].strip if matches
  end
  raise "FLUTTER_ROOT not found in #{generated_xcode_build_settings_path}. Try deleting Generated.xcconfig, then run flutter pub get"
end

require File.expand_path(File.join('packages', 'flutter_tools', 'bin', 'podhelper'), flutter_root)

flutter_ios_podfile_setup

target 'Runner' do
  use_frameworks!
  use_modular_headers!
  flutter_install_all_ios_pods File.dirname(File.realpath(__FILE__))
end

post_install do |installer|
  installer.pods_project.targets.each do |target|
    flutter_additional_ios_build_settings(target)
    target.build_configurations.each do |config|
      config.build_settings['IPHONEOS_DEPLOYMENT_TARGET'] = '13.0'
    end
  end
end
PODFILE

# Step 5: Fix signing with Ruby (xcodeproj gem)
echo ">>> Configuring code signing for team $TEAM_ID..."
ruby << 'RUBY'
require 'xcodeproj'
project = Xcodeproj::Project.open('ios/Runner.xcodeproj')
project.targets.each do |target|
  next unless target.name == 'Runner'
  target.build_configurations.each do |config|
    config.build_settings['DEVELOPMENT_TEAM'] = 'AP4CTZ4AQ4'
    config.build_settings['CODE_SIGN_STYLE'] = 'Automatic'
    config.build_settings.delete('CODE_SIGN_IDENTITY')
    config.build_settings['PRODUCT_BUNDLE_IDENTIFIER'] = 'com.pdcompany.yarnai'
    config.build_settings['IPHONEOS_DEPLOYMENT_TARGET'] = '13.0'
  end
end
project.save
puts "   Team ID configured"
RUBY

# Step 6: Fix CocoaPods xcconfig integration  
echo ">>> Fixing Pods xcconfig..."
echo '#include? "Pods/Target Support Files/Pods-Runner/Pods-Runner.release.xcconfig"' >> ios/Flutter/Release.xcconfig
echo '#include? "Pods/Target Support Files/Pods-Runner/Pods-Runner.debug.xcconfig"' >> ios/Flutter/Debug.xcconfig

# Step 7: Install pods
echo ">>> Installing pods..."
cd ios && pod install && cd ..

# Step 8: Build archive only (skip IPA export to avoid crash)
echo ">>> Building archive..."
flutter clean
flutter pub get
mkdir -p build/ios/ipa
flutter build ios --release

# Step 9: Create archive with xcodebuild
echo ">>> Creating archive..."
xcodebuild -workspace ios/Runner.xcworkspace \
  -scheme Runner \
  -configuration Release \
  -archivePath build/ios/archive/Runner.xcarchive \
  archive \
  CODE_SIGN_STYLE=Automatic \
  DEVELOPMENT_TEAM=$TEAM_ID \
  -quiet

# Step 10: Open archive in Xcode
echo ">>> Opening archive in Xcode..."
open build/ios/archive/Runner.xcarchive

echo ""
echo "=========================================="
echo "SUCCESS! Archive opened in Xcode Organizer"
echo "=========================================="
echo ""
echo "To upload to TestFlight:"
echo "1. Window > Organizer (if not already open)"
echo "2. Select YarnAI archive"
echo "3. Click 'Distribute App'"
echo "4. Choose 'App Store Connect' > 'Upload'"
echo ""
