import 'package:flutter/material.dart';
import 'package:flutter_dotenv/flutter_dotenv.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:hive_flutter/hive_flutter.dart';
import 'core/theme/app_theme.dart';
import 'core/services/router.dart';
import 'core/services/preferences_service.dart';
import 'core/services/elevenlabs_service.dart';

Future<void> main() async {
  WidgetsFlutterBinding.ensureInitialized();
  
  // CRITICAL: Run app immediately, no blocking
  runApp(
    ProviderScope(
      child: YarnAIApp(),
    ),
  );
  
  // Initialize services in background after first frame
  WidgetsBinding.instance.addPostFrameCallback((_) async {
    await _initializeServices();
  });
}

/// Background service initialization (non-blocking to UI)
Future<void> _initializeServices() async {
  try {
    debugPrint('🔥 YarnAI: Starting background initialization...');
    
    // Load environment
    await dotenv.load(fileName: ".env");
    debugPrint('✅ Environment loaded');
    
    // Initialize local storage
    await Hive.initFlutter();
    debugPrint('✅ Hive initialized');
    
    // Initialize preferences
    final prefs = PreferencesService();
    await prefs.initialize();
    debugPrint('✅ Preferences initialized');
    
    // Initialize voice service
    final voice = ElevenLabsService();
    await voice.initialize();
    voice.setPersona(prefs.voicePersona);
    debugPrint('✅ Voice service initialized');
    
    debugPrint('🎉 YarnAI: All services initialized successfully');
  } catch (e) {
    debugPrint('❌ Service initialization error: $e');
    // App still works, just without some features
  }
}

class YarnAIApp extends StatefulWidget {
  const YarnAIApp({super.key});

  @override
  State<YarnAIApp> createState() => _YarnAIAppState();
}

class _YarnAIAppState extends State<YarnAIApp> {
  bool _isDarkMode = true;
  bool _hasSeenWelcome = false;
  bool _isInitialized = false;
  
  @override
  void initState() {
    super.initState();
    _loadPreferences();
  }
  
  /// Load preferences asynchronously (doesn't block UI)
  Future<void> _loadPreferences() async {
    // Wait for PreferencesService to be ready
    int attempts = 0;
    while (attempts < 50) { // Max 5 seconds
      try {
        final prefs = PreferencesService();
        if (prefs.isInitialized) {
          setState(() {
            _isDarkMode = prefs.isDarkMode;
            _hasSeenWelcome = prefs.hasSeenWelcome;
            _isInitialized = true;
          });
          
          PreferencesService.onThemeChanged = (isDark) {
            if (mounted) {
              setState(() => _isDarkMode = isDark);
            }
          };
          return;
        }
      } catch (e) {
        // Service not ready yet
      }
      
      await Future.delayed(const Duration(milliseconds: 100));
      attempts++;
    }
    
    // Timeout - use defaults
    debugPrint('⚠️ Preferences loading timed out, using defaults');
    setState(() => _isInitialized = true);
  }
  
  @override
  void dispose() {
    PreferencesService.onThemeChanged = null;
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    // Show minimal loading screen while preferences load
    if (!_isInitialized) {
      return MaterialApp(
        debugShowCheckedModeBanner: false,
        theme: AppTheme.darkTheme,
        home: Scaffold(
          backgroundColor: const Color(0xFF1A1A2E), // Deep navy
          body: Center(
            child: Column(
              mainAxisAlignment: MainAxisAlignment.center,
              children: [
                Container(
                  width: 160,
                  height: 160,
                  decoration: BoxDecoration(
                    borderRadius: BorderRadius.circular(32),
                    boxShadow: [
                      BoxShadow(
                        color: const Color(0xFFCC6B49).withOpacity(0.3),
                        blurRadius: 30,
                        spreadRadius: 5,
                      ),
                    ],
                  ),
                  child: ClipRRect(
                    borderRadius: BorderRadius.circular(32),
                    child: Image.asset(
                      'assets/images/app_logo.png',
                      fit: BoxFit.cover,
                      errorBuilder: (context, error, stackTrace) => const Icon(
                        Icons.local_fire_department,
                        size: 80,
                        color: Color(0xFFCC6B49),
                      ),
                    ),
                  ),
                ),
                const SizedBox(height: 24),
                const CircularProgressIndicator(
                  color: Color(0xFFCC6B49), // Ochre
                  strokeWidth: 2,
                ),
              ],
            ),
          ),
        ),
      );
    }
    
    return MaterialApp.router(
      title: 'YarnAI',
      debugShowCheckedModeBanner: false,
      theme: AppTheme.lightTheme,
      darkTheme: AppTheme.darkTheme,
      themeMode: _isDarkMode ? ThemeMode.dark : ThemeMode.light,
      routerConfig: createRouter(_hasSeenWelcome),
      builder: (context, child) {
        return GestureDetector(
          onTap: () {
            // Dismiss keyboard when tapping outside text fields
            FocusManager.instance.primaryFocus?.unfocus();
          },
          child: child,
        );
      },
    );
  }
}
