import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:speech_to_text/speech_to_text.dart' as stt;
import '../../../core/theme/app_theme.dart';
import '../../../core/widgets/shared_widgets.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/widgets/ceremonial_fire.dart';

class StoryCatcherPage extends StatefulWidget {
  const StoryCatcherPage({super.key});

  @override
  State<StoryCatcherPage> createState() => _StoryCatcherPageState();
}

class _StoryCatcherPageState extends State<StoryCatcherPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final TextEditingController _textController = TextEditingController();
  final stt.SpeechToText _speech = stt.SpeechToText();
  
  bool _isListening = false;
  bool _speechAvailable = false;
  bool _hasStartedYarn = false;
  String _recognizedText = '';
  
  final List<String> _prompts = [
    "What's on your mind today?",
    "Take your time. I'm here.",
    "Is there more you want to share?",
    "How's that been affecting things?",
  ];
  int _currentPromptIndex = 0;

  @override
  void initState() {
    super.initState();
    _initVoice();
    _initSpeech();
  }

  Future<void> _initVoice() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted && _prefs.autoPlayVoice) {
      await Future.delayed(const Duration(milliseconds: 500));
      _voiceService.speak(
        "This is your space to yarn. Tell me what's happening - "
        "speak or type, whatever feels right. I'm listening."
      );
    }
  }

  Future<void> _initSpeech() async {
    _speechAvailable = await _speech.initialize(
      onStatus: (status) {
        if (status == 'done' || status == 'notListening') {
          if (mounted) {
            setState(() => _isListening = false);
          }
        }
      },
      onError: (error) {
        if (mounted) {
          setState(() => _isListening = false);
        }
      },
    );
    setState(() {});
  }

  @override
  void dispose() {
    _voiceService.stop();
    _speech.stop();
    _textController.dispose();
    super.dispose();
  }

  void _toggleListening() async {
    if (_isListening) {
      await _speech.stop();
      setState(() => _isListening = false);
    } else {
      if (_speechAvailable) {
        setState(() {
          _isListening = true;
          _hasStartedYarn = true;
        });
        
        await _speech.listen(
          onResult: (result) {
            setState(() {
              _recognizedText = result.recognizedWords;
              if (result.finalResult) {
                _textController.text = _textController.text.isEmpty
                    ? _recognizedText
                    : '${_textController.text} $_recognizedText';
                _recognizedText = '';
                
                if (_currentPromptIndex < _prompts.length - 1) {
                  _currentPromptIndex++;
                }
              }
            });
          },
          localeId: 'en_AU',
          listenMode: stt.ListenMode.dictation,
        );
      }
    }
  }

  void _onTextChanged() {
    if (!_hasStartedYarn && _textController.text.isNotEmpty) {
      setState(() => _hasStartedYarn = true);
    }
  }

  void _proceedToMeaningMaker() {
    final story = _textController.text.trim();
    if (story.isEmpty) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text("Tell me a bit about what's going on first")),
      );
      return;
    }
    
    _voiceService.stop();
    context.go('/yarn-flow/meaning-maker', extra: {'story': story});
  }

  void _goBack() {
    _voiceService.stop();
    context.go('/');
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              _buildHeader(isDark),
              Expanded(
                child: ScrollableWithIndicator(
                  padding: const EdgeInsets.all(20),
                  child: Column(
                    children: [
                      const SizedBox(height: 20),
                      _buildPrompt(isDark),
                      const SizedBox(height: 30),
                      _buildFireListener(),
                      const SizedBox(height: 30),
                      _buildTextInput(isDark),
                      if (_hasStartedYarn) ...[
                        const SizedBox(height: 24),
                        _buildContinueButton(isDark),
                      ],
                    ],
                  ),
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildHeader(bool isDark) {
    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 8),
      child: Row(
        children: [
          IconButton(
            icon: const Icon(Icons.arrow_back),
            onPressed: _goBack,
            tooltip: 'Back to home',
          ),
          const SizedBox(width: 8),
          Expanded(
            child: Text(
              "Tell me your story",
              style: Theme.of(context).textTheme.titleLarge,
            ),
          ),
          IconButton(
            icon: const Icon(Icons.volume_up),
            onPressed: () {
              _voiceService.speak(
                "This is where you share what's happening. "
                "Tap the fire to speak, or type below. "
                "When you're ready, tap Continue."
              );
            },
            tooltip: 'Read aloud',
          ),
        ],
      ),
    );
  }

  Widget _buildPrompt(bool isDark) {
    return AnimatedSwitcher(
      duration: const Duration(milliseconds: 300),
      child: Text(
        _prompts[_currentPromptIndex],
        key: ValueKey(_currentPromptIndex),
        style: Theme.of(context).textTheme.headlineSmall?.copyWith(
          color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
        ),
        textAlign: TextAlign.center,
      ),
    );
  }

  Widget _buildFireListener() {
    return Column(
      children: [
        CeremonialFire(
          size: 180,
          isListening: _isListening,
          onTap: _speechAvailable ? _toggleListening : null,
        ),
        const SizedBox(height: 16),
        Text(
          _isListening 
              ? "I'm listening..." 
              : (_speechAvailable ? "Tap the fire to speak" : "Type your story below"),
          style: Theme.of(context).textTheme.bodyMedium?.copyWith(
            color: _isListening ? AppColors.primary : AppColors.textTertiary,
            fontWeight: _isListening ? FontWeight.w600 : FontWeight.normal,
          ),
        ),
        if (_recognizedText.isNotEmpty) ...[
          const SizedBox(height: 12),
          Container(
            padding: const EdgeInsets.all(12),
            decoration: BoxDecoration(
              color: AppColors.ochre.withOpacity(0.1),
              borderRadius: BorderRadius.circular(12),
            ),
            child: Text(
              _recognizedText,
              style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                fontStyle: FontStyle.italic,
              ),
              textAlign: TextAlign.center,
            ),
          ),
        ],
      ],
    );
  }

  Widget _buildTextInput(bool isDark) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          "Or type here",
          style: Theme.of(context).textTheme.labelMedium?.copyWith(
            color: isDark ? AppColors.textTertiaryDark : AppColors.textTertiary,
          ),
        ),
        const SizedBox(height: 8),
        TextField(
          controller: _textController,
          maxLines: 5,
          minLines: 3,
          onChanged: (_) => _onTextChanged(),
          decoration: InputDecoration(
            hintText: "What's been going on? Tell me in your own words...",
            hintStyle: TextStyle(
              color: isDark ? AppColors.textTertiaryDark : AppColors.textTertiary,
            ),
          ),
        ),
      ],
    );
  }

  Widget _buildContinueButton(bool isDark) {
    return SizedBox(
      width: double.infinity,
      child: ElevatedButton.icon(
        onPressed: _proceedToMeaningMaker,
        icon: const Icon(Icons.arrow_forward),
        label: const Text("Continue"),
      ),
    );
  }
}
