import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/widgets/shared_widgets.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';

class MeaningMakerPage extends StatefulWidget {
  final String story;
  
  const MeaningMakerPage({super.key, required this.story});

  @override
  State<MeaningMakerPage> createState() => _MeaningMakerPageState();
}

class _MeaningMakerPageState extends State<MeaningMakerPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _apiClient = ApiClient();
  
  bool _isAnalyzing = true;
  String _reflection = '';
  List<_JourneyPath> _suggestedPaths = [];

  @override
  void initState() {
    super.initState();
    _initVoice();
    _analyzeStory();
  }

  Future<void> _initVoice() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
  }

  Future<void> _analyzeStory() async {
    try {
      final data = await _apiClient.post(
        '/api/yarn-flow/analyze',
        {'story': widget.story},
      );
      
      setState(() {
        _reflection = data['reflection'] ?? "I hear you. That's a lot to carry.";
        _suggestedPaths = (data['paths'] as List? ?? [])
            .map((p) => _JourneyPath(
                  id: p['id'] ?? 'general',
                  title: p['title'] ?? 'Have a yarn',
                  description: p['description'] ?? 'Let\'s talk it through',
                  icon: _getIconForPath(p['id'] ?? 'general'),
                  route: p['route'] ?? '/chat',
                ))
            .toList();
        _isAnalyzing = false;
      });
      
      if (_prefs.autoPlayVoice && _reflection.isNotEmpty) {
        _voiceService.speak(_reflection);
      }
    } catch (e) {
      setState(() {
        _isAnalyzing = false;
        _reflection = "I hear you. Sounds like there's a bit going on.";
        _suggestedPaths = [
          _JourneyPath(
            id: 'chat',
            title: 'Let\'s yarn about it',
            description: 'Talk through what\'s happening together',
            icon: Icons.chat_bubble_outline,
            route: '/chat',
          ),
        ];
      });
      
      if (_prefs.autoPlayVoice) {
        _voiceService.speak(_reflection);
      }
    }
  }

  IconData _getIconForPath(String id) {
    switch (id) {
      case 'housing':
        return Icons.home_outlined;
      case 'money':
      case 'centrelink':
        return Icons.account_balance_wallet_outlined;
      case 'school':
      case 'education':
        return Icons.school_outlined;
      case 'job':
      case 'work':
        return Icons.work_outline;
      case 'health':
        return Icons.favorite_outline;
      case 'crisis':
      case 'safety':
        return Icons.shield_outlined;
      case 'call':
        return Icons.phone_outlined;
      case 'letter':
      case 'document':
        return Icons.description_outlined;
      default:
        return Icons.chat_bubble_outline;
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    super.dispose();
  }

  void _selectPath(_JourneyPath path) {
    _voiceService.stop();
    
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Alright, let's sort that out.");
    }
    
    context.go(path.route, extra: {'story': widget.story, 'journeyId': path.id});
  }

  void _goBack() {
    _voiceService.stop();
    context.go('/yarn-flow/story-catcher');
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              _buildHeader(isDark),
              Expanded(
                child: _isAnalyzing
                    ? _buildLoading(isDark)
                    : _buildContent(isDark),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildHeader(bool isDark) {
    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 8),
      child: Row(
        children: [
          IconButton(
            icon: const Icon(Icons.arrow_back),
            onPressed: _goBack,
            tooltip: 'Back to story',
          ),
          const SizedBox(width: 8),
          Expanded(
            child: Text(
              "What I'm hearing",
              style: Theme.of(context).textTheme.titleLarge,
            ),
          ),
          IconButton(
            icon: const Icon(Icons.volume_up),
            onPressed: () {
              if (_reflection.isNotEmpty) {
                _voiceService.speak(_reflection);
              }
            },
            tooltip: 'Read aloud',
          ),
        ],
      ),
    );
  }

  Widget _buildLoading(bool isDark) {
    return Center(
      child: Column(
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          CircularProgressIndicator(
            color: AppColors.primary,
          ),
          const SizedBox(height: 24),
          Text(
            "Taking in what you've shared...",
            style: Theme.of(context).textTheme.bodyLarge?.copyWith(
              color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildContent(bool isDark) {
    return ScrollableWithIndicator(
      padding: const EdgeInsets.all(20),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          _buildReflection(isDark),
          const SizedBox(height: 32),
          _buildPathsSection(isDark),
          const SizedBox(height: 24),
          _buildAlternative(isDark),
        ],
      ),
    );
  }

  Widget _buildReflection(bool isDark) {
    return Container(
      padding: const EdgeInsets.all(20),
      decoration: BoxDecoration(
        color: (isDark ? AppColors.cardDark : AppColors.cardLight).withOpacity(0.8),
        borderRadius: BorderRadius.circular(16),
        border: Border.all(
          color: AppColors.ochre.withOpacity(0.3),
          width: 2,
        ),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            children: [
              Icon(
                Icons.format_quote,
                color: AppColors.ochre,
                size: 28,
              ),
              const SizedBox(width: 8),
              Text(
                "What I'm hearing",
                style: Theme.of(context).textTheme.titleMedium?.copyWith(
                  color: AppColors.ochre,
                  fontWeight: FontWeight.w600,
                ),
              ),
            ],
          ),
          const SizedBox(height: 16),
          Text(
            _reflection,
            style: Theme.of(context).textTheme.bodyLarge?.copyWith(
              height: 1.6,
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildPathsSection(bool isDark) {
    if (_suggestedPaths.isEmpty) return const SizedBox.shrink();
    
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          "What feels most important right now?",
          style: Theme.of(context).textTheme.titleMedium?.copyWith(
            fontWeight: FontWeight.w600,
          ),
        ),
        const SizedBox(height: 8),
        Text(
          "Pick one, and we'll tackle it together",
          style: Theme.of(context).textTheme.bodyMedium?.copyWith(
            color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
          ),
        ),
        const SizedBox(height: 16),
        ..._suggestedPaths.map((path) => _buildPathCard(path, isDark)),
      ],
    );
  }

  Widget _buildPathCard(_JourneyPath path, bool isDark) {
    return Card(
      margin: const EdgeInsets.only(bottom: 12),
      child: InkWell(
        onTap: () => _selectPath(path),
        borderRadius: BorderRadius.circular(12),
        child: Padding(
          padding: const EdgeInsets.all(16),
          child: Row(
            children: [
              Container(
                padding: const EdgeInsets.all(12),
                decoration: BoxDecoration(
                  color: AppColors.primary.withOpacity(0.15),
                  borderRadius: BorderRadius.circular(12),
                ),
                child: Icon(
                  path.icon,
                  color: AppColors.primary,
                  size: 28,
                ),
              ),
              const SizedBox(width: 16),
              Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      path.title,
                      style: Theme.of(context).textTheme.titleMedium?.copyWith(
                        fontWeight: FontWeight.w600,
                      ),
                    ),
                    const SizedBox(height: 4),
                    Text(
                      path.description,
                      style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                  ],
                ),
              ),
              Icon(
                Icons.arrow_forward_ios,
                color: isDark ? AppColors.textTertiaryDark : AppColors.textTertiary,
                size: 18,
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildAlternative(bool isDark) {
    return Center(
      child: TextButton.icon(
        onPressed: () {
          context.go('/chat', extra: {'story': widget.story});
        },
        icon: const Icon(Icons.chat_bubble_outline),
        label: const Text("Just want to yarn"),
      ),
    );
  }
}

class _JourneyPath {
  final String id;
  final String title;
  final String description;
  final IconData icon;
  final String route;
  
  _JourneyPath({
    required this.id,
    required this.title,
    required this.description,
    required this.icon,
    required this.route,
  });
}
