import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class WelcomeYarnPage extends StatefulWidget {
  const WelcomeYarnPage({super.key});

  @override
  State<WelcomeYarnPage> createState() => _WelcomeYarnPageState();
}

class _WelcomeYarnPageState extends State<WelcomeYarnPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  int _currentStep = 0;
  bool _hasPlayedIntro = false;

  final List<Map<String, String>> _yarnSteps = [
    {
      'title': 'A yarn about new tools',
      'story': '''For thousands of years, our mob have been the cleverest people on Earth at making tools. We made tools to catch fish, to grind seeds, to carry fire across the land. When a new tool comes along, we learn it and make it work for us.

AI is a new kind of tool. Like any tool, it doesn't tell you what to do - you tell it. Let me yarn with you about what this tool is and how it might help.''',
      'voice': "For thousands of years, our mob have been the cleverest people on Earth at making tools. We made tools to catch fish, to grind seeds, to carry fire across the land. When a new tool comes along, we learn it and make it work for us. AI is a new kind of tool. Like any tool, it doesn't tell you what to do. You tell it. Let me yarn with you about what this tool is and how it might help.",
    },
    {
      'title': 'What is AI, really?',
      'story': '''You know how Elders carry knowledge? They've listened to thousands of stories, watched the world for decades, and learned patterns that help them give good advice.

AI is a bit like that - but with computers. It's read millions of books, letters, and conversations. So when you ask it something, it can draw on all that to help you. It's not an Elder - it doesn't have wisdom or Country. But it can be a useful helper.''',
      'voice': "You know how Elders carry knowledge? They've listened to thousands of stories, watched the world for decades, and learned patterns that help them give good advice. AI is a bit like that, but with computers. It's read millions of books, letters, and conversations. So when you ask it something, it can draw on all that to help you. It's not an Elder. It doesn't have wisdom or Country. But it can be a useful helper.",
    },
    {
      'title': 'AI is good at some things',
      'story': '''AI is really good at:

• Explaining confusing letters in simple words
• Helping you write replies to Centrelink, courts, or landlords  
• Answering questions when you're not sure where to start
• Making pictures from descriptions
• Helping prepare for important phone calls

Think of it like having a mate who's good with paperwork, always available when you need to talk something through.''',
      'voice': "AI is really good at explaining confusing letters in simple words. Helping you write replies to Centrelink, courts, or landlords. Answering questions when you're not sure where to start. Making pictures from descriptions. And helping you prepare for important phone calls. Think of it like having a mate who's good with paperwork. Always available when you need to talk something through.",
    },
    {
      'title': 'AI makes mistakes too',
      'story': '''But here's the thing - AI doesn't always get it right. Sometimes it makes stuff up. Sometimes it misses important details. It can't know your situation the way you do.

So always check the important stuff with a real person - especially for legal things, medical things, or money things. Use AI to get started, then talk to someone who knows your situation.''',
      'voice': "But here's the thing. AI doesn't always get it right. Sometimes it makes stuff up. Sometimes it misses important details. It can't know your situation the way you do. So always check the important stuff with a real person. Especially for legal things, medical things, or money things. Use AI to get started, then talk to someone who knows your situation.",
    },
    {
      'title': 'Your story stays yours',
      'story': '''One more thing - what you share with this app stays private. We don't keep your documents after you close the page. We don't share your information with government or anyone else.

This tool is here to help you, not to watch you. You're in control of what you share and when. If something doesn't feel right, you can stop anytime.''',
      'voice': "One more thing. What you share with this app stays private. We don't keep your documents after you close the page. We don't share your information with government or anyone else. This tool is here to help you, not to watch you. You're in control of what you share and when. If something doesn't feel right, you can stop anytime.",
    },
    {
      'title': "What's new in YarnAI",
      'story': '''We've been busy making YarnAI even more helpful! Here's what's new:

• Safety Net - If you're doing it tough, tap to call 13YARN or other crisis lines straight away. Mob talking to mob.
• Money Yarns - Tell me your income and bills, and I'll help you budget.
• Form Filler - I'll walk you through government forms step by step.
• Appointment Helper - Get ready for meetings with Centrelink, housing, doctors, or schools.
• Med Reminder - Set reminders for your tablets, saved private on your phone.
• Housing Helper - Know your rights as a renter and what the landlord has to do.
• School Helper - Understand school letters and write excuse notes.
• Job Spotter - Find work and Indigenous employment programs.
• Story Keeper - A private journal space just for you.''',
      'voice': "We've been busy making YarnAI even more helpful! Safety Net connects you to 13YARN if you're doing it tough. Money Yarns helps with budgeting. Form Filler walks you through government forms. Appointment Helper gets you ready for meetings. Med Reminder keeps your tablets on time. Housing Helper explains your renter rights. School Helper sorts out school paperwork. Job Spotter finds work and Indigenous programs. And Story Keeper is your private journal space.",
    },
    {
      'title': 'Ready to have a go?',
      'story': '''That's the yarn. AI is just a tool - a pretty clever one, but still just a tool. You decide when to use it, what to ask, and whether to trust the answer.

Want to try it out? Have a look around. Ask a question. Paste in a letter. The best way to learn is by doing. And I'll be here if you get stuck.''',
      'voice': "That's the yarn. AI is just a tool. A pretty clever one, but still just a tool. You decide when to use it, what to ask, and whether to trust the answer. Want to try it out? Have a look around. Ask a question. Paste in a letter. The best way to learn is by doing. And I'll be here if you get stuck.",
    },
  ];

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    debugPrint('WelcomeYarnPage: Initializing voice service...');
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playIntro();
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    super.dispose();
  }

  void _playIntro() {
    debugPrint('WelcomeYarnPage: _playIntro called, autoPlay=${_prefs.autoPlayVoice}');
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      _speakCurrentStep();
    }
  }

  void _nextStep() {
    _voiceService.stop();
    if (_currentStep < _yarnSteps.length - 1) {
      setState(() => _currentStep++);
      if (_prefs.autoPlayVoice) {
        Future.delayed(const Duration(milliseconds: 300), _speakCurrentStep);
      }
    } else {
      context.go('/');
    }
  }

  void _prevStep() {
    _voiceService.stop();
    if (_currentStep > 0) {
      setState(() => _currentStep--);
      if (_prefs.autoPlayVoice) {
        Future.delayed(const Duration(milliseconds: 300), _speakCurrentStep);
      }
    }
  }

  void _speakCurrentStep() {
    _voiceService.speak(_yarnSteps[_currentStep]['voice']!);
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final step = _yarnSteps[_currentStep];
    final progress = (_currentStep + 1) / _yarnSteps.length;

    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.close),
          onPressed: () {
            _voiceService.stop();
            context.go('/');
          },
        ),
        title: Text('${_currentStep + 1} of ${_yarnSteps.length}'),
        actions: [
          IconButton(
            icon: const Icon(Icons.volume_up),
            tooltip: 'Listen to this story',
            onPressed: _speakCurrentStep,
          ),
          const SettingsButton(),
        ],
      ),
      body: CulturalBackground(
        child: Column(
        children: [
          LinearProgressIndicator(
            value: progress,
            backgroundColor: isDark ? Colors.grey[800] : Colors.grey[300],
            color: isDark ? AppColors.primaryLight : AppColors.primary,
          ),
          Expanded(
            child: ScrollableWithIndicator(
              padding: const EdgeInsets.all(24),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Row(
                    children: [
                      Expanded(
                        child: Text(
                          step['title']!,
                          style: Theme.of(context).textTheme.headlineSmall?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ),
                      IconButton(
                        icon: Icon(
                          Icons.play_circle_outline,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                          size: 32,
                        ),
                        onPressed: _speakCurrentStep,
                      ),
                    ],
                  ),
                  const SizedBox(height: 24),
                  Text(
                    step['story']!,
                    style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                      height: 1.7,
                    ),
                  ),
                ],
              ),
            ),
          ),
          Padding(
            padding: const EdgeInsets.all(16),
            child: Row(
              children: [
                if (_currentStep > 0)
                  OutlinedButton.icon(
                    onPressed: _prevStep,
                    icon: const Icon(Icons.arrow_back),
                    label: const Text('Back'),
                  )
                else
                  const SizedBox(width: 100),
                const Spacer(),
                Row(
                  mainAxisSize: MainAxisSize.min,
                  children: List.generate(_yarnSteps.length, (i) {
                    return GestureDetector(
                      onTap: () {
                        _voiceService.stop();
                        setState(() => _currentStep = i);
                        if (_prefs.autoPlayVoice) {
                          Future.delayed(const Duration(milliseconds: 300), _speakCurrentStep);
                        }
                      },
                      child: Container(
                        width: 8,
                        height: 8,
                        margin: const EdgeInsets.symmetric(horizontal: 3),
                        decoration: BoxDecoration(
                          shape: BoxShape.circle,
                          color: i == _currentStep
                              ? (isDark ? AppColors.primaryLight : AppColors.primary)
                              : i < _currentStep
                                  ? (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.5)
                                  : Colors.grey.withOpacity(0.3),
                        ),
                      ),
                    );
                  }),
                ),
                const Spacer(),
                ElevatedButton.icon(
                  onPressed: _nextStep,
                  icon: _currentStep == _yarnSteps.length - 1
                      ? const Icon(Icons.check)
                      : const Icon(Icons.arrow_forward),
                  label: Text(_currentStep == _yarnSteps.length - 1 ? "Let's go" : 'Next'),
                ),
              ],
            ),
          ),
        ],
      ),
      ),
    );
  }
}
