import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/voice_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class TrainerHubPage extends StatefulWidget {
  const TrainerHubPage({super.key});

  @override
  State<TrainerHubPage> createState() => _TrainerHubPageState();
}

class _TrainerHubPageState extends State<TrainerHubPage> {
  final String _introText = 
      "Welcome to the Trainer Hub. This is for community workers and trainers "
      "who help others learn to use YarnAI. Here you'll find training resources, "
      "session guides, and tips for running workshops.";

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () => context.go('/settings'),
          tooltip: 'Back to settings',
        ),
        title: const Text('Trainer Hub'),
        actions: [
          IconButton(
            icon: const Icon(Icons.home),
            onPressed: () => context.go('/'),
            tooltip: 'Home',
          ),
          ListenButton(text: _introText, size: 40),
        ],
      ),
      body: ScrollableWithIndicator(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.stretch,
          children: [
            Container(
              padding: const EdgeInsets.all(24),
              decoration: BoxDecoration(
                gradient: LinearGradient(
                  colors: [
                    AppColors.primary.withOpacity(0.8),
                    AppColors.primary,
                  ],
                  begin: Alignment.topLeft,
                  end: Alignment.bottomRight,
                ),
                borderRadius: BorderRadius.circular(16),
              ),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  const Row(
                    children: [
                      Icon(Icons.school, color: Colors.white, size: 32),
                      SizedBox(width: 12),
                      Text(
                        'Trainer Hub',
                        style: TextStyle(
                          color: Colors.white,
                          fontSize: 24,
                          fontWeight: FontWeight.bold,
                        ),
                      ),
                    ],
                  ),
                  const SizedBox(height: 8),
                  const Text(
                    'Resources for community trainers',
                    style: TextStyle(color: Colors.white70),
                  ),
                ],
              ),
            ),
            const SizedBox(height: 24),
            _buildSection(
              icon: Icons.play_circle,
              title: 'Quick Start Guide',
              description: 'Learn the basics of running a YarnAI training session',
              children: [
                _buildGuideStep(1, 'Start with the Call Coach', 'Show how to practice phone calls safely'),
                _buildGuideStep(2, 'Try Explain This', 'Demonstrate document analysis with a sample letter'),
                _buildGuideStep(3, 'Explore voice features', 'Show how to use Aunty/Uncle voices'),
                _buildGuideStep(4, 'Practice together', 'Let participants try features hands-on'),
              ],
            ),
            const SizedBox(height: 16),
            _buildSection(
              icon: Icons.tips_and_updates,
              title: 'Training Tips',
              description: 'Best practices for effective sessions',
              children: [
                _buildTipCard(Icons.group, 'Small groups work best', 'Keep groups to 5-8 people for hands-on help'),
                _buildTipCard(Icons.timer, 'Take breaks', '10 minutes break every 45 minutes'),
                _buildTipCard(Icons.accessibility, 'Be patient', 'Some learners may need extra time'),
                _buildTipCard(Icons.local_cafe, 'Create comfort', 'Tea and snacks help people relax'),
              ],
            ),
            const SizedBox(height: 16),
            _buildSection(
              icon: Icons.folder,
              title: 'Session Materials',
              description: 'Downloadable resources for your workshops',
              children: [
                _buildResourceCard('Training Slide Deck', 'PowerPoint presentation', Icons.slideshow),
                _buildResourceCard('Participant Handout', 'Quick reference guide', Icons.description),
                _buildResourceCard('Sample Documents', 'Practice materials', Icons.folder_copy),
                _buildResourceCard('Feedback Form', 'Session evaluation', Icons.rate_review),
              ],
            ),
            const SizedBox(height: 16),
            _buildSection(
              icon: Icons.help_center,
              title: 'Common Questions',
              description: 'Answers to frequent training queries',
              children: [
                _buildFAQCard(
                  'Do I need internet?',
                  'Basic features work offline, but AI features need connection.',
                ),
                _buildFAQCard(
                  'Is data stored online?',
                  'No. All personal data stays on the device only.',
                ),
                _buildFAQCard(
                  'Can I train elders?',
                  'Yes! Voice features make it accessible for low-literacy users.',
                ),
              ],
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildSection({
    required IconData icon,
    required String title,
    required String description,
    required List<Widget> children,
  }) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                Icon(icon, color: AppColors.primary),
                const SizedBox(width: 8),
                Expanded(
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Text(
                        title,
                        style: const TextStyle(
                          fontSize: 18,
                          fontWeight: FontWeight.bold,
                        ),
                      ),
                      Text(
                        description,
                        style: TextStyle(fontSize: 12, color: Colors.grey[600]),
                      ),
                    ],
                  ),
                ),
              ],
            ),
            const SizedBox(height: 16),
            ...children,
          ],
        ),
      ),
    );
  }

  Widget _buildGuideStep(int number, String title, String description) {
    return Container(
      margin: const EdgeInsets.only(bottom: 12),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Container(
            width: 28,
            height: 28,
            decoration: BoxDecoration(
              color: AppColors.primary,
              borderRadius: BorderRadius.circular(14),
            ),
            child: Center(
              child: Text(
                '$number',
                style: const TextStyle(
                  color: Colors.white,
                  fontWeight: FontWeight.bold,
                ),
              ),
            ),
          ),
          const SizedBox(width: 12),
          Expanded(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(title, style: const TextStyle(fontWeight: FontWeight.w600)),
                Text(description, style: TextStyle(fontSize: 12, color: Colors.grey[600])),
              ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildTipCard(IconData icon, String title, String description) {
    return Container(
      margin: const EdgeInsets.only(bottom: 8),
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        color: Colors.amber.withOpacity(0.1),
        borderRadius: BorderRadius.circular(8),
        border: Border.all(color: Colors.amber.withOpacity(0.3)),
      ),
      child: Row(
        children: [
          Icon(icon, color: Colors.amber[700], size: 24),
          const SizedBox(width: 12),
          Expanded(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(title, style: const TextStyle(fontWeight: FontWeight.w600)),
                Text(description, style: TextStyle(fontSize: 12, color: Colors.grey[600])),
              ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildResourceCard(String title, String type, IconData icon) {
    return Container(
      margin: const EdgeInsets.only(bottom: 8),
      child: ListTile(
        leading: Icon(icon, color: AppColors.primary),
        title: Text(title),
        subtitle: Text(type),
        trailing: IconButton(
          icon: const Icon(Icons.download),
          onPressed: () {
            ScaffoldMessenger.of(context).showSnackBar(
              SnackBar(
                content: Text('Downloading $title...'),
                backgroundColor: AppColors.primary,
              ),
            );
          },
        ),
        shape: RoundedRectangleBorder(
          borderRadius: BorderRadius.circular(8),
          side: BorderSide(color: Colors.grey.withOpacity(0.3)),
        ),
      ),
    );
  }

  Widget _buildFAQCard(String question, String answer) {
    return Container(
      margin: const EdgeInsets.only(bottom: 8),
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        border: Border.all(color: Colors.grey.withOpacity(0.3)),
        borderRadius: BorderRadius.circular(8),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(question, style: const TextStyle(fontWeight: FontWeight.w600)),
          const SizedBox(height: 4),
          Text(answer, style: TextStyle(color: Colors.grey[600])),
        ],
      ),
    );
  }
}
