import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/widgets/shared_widgets.dart';

class SkillsPage extends StatefulWidget {
  const SkillsPage({super.key});

  @override
  State<SkillsPage> createState() => _SkillsPageState();
}

class _SkillsPageState extends State<SkillsPage> with SingleTickerProviderStateMixin {
  late TabController _tabController;
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _apiClient = ApiClient();
  
  bool _isGeneratingResume = false;
  bool _isGeneratingCoverLetter = false;
  bool _isGeneratingPricing = false;
  String? _generatedResume;
  String? _generatedCoverLetter;
  String? _generatedPricing;
  bool _hasPlayedIntro = false;
  
  final TextEditingController _skillsController = TextEditingController();
  final TextEditingController _experienceController = TextEditingController();
  final TextEditingController _jobDescController = TextEditingController();
  final TextEditingController _pricingDescController = TextEditingController();
  
  String? _selectedWorkType;
  
  bool _isListening = false;
  String _activeField = '';
  
  final List<Map<String, dynamic>> _workTypes = [
    {'id': 'art', 'name': 'Art & Craft', 'icon': Icons.palette},
    {'id': 'cleaning', 'name': 'Cleaning', 'icon': Icons.cleaning_services},
    {'id': 'gardening', 'name': 'Gardening', 'icon': Icons.yard},
    {'id': 'repairs', 'name': 'Repairs', 'icon': Icons.build},
    {'id': 'childcare', 'name': 'Childcare', 'icon': Icons.child_care},
    {'id': 'cooking', 'name': 'Cooking', 'icon': Icons.restaurant},
    {'id': 'driving', 'name': 'Driving', 'icon': Icons.directions_car},
    {'id': 'other', 'name': 'Other', 'icon': Icons.more_horiz},
  ];

  @override
  void initState() {
    super.initState();
    _tabController = TabController(length: 3, vsync: this);
    _tabController.addListener(_onTabChanged);
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    debugPrint('SkillsPage: Initializing voice service...');
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playIntro();
    }
  }

  @override
  void dispose() {
    _tabController.removeListener(_onTabChanged);
    _tabController.dispose();
    _skillsController.dispose();
    _experienceController.dispose();
    _jobDescController.dispose();
    _pricingDescController.dispose();
    _voiceService.stop();
    super.dispose();
  }

  void _onTabChanged() {
    if (!_tabController.indexIsChanging && _prefs.autoPlayVoice) {
      _voiceService.stop();
      switch (_tabController.index) {
        case 0:
          _voiceService.speak("Resume tab. Tell me your skills and experience, and I'll build a resume for you.");
          break;
        case 1:
          _voiceService.speak("Cover letter tab. Paste a job ad and I'll write a cover letter that matches your skills.");
          break;
        case 2:
          _voiceService.speak("Pricing tab. Tell me what work you're doing and I'll help you figure out a fair price to charge.");
          break;
      }
    }
  }

  void _playIntro() {
    debugPrint('SkillsPage: _playIntro called, autoPlay=${_prefs.autoPlayVoice}');
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      _voiceService.speak(
        "Work stuff! I can help you build a resume, write a cover letter, or figure out what to charge. "
        "Use the tabs at the top. Tap the mic on any field to speak instead of type."
      );
    }
  }

  Future<void> _startListening(String field, TextEditingController controller) async {
    _voiceService.stop();
    setState(() {
      _isListening = true;
      _activeField = field;
    });
    
    String prompt;
    switch (field) {
      case 'skills':
        prompt = "Tell me what you're good at. What skills do you have?";
        break;
      case 'experience':
        prompt = "Tell me about your work or life experience. What have you done?";
        break;
      case 'jobDesc':
        prompt = "Tell me about the job you're applying for. What does it involve?";
        break;
      case 'pricing':
        prompt = "Describe the work you want to price. What's the job?";
        break;
      default:
        prompt = "I'm listening. Tell me what to write.";
    }
    
    await _voiceService.speakThenListen(
      prompt: prompt,
      onResult: (text) {
        setState(() {
          _isListening = false;
          _activeField = '';
          controller.text = controller.text.isEmpty ? text : '${controller.text} $text';
        });
        if (_prefs.autoPlayVoice && text.isNotEmpty) {
          _voiceService.speak("Got it. Tap the mic to add more, or move on to the next field.");
        }
      },
      onError: (error) {
        setState(() {
          _isListening = false;
          _activeField = '';
        });
        if (_prefs.autoPlayVoice) {
          _voiceService.speak("I couldn't hear that clearly. Tap the mic to try again.");
        }
      },
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () {
            _voiceService.stop();
            context.go('/');
          },
        ),
        title: const Text('Work Stuff'),
        actions: [
          IconButton(
            icon: const Icon(Icons.volume_up),
            tooltip: 'Read page aloud',
            onPressed: () {
              switch (_tabController.index) {
                case 0:
                  _voiceService.speak(
                    _generatedResume != null
                        ? "Your resume is ready. Scroll down to see it, or tap the speaker icon to hear it."
                        : "Fill in your skills and experience, then tap Build My Resume."
                  );
                  break;
                case 1:
                  _voiceService.speak(
                    _generatedCoverLetter != null
                        ? "Your cover letter is ready. Scroll down to see it."
                        : "Paste the job ad or describe the role, then tap Write Cover Letter."
                  );
                  break;
                case 2:
                  _voiceService.speak(
                    _generatedPricing != null
                        ? "Your pricing suggestion is ready. Scroll down to see it, or tap the speaker icon to hear it."
                        : "Pick a work type, describe the job, then tap Get Price Suggestion."
                  );
                  break;
              }
            },
          ),
          const SettingsButton(),
        ],
        bottom: TabBar(
          controller: _tabController,
          tabs: const [
            Tab(text: 'Resume'),
            Tab(text: 'Cover Letter'),
            Tab(text: 'Pricing'),
          ],
        ),
      ),
      body: CulturalBackground(
        child: TabBarView(
          controller: _tabController,
          children: [
            _buildResumeTab(),
            _buildCoverLetterTab(),
            _buildPricingTab(),
          ],
        ),
      ),
    );
  }

  Future<void> _generateResume() async {
    if (_skillsController.text.trim().isEmpty && _experienceController.text.trim().isEmpty) {
      _voiceService.speak("Tell me about your skills or experience first, then I can help build your resume.");
      return;
    }
    
    _voiceService.stop();
    setState(() => _isGeneratingResume = true);
    
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Building your resume. Give me a moment...");
    }
    
    try {
      final result = await _apiClient.post('/api/skills/resume', {
        'name': '',
        'phone': '',
        'email': '',
        'skills': _skillsController.text,
        'experience': _experienceController.text,
        'education': '',
      });
      
      if (result.isNotEmpty) {
        final resume = result['resume'] as String?;
        setState(() {
          _generatedResume = resume;
          _isGeneratingResume = false;
        });
        
        if (resume != null && _prefs.autoPlayVoice) {
          _voiceService.speak("Done! I've put together a resume based on what you told me. Scroll down to see it. Tap the speaker icon to hear it read aloud.");
        }
      }
    } catch (e) {
      setState(() => _isGeneratingResume = false);
      _voiceService.speak("Sorry, something went wrong. Check your internet and try again.");
    }
  }

  Widget _buildResumeTab() {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.all(16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            children: [
              Expanded(
                child: Text(
                  'Your Skills',
                  style: Theme.of(context).textTheme.titleMedium,
                ),
              ),
              if (_skillsController.text.isEmpty)
                TextButton.icon(
                  onPressed: () {
                    _voiceService.speak("What are you good at? Think about things like driving, cooking, caring for people, building, fixing things, or working with computers.");
                  },
                  icon: Icon(Icons.help_outline, size: 16),
                  label: Text('Help', style: TextStyle(fontSize: 12)),
                ),
            ],
          ),
          const SizedBox(height: 8),
          _buildVoiceTextField(
            controller: _skillsController,
            hint: 'What are you good at? Tap mic to speak...',
            field: 'skills',
          ),
          const SizedBox(height: 16),
          Row(
            children: [
              Expanded(
                child: Text(
                  'Your Experience',
                  style: Theme.of(context).textTheme.titleMedium,
                ),
              ),
              if (_experienceController.text.isEmpty)
                TextButton.icon(
                  onPressed: () {
                    _voiceService.speak("What work have you done before? It doesn't have to be paid work. Think about volunteer work, helping family, community work, or any jobs you've had.");
                  },
                  icon: Icon(Icons.help_outline, size: 16),
                  label: Text('Help', style: TextStyle(fontSize: 12)),
                ),
            ],
          ),
          const SizedBox(height: 8),
          _buildVoiceTextField(
            controller: _experienceController,
            hint: 'Tell me about your work or life experience...',
            field: 'experience',
            maxLines: 6,
          ),
          const SizedBox(height: 24),
          SizedBox(
            width: double.infinity,
            child: ElevatedButton(
              onPressed: _isGeneratingResume ? null : _generateResume,
              child: _isGeneratingResume
                  ? Row(
                      mainAxisAlignment: MainAxisAlignment.center,
                      children: [
                        const SizedBox(
                          height: 20,
                          width: 20,
                          child: CircularProgressIndicator(strokeWidth: 2, color: Colors.white),
                        ),
                        const SizedBox(width: 12),
                        const Text('Building...'),
                      ],
                    )
                  : const Text('Build My Resume'),
            ),
          ),
          if (_generatedResume != null) ...[
            const SizedBox(height: 24),
            Card(
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.description, color: isDark ? AppColors.primaryLight : AppColors.primary),
                        const SizedBox(width: 8),
                        Expanded(child: Text('Your Resume', style: Theme.of(context).textTheme.titleMedium)),
                        IconButton(
                          icon: Icon(Icons.volume_up, color: isDark ? AppColors.primaryLight : AppColors.primary),
                          tooltip: 'Read resume aloud',
                          onPressed: () => _voiceService.speak(_generatedResume!),
                        ),
                      ],
                    ),
                    const SizedBox(height: 12),
                    Text(_generatedResume!, style: Theme.of(context).textTheme.bodyMedium),
                    const SizedBox(height: 16),
                    SizedBox(
                      width: double.infinity,
                      child: OutlinedButton.icon(
                        onPressed: () {
                          setState(() => _generatedResume = null);
                          if (_prefs.autoPlayVoice) {
                            _voiceService.speak("Let's try again. Update your skills or experience and tap Build My Resume.");
                          }
                        },
                        icon: const Icon(Icons.refresh),
                        label: const Text('Make Changes'),
                      ),
                    ),
                  ],
                ),
              ),
            ),
          ],
        ],
      ),
    );
  }

  Future<void> _generateCoverLetter() async {
    if (_jobDescController.text.trim().isEmpty) {
      _voiceService.speak("Paste the job ad or tell me about the role first, then I can write your cover letter.");
      return;
    }
    
    _voiceService.stop();
    setState(() => _isGeneratingCoverLetter = true);
    
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Writing your cover letter. Give me a moment...");
    }
    
    try {
      final result = await _apiClient.post('/api/skills/cover-letter', {
        'jobDescription': _jobDescController.text,
        'resumeData': {
          'skills': _skillsController.text,
          'experience': _experienceController.text,
        },
      });
      
      if (result.isNotEmpty) {
        final coverLetter = result['coverLetter'] as String?;
        setState(() {
          _generatedCoverLetter = coverLetter;
          _isGeneratingCoverLetter = false;
        });
        
        if (coverLetter != null && _prefs.autoPlayVoice) {
          _voiceService.speak("Done! I've written a cover letter that matches your skills to this job. Scroll down to read it.");
        }
      }
    } catch (e) {
      setState(() => _isGeneratingCoverLetter = false);
      _voiceService.speak("Sorry, something went wrong. Check your internet and try again.");
    }
  }

  Widget _buildCoverLetterTab() {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.all(16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(
            'Job Description',
            style: Theme.of(context).textTheme.titleMedium,
          ),
          const SizedBox(height: 8),
          Text(
            'Paste the job ad here, or describe what the job involves.',
            style: Theme.of(context).textTheme.bodySmall,
          ),
          const SizedBox(height: 8),
          _buildVoiceTextField(
            controller: _jobDescController,
            hint: 'Paste the job ad here or describe the role...',
            field: 'jobDesc',
            maxLines: 6,
          ),
          const SizedBox(height: 8),
          Text(
            'Tip: If you\'ve filled in your skills on the Resume tab, I\'ll use those to write your cover letter.',
            style: Theme.of(context).textTheme.bodySmall?.copyWith(
              fontStyle: FontStyle.italic,
            ),
          ),
          const SizedBox(height: 24),
          SizedBox(
            width: double.infinity,
            child: ElevatedButton(
              onPressed: _isGeneratingCoverLetter ? null : _generateCoverLetter,
              child: _isGeneratingCoverLetter
                  ? Row(
                      mainAxisAlignment: MainAxisAlignment.center,
                      children: [
                        const SizedBox(
                          height: 20,
                          width: 20,
                          child: CircularProgressIndicator(strokeWidth: 2, color: Colors.white),
                        ),
                        const SizedBox(width: 12),
                        const Text('Writing...'),
                      ],
                    )
                  : const Text('Write Cover Letter'),
            ),
          ),
          if (_generatedCoverLetter != null) ...[
            const SizedBox(height: 24),
            Card(
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.mail_outline, color: isDark ? AppColors.primaryLight : AppColors.primary),
                        const SizedBox(width: 8),
                        Expanded(child: Text('Your Cover Letter', style: Theme.of(context).textTheme.titleMedium)),
                        IconButton(
                          icon: Icon(Icons.volume_up, color: isDark ? AppColors.primaryLight : AppColors.primary),
                          tooltip: 'Read cover letter aloud',
                          onPressed: () => _voiceService.speak(_generatedCoverLetter!),
                        ),
                      ],
                    ),
                    const SizedBox(height: 12),
                    Text(_generatedCoverLetter!, style: Theme.of(context).textTheme.bodyMedium),
                    const SizedBox(height: 16),
                    SizedBox(
                      width: double.infinity,
                      child: OutlinedButton.icon(
                        onPressed: () {
                          setState(() => _generatedCoverLetter = null);
                          if (_prefs.autoPlayVoice) {
                            _voiceService.speak("Let's try again. Update the job description and tap Write Cover Letter.");
                          }
                        },
                        icon: const Icon(Icons.refresh),
                        label: const Text('Make Changes'),
                      ),
                    ),
                  ],
                ),
              ),
            ),
          ],
        ],
      ),
    );
  }

  Future<void> _generatePricing() async {
    if (_selectedWorkType == null && _pricingDescController.text.trim().isEmpty) {
      _voiceService.speak("Tell me what kind of work you're pricing. Tap a button or use the mic to describe it.");
      return;
    }
    
    _voiceService.stop();
    setState(() => _isGeneratingPricing = true);
    
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Working out a fair price for you. Give me a moment...");
    }
    
    try {
      final workTypeName = _selectedWorkType != null 
          ? _workTypes.firstWhere((w) => w['id'] == _selectedWorkType)['name']
          : 'General work';
      
      final result = await _apiClient.post('/api/skills/pricing', {
        'workType': workTypeName,
        'description': _pricingDescController.text,
        'location': 'Australia',
        'materials': '',
        'timeEstimate': '',
      });
      
      if (result.isNotEmpty) {
        final pricing = result['pricing'] as String?;
        if (pricing != null && pricing.isNotEmpty) {
          setState(() => _generatedPricing = pricing);
          if (_prefs.autoPlayVoice) {
            _voiceService.speak("Done! I've got some pricing suggestions for you. Scroll down to see them, or tap the speaker icon to hear them.");
          }
        }
      } else {
        if (_prefs.autoPlayVoice) {
          _voiceService.speak("Sorry, I couldn't work out the pricing. Try giving me more details.");
        }
      }
    } catch (e) {
      debugPrint('Pricing error: $e');
      if (_prefs.autoPlayVoice) {
        _voiceService.speak("Something went wrong. Check your connection and try again.");
      }
    } finally {
      setState(() => _isGeneratingPricing = false);
    }
  }

  void _speakPricing() {
    if (_generatedPricing != null) {
      _voiceService.stop();
      _voiceService.speak(_generatedPricing!);
    }
  }

  Widget _buildPricingTab() {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.all(16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(
            'Price Your Work',
            style: Theme.of(context).textTheme.headlineSmall,
          ),
          const SizedBox(height: 8),
          Text(
            'Not sure what to charge? Tell me about the job and I\'ll help you figure out a fair price.',
            style: Theme.of(context).textTheme.bodyMedium,
          ),
          const SizedBox(height: 20),
          
          Text(
            'What kind of work?',
            style: Theme.of(context).textTheme.titleMedium,
          ),
          const SizedBox(height: 12),
          Wrap(
            spacing: 8,
            runSpacing: 8,
            children: _workTypes.map((type) {
              final isSelected = _selectedWorkType == type['id'];
              return ChoiceChip(
                label: Row(
                  mainAxisSize: MainAxisSize.min,
                  children: [
                    Icon(
                      type['icon'] as IconData,
                      size: 18,
                      color: isSelected 
                          ? Colors.white 
                          : (isDark ? AppColors.primaryLight : AppColors.primary),
                    ),
                    const SizedBox(width: 6),
                    Text(type['name'] as String),
                  ],
                ),
                selected: isSelected,
                selectedColor: isDark ? AppColors.primaryLight : AppColors.primary,
                onSelected: (selected) {
                  setState(() => _selectedWorkType = selected ? type['id'] as String : null);
                  if (selected && _prefs.autoPlayVoice) {
                    _voiceService.speak("${type['name']} selected. Now describe the specific job.");
                  }
                },
              );
            }).toList(),
          ),
          
          const SizedBox(height: 20),
          Text(
            'Describe the job',
            style: Theme.of(context).textTheme.titleMedium,
          ),
          const SizedBox(height: 8),
          _buildVoiceTextField(
            controller: _pricingDescController,
            hint: 'E.g., "Painting a small canvas" or "Cleaning a 3-bedroom house" or "Mowing a large yard"',
            field: 'pricing',
            maxLines: 3,
          ),
          
          const SizedBox(height: 20),
          SizedBox(
            width: double.infinity,
            child: ElevatedButton.icon(
              onPressed: _isGeneratingPricing ? null : _generatePricing,
              icon: _isGeneratingPricing 
                  ? const SizedBox(
                      width: 20,
                      height: 20,
                      child: CircularProgressIndicator(strokeWidth: 2, color: Colors.white),
                    )
                  : const Icon(Icons.attach_money),
              label: Text(_isGeneratingPricing ? 'Working it out...' : 'Get Price Suggestion'),
              style: ElevatedButton.styleFrom(
                backgroundColor: isDark ? AppColors.primaryLight : AppColors.primary,
                foregroundColor: Colors.white,
                padding: const EdgeInsets.symmetric(vertical: 16),
              ),
            ),
          ),
          
          if (_generatedPricing != null) ...[
            const SizedBox(height: 24),
            Card(
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(
                          Icons.lightbulb,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                        ),
                        const SizedBox(width: 8),
                        Expanded(
                          child: Text(
                            'Pricing Suggestion',
                            style: Theme.of(context).textTheme.titleMedium,
                          ),
                        ),
                        IconButton(
                          icon: const Icon(Icons.volume_up),
                          onPressed: _speakPricing,
                          tooltip: 'Listen to pricing advice',
                        ),
                      ],
                    ),
                    const SizedBox(height: 12),
                    Text(_generatedPricing!, style: Theme.of(context).textTheme.bodyMedium),
                    const SizedBox(height: 16),
                    SizedBox(
                      width: double.infinity,
                      child: OutlinedButton.icon(
                        onPressed: () {
                          setState(() {
                            _generatedPricing = null;
                            _selectedWorkType = null;
                            _pricingDescController.clear();
                          });
                          if (_prefs.autoPlayVoice) {
                            _voiceService.speak("Let's price something else. What kind of work?");
                          }
                        },
                        icon: const Icon(Icons.refresh),
                        label: const Text('Price Something Else'),
                      ),
                    ),
                  ],
                ),
              ),
            ),
          ],
        ],
      ),
    );
  }

  Widget _buildVoiceTextField({
    required TextEditingController controller,
    required String hint,
    required String field,
    int maxLines = 4,
  }) {
    final isActive = _isListening && _activeField == field;
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Stack(
      children: [
        TextField(
          controller: controller,
          maxLines: maxLines,
          decoration: InputDecoration(
            hintText: hint,
          ),
        ),
        Positioned(
          right: 8,
          top: 8,
          child: FloatingActionButton.small(
            onPressed: isActive ? null : () => _startListening(field, controller),
            backgroundColor: isActive ? Colors.red : (isDark ? AppColors.primaryLight : AppColors.primary),
            child: Icon(
              isActive ? Icons.hearing : Icons.mic,
              color: Colors.white,
            ),
          ),
        ),
      ],
    );
  }
}
