import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/models/region.dart';
import '../../../core/widgets/shared_widgets.dart';

class SettingsPage extends StatefulWidget {
  const SettingsPage({super.key});

  @override
  State<SettingsPage> createState() => _SettingsPageState();
}

class _SettingsPageState extends State<SettingsPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ScrollController _scrollController = ScrollController();
  late VoicePersona _selectedPersona;
  late bool _autoPlayVoice;
  late AustralianRegion _selectedRegion;
  late String _preferredGreeting;
  final TextEditingController _customGreetingController = TextEditingController();
  bool _isPlaying = false;
  bool _hasPlayedIntro = false;

  @override
  void initState() {
    super.initState();
    _selectedPersona = _prefs.voicePersona;
    _autoPlayVoice = _prefs.autoPlayVoice;
    _selectedRegion = _prefs.region;
    _preferredGreeting = _prefs.preferredGreeting;
    _customGreetingController.text = _prefs.customGreeting ?? '';
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    debugPrint('SettingsPage: Initializing voice service...');
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playIntro();
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _scrollController.dispose();
    _customGreetingController.dispose();
    super.dispose();
  }

  void _playIntro() {
    debugPrint('SettingsPage: _playIntro called, autoPlay=${_prefs.autoPlayVoice}');
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      _voiceService.speak(
        "Settings. You can change the voice, turn auto-play on or off, or clear your saved data."
      );
    }
  }

  void _changeVoice(VoicePersona persona) async {
    _voiceService.stop();
    setState(() {
      _selectedPersona = persona;
      _voiceService.setPersona(persona);
    });
    await _prefs.setVoicePersona(persona);
    
    setState(() => _isPlaying = true);
    final text = persona == VoicePersona.aunty
        ? "This is how I'll sound now. Nice and clear."
        : "This is my voice. Ready to help you out.";
    await _voiceService.speak(text);
    if (mounted) setState(() => _isPlaying = false);
  }

  void _toggleAutoPlay(bool value) async {
    setState(() => _autoPlayVoice = value);
    await _prefs.setAutoPlayVoice(value);
  }

  void _testVoice() async {
    setState(() => _isPlaying = true);
    await _voiceService.speak("G'day! This is a test of the voice. Sounds good, yeah?");
    if (mounted) setState(() => _isPlaying = false);
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () => context.go('/'),
        ),
        title: const Text('Settings'),
      ),
      body: CulturalBackground(
        child: Stack(
          children: [
            ListView(
              controller: _scrollController,
              padding: const EdgeInsets.all(16),
              children: [
            _SectionHeader(title: 'Voice'),
          Card(
            child: Padding(
              padding: const EdgeInsets.all(16),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(
                    'Choose your voice',
                    style: Theme.of(context).textTheme.titleMedium?.copyWith(
                      fontWeight: FontWeight.w600,
                    ),
                  ),
                  const SizedBox(height: 4),
                  Text(
                    'This is the voice that reads things out for you',
                    style: Theme.of(context).textTheme.bodySmall?.copyWith(
                      color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                    ),
                  ),
                  const SizedBox(height: 16),
                  Row(
                    children: [
                      Expanded(
                        child: _VoiceButton(
                          label: "Aunty",
                          isSelected: _selectedPersona == VoicePersona.aunty,
                          isPlaying: _isPlaying && _selectedPersona == VoicePersona.aunty,
                          onTap: () => _changeVoice(VoicePersona.aunty),
                        ),
                      ),
                      const SizedBox(width: 12),
                      Expanded(
                        child: _VoiceButton(
                          label: "Uncle",
                          isSelected: _selectedPersona == VoicePersona.uncle,
                          isPlaying: _isPlaying && _selectedPersona == VoicePersona.uncle,
                          onTap: () => _changeVoice(VoicePersona.uncle),
                        ),
                      ),
                    ],
                  ),
                  const SizedBox(height: 12),
                  Center(
                    child: TextButton.icon(
                      onPressed: _isPlaying ? null : _testVoice,
                      icon: Icon(_isPlaying ? Icons.volume_up : Icons.play_arrow),
                      label: Text(_isPlaying ? 'Playing...' : 'Test voice'),
                    ),
                  ),
                ],
              ),
            ),
          ),
          const SizedBox(height: 8),
          Card(
            child: SwitchListTile(
              title: const Text('Auto-play voice'),
              subtitle: const Text('Automatically read out important info'),
              value: _autoPlayVoice,
              onChanged: _toggleAutoPlay,
            ),
          ),
          const SizedBox(height: 24),
          _SectionHeader(title: "Where's Your Country?"),
          Card(
            child: Padding(
              padding: const EdgeInsets.all(16),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(
                    'Optional - helps us personalise your experience',
                    style: Theme.of(context).textTheme.bodySmall?.copyWith(
                      color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                    ),
                  ),
                  const SizedBox(height: 12),
                  DropdownButtonFormField<AustralianRegion>(
                    value: _selectedRegion,
                    isExpanded: true,
                    decoration: InputDecoration(
                      contentPadding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
                      border: OutlineInputBorder(
                        borderRadius: BorderRadius.circular(8),
                      ),
                    ),
                    items: AustralianRegion.values.map((region) {
                      final info = getRegionInfo(region);
                      return DropdownMenuItem(
                        value: region,
                        child: Text(
                          region == AustralianRegion.notSet 
                            ? "I'd rather not say"
                            : '${info.selfIdentifyingTerm} - ${info.geographicArea}',
                        ),
                      );
                    }).toList(),
                    onChanged: (region) {
                      if (region != null) {
                        setState(() => _selectedRegion = region);
                        _prefs.setRegion(region);
                        if (region != AustralianRegion.notSet) {
                          final info = getRegionInfo(region);
                          _voiceService.speak("${info.selfIdentifyingTerm} mob, deadly. I'll remember that for you.");
                        }
                      }
                    },
                  ),
                  if (_selectedRegion != AustralianRegion.notSet) ...[
                    const SizedBox(height: 12),
                    Container(
                      padding: const EdgeInsets.all(12),
                      decoration: BoxDecoration(
                        color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.1),
                        borderRadius: BorderRadius.circular(8),
                      ),
                      child: Text(
                        getRegionInfo(_selectedRegion).acknowledgment,
                        style: Theme.of(context).textTheme.bodySmall?.copyWith(
                          fontStyle: FontStyle.italic,
                        ),
                      ),
                    ),
                  ],
                ],
              ),
            ),
          ),
          const SizedBox(height: 24),
          _SectionHeader(title: 'How Should We Greet You?'),
          Card(
            child: Padding(
              padding: const EdgeInsets.all(16),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(
                    'Optional - we want to address you respectfully',
                    style: Theme.of(context).textTheme.bodySmall?.copyWith(
                      color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                    ),
                  ),
                  const SizedBox(height: 8),
                  Text(
                    'Everyone deserves to be addressed in a way that feels right to them. This is stored only on your device.',
                    style: Theme.of(context).textTheme.bodySmall?.copyWith(
                      color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      fontStyle: FontStyle.italic,
                    ),
                  ),
                  const SizedBox(height: 12),
                  DropdownButtonFormField<String>(
                    value: _preferredGreeting,
                    isExpanded: true,
                    decoration: InputDecoration(
                      contentPadding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
                      border: OutlineInputBorder(
                        borderRadius: BorderRadius.circular(8),
                      ),
                    ),
                    items: const [
                      DropdownMenuItem(value: 'none', child: Text("Just use my name")),
                      DropdownMenuItem(value: 'sister', child: Text("Sister")),
                      DropdownMenuItem(value: 'brother', child: Text("Brother")),
                      DropdownMenuItem(value: 'sista', child: Text("Sista")),
                      DropdownMenuItem(value: 'bro', child: Text("Bro")),
                      DropdownMenuItem(value: 'cuz', child: Text("Cuz")),
                      DropdownMenuItem(value: 'sis', child: Text("Sis")),
                      DropdownMenuItem(value: 'friend', child: Text("Friend")),
                      DropdownMenuItem(value: 'mob', child: Text("Mob")),
                      DropdownMenuItem(value: 'custom', child: Text("Something else...")),
                    ],
                    onChanged: (value) {
                      if (value != null) {
                        setState(() => _preferredGreeting = value);
                        _prefs.setPreferredGreeting(value);
                        if (value != 'none' && value != 'custom') {
                          final title = value[0].toUpperCase() + value.substring(1);
                          _voiceService.speak("$title it is. Respect.");
                        }
                      }
                    },
                  ),
                  if (_preferredGreeting == 'custom') ...[
                    const SizedBox(height: 12),
                    TextField(
                      controller: _customGreetingController,
                      decoration: InputDecoration(
                        hintText: 'Enter how you\'d like to be addressed',
                        border: OutlineInputBorder(
                          borderRadius: BorderRadius.circular(8),
                        ),
                        contentPadding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
                      ),
                      onChanged: (value) {
                        _prefs.setCustomGreeting(value);
                      },
                    ),
                  ],
                ],
              ),
            ),
          ),
          const SizedBox(height: 24),
          _SectionHeader(title: 'About'),
          Card(
            child: Column(
              children: [
                ListTile(
                  leading: const Icon(Icons.auto_stories_outlined),
                  title: const Text('What is AI?'),
                  subtitle: const Text('Learn about this tool'),
                  trailing: const Icon(Icons.chevron_right),
                  onTap: () => context.go('/welcome-yarn'),
                ),
                const Divider(height: 1),
                ListTile(
                  leading: const Icon(Icons.shield_outlined),
                  title: const Text('Privacy Policy'),
                  subtitle: const Text('How we protect your data'),
                  trailing: const Icon(Icons.chevron_right),
                  onTap: () => context.go('/privacy'),
                ),
                const Divider(height: 1),
                ListTile(
                  leading: const Icon(Icons.gavel_outlined),
                  title: const Text('Know Your Rights'),
                  subtitle: const Text('Police, Centrelink, housing'),
                  trailing: const Icon(Icons.chevron_right),
                  onTap: () => context.go('/rights'),
                ),
              ],
            ),
          ),
          const SizedBox(height: 24),
          _SectionHeader(title: 'For Support Workers'),
          Card(
            child: Column(
              children: [
                ListTile(
                  leading: const Icon(Icons.business_outlined),
                  title: const Text('Partner Console'),
                  subtitle: const Text('Community impact data for agencies'),
                  trailing: const Icon(Icons.chevron_right),
                  onTap: () => context.go('/partner'),
                ),
                const Divider(height: 1),
                ListTile(
                  leading: const Icon(Icons.school),
                  title: const Text('Trainer Hub'),
                  subtitle: const Text('Track features shown to clients'),
                  trailing: const Icon(Icons.chevron_right),
                  onTap: () => context.go('/trainer'),
                ),
                const Divider(height: 1),
                ListTile(
                  leading: const Icon(Icons.bar_chart),
                  title: const Text('Impact Metrics'),
                  subtitle: const Text('Anonymous usage statistics'),
                  trailing: const Icon(Icons.chevron_right),
                  onTap: () => context.go('/impact'),
                ),
              ],
            ),
          ),
          const SizedBox(height: 24),
          _SectionHeader(title: 'Data'),
          Card(
            child: ListTile(
              leading: Icon(Icons.delete_outline, color: Colors.red[400]),
              title: Text('Clear all saved data', style: TextStyle(color: Colors.red[400])),
              subtitle: const Text('Remove all notes and preferences'),
              onTap: () => _showClearDataDialog(),
            ),
          ),
          const SizedBox(height: 32),
          Center(
            child: Text(
              'YarnAI v1.0.0',
              style: Theme.of(context).textTheme.bodySmall?.copyWith(
                color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
              ),
            ),
          ),
          const SizedBox(height: 8),
          Center(
            child: Text(
              'Made with care for mob',
              style: Theme.of(context).textTheme.bodySmall?.copyWith(
                color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
              ),
            ),
          ),
              ],
            ),
            ScrollIndicator(scrollController: _scrollController, bottomPadding: 20),
          ],
        ),
      ),
    );
  }

  void _showClearDataDialog() {
    showDialog(
      context: context,
      builder: (context) => AlertDialog(
        title: const Text('Clear all data?'),
        content: const Text('This will remove all your saved notes and preferences. This cannot be undone.'),
        actions: [
          TextButton(
            onPressed: () => Navigator.pop(context),
            child: const Text('Cancel'),
          ),
          TextButton(
            onPressed: () async {
              await _prefs.resetAll();
              if (mounted) {
                Navigator.pop(context);
                ScaffoldMessenger.of(context).showSnackBar(
                  const SnackBar(content: Text('All data cleared')),
                );
              }
            },
            child: Text('Clear', style: TextStyle(color: Colors.red[400])),
          ),
        ],
      ),
    );
  }
}

class _SectionHeader extends StatelessWidget {
  final String title;
  const _SectionHeader({required this.title});

  @override
  Widget build(BuildContext context) {
    return Padding(
      padding: const EdgeInsets.only(left: 4, bottom: 8, top: 8),
      child: Text(
        title,
        style: Theme.of(context).textTheme.titleSmall?.copyWith(
          fontWeight: FontWeight.w600,
          color: Theme.of(context).brightness == Brightness.dark
              ? AppColors.textSecondaryDark
              : AppColors.textSecondary,
        ),
      ),
    );
  }
}

class _VoiceButton extends StatelessWidget {
  final String label;
  final bool isSelected;
  final bool isPlaying;
  final VoidCallback onTap;

  const _VoiceButton({
    required this.label,
    required this.isSelected,
    required this.isPlaying,
    required this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final primaryColor = isDark ? AppColors.primaryLight : AppColors.primary;

    return OutlinedButton(
      onPressed: onTap,
      style: OutlinedButton.styleFrom(
        backgroundColor: isSelected ? primaryColor.withOpacity(0.15) : null,
        side: BorderSide(
          color: isSelected ? primaryColor : Colors.grey,
          width: isSelected ? 2 : 1,
        ),
        padding: const EdgeInsets.symmetric(vertical: 12),
      ),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          if (isPlaying)
            SizedBox(
              width: 16,
              height: 16,
              child: CircularProgressIndicator(strokeWidth: 2, color: primaryColor),
            )
          else
            Icon(
              isSelected ? Icons.check_circle : Icons.circle_outlined,
              size: 18,
              color: isSelected ? primaryColor : Colors.grey,
            ),
          const SizedBox(width: 8),
          Text(
            label,
            style: TextStyle(
              color: isSelected ? primaryColor : null,
              fontWeight: isSelected ? FontWeight.w600 : null,
            ),
          ),
        ],
      ),
    );
  }
}
