import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/widgets/shared_widgets.dart';

class SchoolPage extends StatefulWidget {
  const SchoolPage({super.key});

  @override
  State<SchoolPage> createState() => _SchoolPageState();
}

class _SchoolPageState extends State<SchoolPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _api = ApiClient();
  
  final _letterController = TextEditingController();
  final _contextController = TextEditingController();
  
  bool _hasPlayedIntro = false;
  bool _isLoading = false;
  String _selectedHelpType = 'understand';
  Map<String, dynamic>? _helpResult;
  String? _error;

  final List<Map<String, String>> _helpTypes = [
    {'id': 'enrolment', 'name': 'Enrolment Help', 'icon': 'school'},
    {'id': 'understand', 'name': 'Understand a Letter', 'icon': 'mail'},
    {'id': 'meeting', 'name': 'Prepare for Meeting', 'icon': 'people'},
    {'id': 'homework', 'name': 'Homework Question', 'icon': 'book'},
    {'id': 'permission', 'name': 'Permission Slip', 'icon': 'check'},
    {'id': 'excuse', 'name': 'Write Excuse Note', 'icon': 'note'},
  ];

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    if (mounted && !_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      await Future.delayed(const Duration(milliseconds: 300));
      _voiceService.speak(
        "Hey there, parents and carers - this is School Helper for all that school paperwork. "
        "Got a letter you don't understand? Need to write an excuse note? Getting ready to meet with teachers? "
        "Pick what you need help with and I'll walk you through it. "
        "I can even help the little ones with their homework."
      );
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _letterController.dispose();
    _contextController.dispose();
    super.dispose();
  }

  IconData _getIconForType(String type) {
    switch (type) {
      case 'mail': return Icons.mail_outline;
      case 'note': return Icons.note_alt_outlined;
      case 'people': return Icons.people_outline;
      case 'school': return Icons.school_outlined;
      case 'book': return Icons.menu_book_outlined;
      case 'check': return Icons.check_circle_outline;
      default: return Icons.help_outline;
    }
  }

  Future<void> _getHelp() async {
    setState(() {
      _isLoading = true;
      _error = null;
    });
    
    try {
      final result = await _api.post('/api/school/help', {
        'helpType': _selectedHelpType,
        'letterContent': _letterController.text,
        'context': _contextController.text,
      });
      
      setState(() {
        _helpResult = result;
        _isLoading = false;
      });
      
      if (_prefs.autoPlayVoice && result['summary'] != null) {
        _voiceService.speak(result['summary']);
      }
    } catch (e) {
      setState(() {
        _error = 'Having trouble. Try again?';
        _isLoading = false;
      });
    }
  }

  void _reset() {
    setState(() {
      _helpResult = null;
      _letterController.clear();
      _contextController.clear();
    });
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              Padding(
                padding: const EdgeInsets.all(16.0),
                child: Row(
                  children: [
                    IconButton(
                      icon: const Icon(Icons.arrow_back),
                      onPressed: () {
                        if (_helpResult != null) {
                          _reset();
                        } else {
                          context.go('/');
                        }
                      },
                    ),
                    const Spacer(),
                    IconButton(
                      icon: const Icon(Icons.volume_up),
                      onPressed: () {
                        _voiceService.speak(
                          "Pick what kind of school help you need, then tell me more about it."
                        );
                      },
                    ),
                    IconButton(
                      icon: const Icon(Icons.settings_outlined),
                      onPressed: () => context.go('/settings'),
                    ),
                  ],
                ),
              ),
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 20.0),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(
                          Icons.school_outlined,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                          size: 28,
                        ),
                        const SizedBox(width: 12),
                        Text(
                          'School Helper',
                          style: Theme.of(context).textTheme.headlineMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(
                      'Help with school letters and notes',
                      style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                  ],
                ),
              ),
              const SizedBox(height: 20),
              Expanded(
                child: _helpResult == null
                  ? _buildInputSection(isDark)
                  : _buildResultSection(isDark),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildInputSection(bool isDark) {
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(
            'What do you need help with?',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 12),
          Wrap(
            spacing: 8,
            runSpacing: 8,
            children: _helpTypes.map((type) {
              final isSelected = _selectedHelpType == type['id'];
              return FilterChip(
                selected: isSelected,
                label: Row(
                  mainAxisSize: MainAxisSize.min,
                  children: [
                    Icon(
                      _getIconForType(type['icon']!),
                      size: 18,
                      color: isSelected 
                        ? Colors.white 
                        : (isDark ? AppColors.primaryLight : AppColors.primary),
                    ),
                    const SizedBox(width: 8),
                    Text(type['name']!),
                  ],
                ),
                onSelected: (selected) {
                  setState(() => _selectedHelpType = type['id']!);
                },
              );
            }).toList(),
          ),
          const SizedBox(height: 20),
          
          if (_selectedHelpType == 'understand') ...[
            Text(
              'Paste or type the school letter here',
              style: Theme.of(context).textTheme.titleMedium?.copyWith(
                fontWeight: FontWeight.bold,
              ),
            ),
            const SizedBox(height: 8),
            TextField(
              controller: _letterController,
              decoration: InputDecoration(
                hintText: 'Copy and paste the letter content here...',
                border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
                suffixIcon: MicButton(onResult: (t) => _letterController.text = t),
              ),
              maxLines: 6,
            ),
          ] else ...[
            Text(
              'Tell me more about what you need',
              style: Theme.of(context).textTheme.titleMedium?.copyWith(
                fontWeight: FontWeight.bold,
              ),
            ),
            const SizedBox(height: 8),
            TextField(
              controller: _contextController,
              decoration: InputDecoration(
                hintText: _getHintText(),
                border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
                suffixIcon: MicButton(onResult: (t) => _contextController.text = t),
              ),
              maxLines: 4,
            ),
          ],
          const SizedBox(height: 24),
          
          if (_error != null)
            Padding(
              padding: const EdgeInsets.only(bottom: 16),
              child: Text(_error!, style: TextStyle(color: Colors.red[400])),
            ),
          
          SizedBox(
            width: double.infinity,
            child: ElevatedButton.icon(
              onPressed: _isLoading ? null : _getHelp,
              icon: _isLoading 
                ? const SizedBox(width: 20, height: 20, child: CircularProgressIndicator(strokeWidth: 2))
                : const Icon(Icons.check),
              label: Text(_isLoading ? 'Working on it...' : 'Get Help'),
              style: ElevatedButton.styleFrom(
                padding: const EdgeInsets.symmetric(vertical: 16),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
            ),
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }

  String _getHintText() {
    switch (_selectedHelpType) {
      case 'excuse': return 'e.g., My child was sick yesterday with a cold';
      case 'meeting': return 'e.g., Meeting about my child\'s reading progress';
      case 'enrolment': return 'e.g., Enrolling my 5 year old for kindy';
      case 'homework': return 'e.g., Help with a maths question about fractions';
      case 'permission': return 'e.g., Excursion to the zoo next week';
      default: return 'Tell me what you need...';
    }
  }

  Widget _buildResultSection(bool isDark) {
    final result = _helpResult!;
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          if (result['summary'] != null) ...[
            Card(
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.lightbulb_outline, color: isDark ? AppColors.primaryLight : AppColors.primary),
                        const SizedBox(width: 8),
                        Text(
                          'Summary',
                          style: Theme.of(context).textTheme.titleMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                        const Spacer(),
                        IconButton(
                          icon: const Icon(Icons.volume_up, size: 20),
                          onPressed: () => _voiceService.speak(result['summary']),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(result['summary']),
                  ],
                ),
              ),
            ),
            const SizedBox(height: 16),
          ],
          
          if (result['actionNeeded'] != null) ...[
            Row(
              children: [
                Icon(Icons.assignment_turned_in, color: Colors.orange[400]),
                const SizedBox(width: 8),
                Text(
                  'Action Needed',
                  style: Theme.of(context).textTheme.titleMedium?.copyWith(
                    fontWeight: FontWeight.bold,
                  ),
                ),
              ],
            ),
            const SizedBox(height: 8),
            Text(result['actionNeeded']),
            const SizedBox(height: 16),
          ],
          
          if (result['deadline'] != null && result['deadline'].toString().isNotEmpty) ...[
            Card(
              color: Colors.red.withOpacity(0.1),
              child: Padding(
                padding: const EdgeInsets.all(12),
                child: Row(
                  children: [
                    Icon(Icons.calendar_today, color: Colors.red[400]),
                    const SizedBox(width: 8),
                    Text(
                      'Deadline: ${result['deadline']}',
                      style: const TextStyle(fontWeight: FontWeight.bold),
                    ),
                  ],
                ),
              ),
            ),
            const SizedBox(height: 16),
          ],
          
          if (result['draftResponse'] != null && result['draftResponse'].toString().isNotEmpty) ...[
            Card(
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        const Icon(Icons.edit),
                        const SizedBox(width: 8),
                        Text(
                          'Draft Response',
                          style: Theme.of(context).textTheme.titleMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 12),
                    Container(
                      padding: const EdgeInsets.all(12),
                      decoration: BoxDecoration(
                        color: isDark ? Colors.black26 : Colors.grey[100],
                        borderRadius: BorderRadius.circular(8),
                      ),
                      child: Text(result['draftResponse'].toString()),
                    ),
                  ],
                ),
              ),
            ),
            const SizedBox(height: 16),
          ],
          
          if (result['tips'] != null && (result['tips'] as List).isNotEmpty) ...[
            Text(
              'Tips',
              style: Theme.of(context).textTheme.titleMedium?.copyWith(
                fontWeight: FontWeight.bold,
              ),
            ),
            const SizedBox(height: 8),
            ...((result['tips'] as List).map((tip) => Padding(
              padding: const EdgeInsets.only(bottom: 8),
              child: Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Icon(Icons.check_circle, color: Colors.green[400], size: 18),
                  const SizedBox(width: 8),
                  Expanded(child: Text(tip.toString())),
                ],
              ),
            ))),
            const SizedBox(height: 16),
          ],
          
          SizedBox(
            width: double.infinity,
            child: OutlinedButton.icon(
              onPressed: _reset,
              icon: const Icon(Icons.refresh),
              label: const Text('Ask About Something Else'),
              style: OutlinedButton.styleFrom(
                padding: const EdgeInsets.symmetric(vertical: 16),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
            ),
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }
}
