import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:url_launcher/url_launcher.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class RightsPage extends StatefulWidget {
  const RightsPage({super.key});

  @override
  State<RightsPage> createState() => _RightsPageState();
}

class _RightsPageState extends State<RightsPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ScrollController _scrollController = ScrollController();
  bool _hasPlayedIntro = false;

  final List<Map<String, dynamic>> _categories = [
    {
      'id': 'police',
      'title': 'Police',
      'icon': Icons.local_police,
      'youCan': [
        'Ask why you\'re being stopped',
        'Stay silent (except for giving your name and address)',
        'Ask if you\'re free to go',
        'Have a lawyer present during questioning',
      ],
      'theyMust': [
        'Tell you why you\'re being arrested',
        'Let you call a lawyer or family member',
        'Treat you with respect',
        'Not use excessive force',
      ],
      'youDontHaveTo': [
        'Answer questions beyond your name and address',
        'Consent to a search (unless they have a warrant or reasonable suspicion)',
        'Go to the station unless formally arrested',
        'Sign anything without legal advice',
      ],
    },
    {
      'id': 'centrelink',
      'title': 'Centrelink',
      'icon': Icons.account_balance,
      'youCan': [
        'Ask for decisions in writing',
        'Request a review of any decision',
        'Ask for an Indigenous liaison officer',
        'Set up a payment plan for debts',
      ],
      'theyMust': [
        'Explain decisions clearly',
        'Give you time to provide documents',
        'Consider hardship claims',
        'Provide interpreters if needed',
      ],
      'youDontHaveTo': [
        'Pay a debt immediately - you can negotiate',
        'Accept a decision without review',
        'Attend in person if you can\'t',
        'Give information they don\'t need',
      ],
    },
    {
      'id': 'housing',
      'title': 'Housing',
      'icon': Icons.home,
      'youCan': [
        'Request repairs in writing',
        'Ask for rent reductions if issues aren\'t fixed',
        'Have a support person at inspections',
        'Challenge unfair eviction notices',
      ],
      'theyMust': [
        'Keep the property safe and livable',
        'Give proper notice before inspections',
        'Follow the correct eviction process',
        'Return your bond fairly',
      ],
      'youDontHaveTo': [
        'Let them in without proper notice',
        'Accept rent increases without notice',
        'Leave immediately on verbal demand',
        'Pay for normal wear and tear',
      ],
    },
    {
      'id': 'schools',
      'title': 'Schools',
      'icon': Icons.school,
      'youCan': [
        'Take cultural leave for sorry business',
        'Have an Aboriginal education worker involved',
        'Request meetings be at suitable times',
        'Bring a support person to meetings',
      ],
      'theyMust': [
        'Consider cultural reasons for absences',
        'Follow suspension procedures fairly',
        'Communicate clearly about issues',
        'Support your child\'s learning needs',
      ],
      'youDontHaveTo': [
        'Accept unfair punishment',
        'Attend meetings without support',
        'Accept decisions without appeal',
        'Share personal family details',
      ],
    },
    {
      'id': 'employers',
      'title': 'Employers',
      'icon': Icons.work,
      'youCan': [
        'Ask for pay slips and written contracts',
        'Take sick leave and personal leave',
        'Request reasonable adjustments',
        'Join a union for support',
      ],
      'theyMust': [
        'Pay at least minimum wage',
        'Provide safe working conditions',
        'Not discriminate based on race',
        'Follow fair dismissal procedures',
      ],
      'youDontHaveTo': [
        'Work for free or unpaid training',
        'Accept unsafe conditions',
        'Answer personal questions in interviews',
        'Accept unfair dismissal quietly',
      ],
    },
  ];

  final List<Map<String, String>> _helpServices = [
    {'name': 'Aboriginal Legal Service', 'phone': '1800 765 767', 'mob': 'true'},
    {'name': '13YARN Crisis Line', 'phone': '13 92 76', 'mob': 'true'},
    {'name': 'Legal Aid NSW', 'phone': '1300 888 529', 'mob': 'false'},
    {'name': 'National Debt Helpline', 'phone': '1800 007 007', 'mob': 'false'},
  ];

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    debugPrint('RightsPage: Initializing voice service...');
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playIntro();
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _scrollController.dispose();
    super.dispose();
  }

  void _playIntro() {
    debugPrint('RightsPage: _playIntro called, autoPlay=${_prefs.autoPlayVoice}');
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      _voiceService.speak(
        "This page is about your power. "
        "When you're dealing with police, Centrelink, housing, schools, or bosses, they have to follow rules too. "
        "Tap any topic to see what you can do, what they must do, and what you don't have to do. "
        "There's also a list of free help services at the bottom."
      );
    }
  }

  Future<void> _callNumber(String name, String phone, bool isMob) async {
    final uri = Uri.parse('tel:${phone.replaceAll(' ', '')}');
    final mobText = isMob ? ', which is a mob-run service' : '';
    
    await _voiceService.speak('Calling $name on $phone$mobText.');
    
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri);
    } else {
      await _voiceService.speak('Sorry, couldn\'t place that call. Try dialling $phone directly.');
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text('Couldn\'t place call. Dial $phone directly.')),
        );
      }
    }
  }

  void _speakCategory(Map<String, dynamic> category) {
    _voiceService.stop();
    final title = category['title'] as String;
    final youCan = (category['youCan'] as List).join('. ');
    final theyMust = (category['theyMust'] as List).join('. ');
    final youDontHaveTo = (category['youDontHaveTo'] as List).join('. ');
    
    _voiceService.speak(
      'When dealing with $title: You CAN: $youCan. They MUST: $theyMust. You DON\'T have to: $youDontHaveTo.'
    );
  }

  void _speakSection(String sectionTitle, List<dynamic> items) {
    _voiceService.stop();
    final itemsText = items.join('. ');
    _voiceService.speak('$sectionTitle: $itemsText');
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () {
            _voiceService.stop();
            context.go('/');
          },
        ),
        title: const Text('Know Your Rights'),
        actions: [
          IconButton(
            icon: const Icon(Icons.volume_up),
            tooltip: 'Read page aloud',
            onPressed: () {
              _voiceService.speak(
                "Know your rights when dealing with police, Centrelink, housing, schools, or employers. "
                "Tap any topic to see what you can do, what they must do, and what you don't have to do."
              );
            },
          ),
          const SettingsButton(),
        ],
      ),
      body: CulturalBackground(
        child: Stack(
          children: [
            ListView(
              controller: _scrollController,
              padding: const EdgeInsets.all(16),
              children: [
            Text(
              'Know what you can do, what they must do, and what you don\'t have to do.',
              style: Theme.of(context).textTheme.bodyLarge,
            ),
          const SizedBox(height: 16),
          ..._categories.map((category) => Card(
            margin: const EdgeInsets.only(bottom: 12),
            child: ExpansionTile(
              leading: Icon(
                category['icon'] as IconData,
                color: isDark ? AppColors.primaryLight : AppColors.primary,
              ),
              title: Row(
                children: [
                  Expanded(child: Text(category['title'] as String)),
                  IconButton(
                    icon: const Icon(Icons.volume_up, size: 20),
                    tooltip: 'Listen to all rights for ${category['title']}',
                    onPressed: () => _speakCategory(category),
                  ),
                ],
              ),
              children: [
                _buildRightsSection(
                  'You CAN',
                  category['youCan'] as List,
                  Colors.green,
                  Icons.check_circle,
                  isDark,
                ),
                _buildRightsSection(
                  'They MUST',
                  category['theyMust'] as List,
                  Colors.blue,
                  Icons.gavel,
                  isDark,
                ),
                _buildRightsSection(
                  'You DON\'T have to',
                  category['youDontHaveTo'] as List,
                  Colors.orange,
                  Icons.block,
                  isDark,
                ),
              ],
            ),
          )),
          const SizedBox(height: 24),
          Text(
            'Free Help Services',
            style: Theme.of(context).textTheme.titleMedium,
          ),
          const SizedBox(height: 12),
          ..._helpServices.map((service) {
            final isMob = service['mob'] == 'true';
            return Card(
              margin: const EdgeInsets.only(bottom: 8),
              child: ListTile(
                leading: Icon(
                  Icons.phone,
                  color: isMob ? Colors.green : (isDark ? AppColors.primaryLight : AppColors.primary),
                ),
                title: Row(
                  children: [
                    Flexible(child: Text(service['name']!)),
                    if (isMob) ...[
                      const SizedBox(width: 8),
                      Container(
                        padding: const EdgeInsets.symmetric(horizontal: 6, vertical: 2),
                        decoration: BoxDecoration(
                          color: Colors.green.withOpacity(0.2),
                          borderRadius: BorderRadius.circular(4),
                        ),
                        child: const Text(
                          'Mob-run',
                          style: TextStyle(fontSize: 10, color: Colors.green, fontWeight: FontWeight.bold),
                        ),
                      ),
                    ],
                  ],
                ),
                subtitle: Text(service['phone']!),
                trailing: Row(
                  mainAxisSize: MainAxisSize.min,
                  children: [
                    IconButton(
                      icon: Icon(
                        Icons.volume_up,
                        size: 20,
                        color: isDark ? AppColors.primaryLight : AppColors.primary,
                      ),
                      tooltip: 'Listen to service info',
                      onPressed: () {
                        _voiceService.stop();
                        final mobText = isMob ? ' This is a mob-run service.' : '';
                        _voiceService.speak('${service['name']}, phone number ${service['phone']}.$mobText');
                      },
                    ),
                    IconButton(
                      icon: const Icon(Icons.call),
                      tooltip: 'Call ${service['name']}',
                      onPressed: () => _callNumber(service['name']!, service['phone']!, isMob),
                    ),
                  ],
                ),
              ),
            );
          }),
              ],
            ),
            ScrollIndicator(scrollController: _scrollController, bottomPadding: 20),
          ],
        ),
      ),
    );
  }

  Widget _buildRightsSection(String title, List<dynamic> items, Color color, IconData icon, bool isDark) {
    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 8),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            children: [
              Icon(icon, color: color, size: 18),
              const SizedBox(width: 8),
              Text(
                title,
                style: TextStyle(
                  fontWeight: FontWeight.bold,
                  color: color,
                ),
              ),
              const Spacer(),
              IconButton(
                icon: Icon(Icons.volume_up, size: 18, color: isDark ? AppColors.primaryLight : AppColors.primary),
                tooltip: 'Listen to $title section',
                onPressed: () => _speakSection(title, items),
              ),
            ],
          ),
          const SizedBox(height: 8),
          ...items.map((item) => Padding(
            padding: const EdgeInsets.only(left: 26, bottom: 4),
            child: Row(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text('\u2022 ', style: TextStyle(color: color)),
                Expanded(child: Text(item as String)),
              ],
            ),
          )),
        ],
      ),
    );
  }
}
