import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class PrivacyPage extends StatefulWidget {
  const PrivacyPage({super.key});

  @override
  State<PrivacyPage> createState() => _PrivacyPageState();
}

class _PrivacyPageState extends State<PrivacyPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ScrollController _scrollController = ScrollController();
  bool _hasPlayedIntro = false;

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    debugPrint('PrivacyPage: Initializing voice service...');
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playIntro();
    }
  }

  @override
  void dispose() {
    _scrollController.dispose();
    _voiceService.stop();
    super.dispose();
  }

  void _playIntro() {
    debugPrint('PrivacyPage: _playIntro called, autoPlay=${_prefs.autoPlayVoice}');
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      _voiceService.speak(
        "This is our privacy promise to you. "
        "Your data belongs to you and your community. "
        "We don't store your documents. We don't sell your information. "
        "Tap any section to hear it read aloud."
      );
    }
  }

  void _speakPrivacyOverview() {
    _voiceService.stop();
    _voiceService.speak(
      "Your privacy and data sovereignty matter to us. YarnAI is built with respect for Indigenous data sovereignty and Australian Privacy Principles. "
      "We believe your data belongs to you and your community. "
      "We don't store your documents. Documents you share are processed to help you understand them, then immediately discarded. "
      "You have the right to deletion. You can delete your account and all data at any time. "
      "We support community governance. Community data should be governed by the community. "
      "Your information is stored in Australia, subject to Australian privacy laws."
    );
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () {
            _voiceService.stop();
            context.go('/');
          },
        ),
        title: const Text('Privacy Policy'),
        actions: [
          IconButton(
            icon: const Icon(Icons.volume_up),
            tooltip: 'Listen to privacy policy summary',
            onPressed: _speakPrivacyOverview,
          ),
          const SettingsButton(),
        ],
      ),
      body: CulturalBackground(
        child: Stack(
          children: [
            ListView(
              controller: _scrollController,
              padding: const EdgeInsets.all(16),
              children: [
            Icon(
              Icons.shield,
              size: 48,
              color: isDark ? AppColors.primaryLight : AppColors.primary,
            ),
          const SizedBox(height: 16),
          Text(
            'Your privacy and data sovereignty matter to us.',
            style: Theme.of(context).textTheme.titleMedium,
            textAlign: TextAlign.center,
          ),
          const SizedBox(height: 24),
          _buildSection(
            context,
            'Data Protection Principles',
            Icons.lock,
            [
              'YarnAI is built with respect for Indigenous data sovereignty and Australian Privacy Principles.',
              'We believe your data belongs to you and your community.',
            ],
            isDark,
            () {
              _voiceService.stop();
              _voiceService.speak(
                "Data Protection Principles. YarnAI is built with respect for Indigenous data sovereignty and Australian Privacy Principles. We believe your data belongs to you and your community."
              );
            },
          ),
          _buildInfoCard(
            context,
            isDark,
            Icons.visibility_off,
            'We Don\'t Store Your Documents',
            'Documents you share are processed to help you understand them, then immediately discarded. We never keep copies.',
            () {
              _voiceService.stop();
              _voiceService.speak(
                "We don't store your documents. Documents you share are processed to help you understand them, then immediately discarded. We never keep copies."
              );
            },
          ),
          _buildInfoCard(
            context,
            isDark,
            Icons.delete,
            'Right to Deletion',
            'You can delete your account and all data at any time. We will comply within 30 days.',
            () {
              _voiceService.stop();
              _voiceService.speak(
                "Right to deletion. You can delete your account and all data at any time. We will comply within 30 days."
              );
            },
          ),
          _buildInfoCard(
            context,
            isDark,
            Icons.people,
            'Community Governance',
            'We support Indigenous data sovereignty principles. Community data should be governed by the community.',
            () {
              _voiceService.stop();
              _voiceService.speak(
                "Community governance. We support Indigenous data sovereignty principles. Community data should be governed by the community."
              );
            },
          ),
          _buildInfoCard(
            context,
            isDark,
            Icons.location_on,
            'Australian Data Storage',
            'Your information is stored in Australia, subject to Australian privacy laws.',
            () {
              _voiceService.stop();
              _voiceService.speak(
                "Australian data storage. Your information is stored in Australia, subject to Australian privacy laws."
              );
            },
          ),
          const SizedBox(height: 24),
          _buildSection(
            context,
            'What We Don\'t Do',
            Icons.block,
            [
              'We never sell your personal information',
              'We never store the content of documents you analyze',
              'We never share your data with government agencies without legal requirement',
              'We never use your data to train AI models without explicit consent',
            ],
            isDark,
            () {
              _voiceService.stop();
              _voiceService.speak(
                "What we don't do. We never sell your personal information. We never store the content of documents you analyze. We never share your data with government agencies without legal requirement. We never use your data to train AI models without explicit consent."
              );
            },
          ),
          const SizedBox(height: 24),
          _buildSection(
            context,
            'Your Rights',
            Icons.gavel,
            [
              'Access your personal information',
              'Request correction of inaccurate information',
              'Request deletion of your data',
              'Know how your information is used',
              'Complain to the Office of the Australian Information Commissioner (OAIC)',
            ],
            isDark,
            () {
              _voiceService.stop();
              _voiceService.speak(
                "Your rights. Access your personal information. Request correction of inaccurate information. Request deletion of your data. Know how your information is used. Complain to the Office of the Australian Information Commissioner."
              );
            },
          ),
          const SizedBox(height: 24),
          Card(
            child: Padding(
              padding: const EdgeInsets.all(16),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Row(
                    children: [
                      Icon(Icons.contact_phone, color: isDark ? AppColors.primaryLight : AppColors.primary),
                      const SizedBox(width: 8),
                      Expanded(
                        child: Text(
                          'Contact OAIC',
                          style: Theme.of(context).textTheme.titleMedium,
                        ),
                      ),
                      IconButton(
                        icon: Icon(Icons.volume_up, size: 20, color: isDark ? AppColors.primaryLight : AppColors.primary),
                        tooltip: 'Listen to contact info',
                        onPressed: () {
                          _voiceService.stop();
                          _voiceService.speak(
                            "If you are not satisfied with our response, you can complain to the Office of the Australian Information Commissioner. Website: oaic.gov.au. Phone: 1300 363 992. Email: enquiries at oaic dot gov dot au."
                          );
                        },
                      ),
                    ],
                  ),
                  const SizedBox(height: 12),
                  const Text('If you are not satisfied with our response, you can complain to:'),
                  const SizedBox(height: 8),
                  const Text('Website: oaic.gov.au'),
                  const Text('Phone: 1300 363 992'),
                  const Text('Email: enquiries@oaic.gov.au'),
                ],
              ),
            ),
          ),
          const SizedBox(height: 24),
          Text(
            'Last updated: January 2026',
            style: Theme.of(context).textTheme.bodySmall,
            textAlign: TextAlign.center,
          ),
          ],
        ),
        ScrollIndicator(scrollController: _scrollController, bottomPadding: 20),
          ],
        ),
      ),
    );
  }

  Widget _buildSection(
    BuildContext context,
    String title,
    IconData icon,
    List<String> items,
    bool isDark,
    VoidCallback onListen,
  ) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Row(
          children: [
            Icon(icon, color: isDark ? AppColors.primaryLight : AppColors.primary),
            const SizedBox(width: 8),
            Expanded(child: Text(title, style: Theme.of(context).textTheme.titleMedium)),
            IconButton(
              icon: Icon(Icons.volume_up, size: 20, color: isDark ? AppColors.primaryLight : AppColors.primary),
              tooltip: 'Listen to $title',
              onPressed: onListen,
            ),
          ],
        ),
        const SizedBox(height: 12),
        ...items.map((item) => Padding(
          padding: const EdgeInsets.only(left: 32, bottom: 8),
          child: Row(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              const Text('\u2022 '),
              Expanded(child: Text(item)),
            ],
          ),
        )),
      ],
    );
  }

  Widget _buildInfoCard(
    BuildContext context,
    bool isDark,
    IconData icon,
    String title,
    String content,
    VoidCallback onListen,
  ) {
    return Card(
      margin: const EdgeInsets.only(bottom: 12),
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Row(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Icon(icon, color: isDark ? AppColors.primaryLight : AppColors.primary),
            const SizedBox(width: 12),
            Expanded(
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(title, style: const TextStyle(fontWeight: FontWeight.bold)),
                  const SizedBox(height: 4),
                  Text(content, style: Theme.of(context).textTheme.bodyMedium),
                ],
              ),
            ),
            IconButton(
              icon: Icon(Icons.volume_up, size: 20, color: isDark ? AppColors.primaryLight : AppColors.primary),
              tooltip: 'Listen',
              onPressed: onListen,
            ),
          ],
        ),
      ),
    );
  }
}
