import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'dart:convert';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/voice_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class PartnerConsolePage extends StatefulWidget {
  const PartnerConsolePage({super.key});

  @override
  State<PartnerConsolePage> createState() => _PartnerConsolePageState();
}

class _PartnerConsolePageState extends State<PartnerConsolePage> {
  bool _hasConsent = false;
  Map<String, dynamic>? _metrics;

  final String _introText = 
      "Welcome to the Partner Console. This is for support workers and organisations "
      "who want to see how the community is using this tool. All data here is completely "
      "anonymised. No names, no personal details. Just numbers that show the kinds of "
      "things people are getting help with.";

  @override
  void initState() {
    super.initState();
    _checkConsent();
  }

  Future<void> _checkConsent() async {
    final prefs = await SharedPreferences.getInstance();
    final consent = prefs.getBool('partner_consent') ?? false;
    if (consent) {
      setState(() => _hasConsent = true);
      _loadMetrics();
    }
  }

  Future<void> _loadMetrics() async {
    final prefs = await SharedPreferences.getInstance();
    
    final callsCount = prefs.getInt('calls_practiced') ?? 0;
    final docsCount = prefs.getInt('documents_analyzed') ?? 0;
    final chatsCount = prefs.getInt('chats_count') ?? 0;
    
    setState(() {
      _metrics = {
        'totalInteractions': callsCount + docsCount + chatsCount,
        'callsPracticed': callsCount,
        'documentsExplained': docsCount,
        'chatsCount': chatsCount,
      };
    });
  }

  Future<void> _grantConsent() async {
    final prefs = await SharedPreferences.getInstance();
    await prefs.setBool('partner_consent', true);
    setState(() => _hasConsent = true);
    _loadMetrics();
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () => context.go('/settings'),
          tooltip: 'Back to settings',
        ),
        title: const Text('Partner Console'),
        actions: [
          IconButton(
            icon: const Icon(Icons.home),
            onPressed: () => context.go('/'),
            tooltip: 'Home',
          ),
          ListenButton(text: _introText, size: 40),
        ],
      ),
      body: ScrollableWithIndicator(
        padding: const EdgeInsets.all(16),
        child: !_hasConsent ? _buildConsentScreen() : _buildDashboard(),
      ),
    );
  }

  Widget _buildConsentScreen() {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.stretch,
      children: [
        const Icon(Icons.business, size: 64, color: AppColors.primary),
        const SizedBox(height: 24),
        const Text(
          'Partner Console',
          style: TextStyle(fontSize: 24, fontWeight: FontWeight.bold),
          textAlign: TextAlign.center,
        ),
        const SizedBox(height: 8),
        Text(
          'For Support Workers & Organisations',
          style: TextStyle(color: Colors.grey[600]),
          textAlign: TextAlign.center,
        ),
        const SizedBox(height: 24),
        Container(
          padding: const EdgeInsets.all(16),
          decoration: BoxDecoration(
            color: AppColors.primary.withOpacity(0.1),
            borderRadius: BorderRadius.circular(12),
          ),
          child: Column(
            children: [
              const Icon(Icons.analytics, color: AppColors.primary),
              const SizedBox(height: 8),
              const Text(
                'View anonymized community impact data',
                style: TextStyle(fontWeight: FontWeight.w500),
                textAlign: TextAlign.center,
              ),
              const SizedBox(height: 8),
              Text(
                'This data shows how the community uses YarnAI without any personal information. '
                'You can use this for grant reports and understanding community needs.',
                style: TextStyle(color: Colors.grey[600], fontSize: 12),
                textAlign: TextAlign.center,
              ),
            ],
          ),
        ),
        const SizedBox(height: 16),
        Container(
          padding: const EdgeInsets.all(16),
          decoration: BoxDecoration(
            color: Colors.green.withOpacity(0.1),
            borderRadius: BorderRadius.circular(12),
          ),
          child: Row(
            children: [
              const Icon(Icons.lock, color: Colors.green),
              const SizedBox(width: 12),
              Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    const Text(
                      'Privacy Protected',
                      style: TextStyle(fontWeight: FontWeight.w600),
                    ),
                    Text(
                      'All data is aggregated and anonymized',
                      style: TextStyle(fontSize: 12, color: Colors.grey[600]),
                    ),
                  ],
                ),
              ),
            ],
          ),
        ),
        const SizedBox(height: 24),
        ElevatedButton(
          onPressed: _grantConsent,
          style: ElevatedButton.styleFrom(
            padding: const EdgeInsets.all(16),
          ),
          child: const Text('Access Dashboard'),
        ),
      ],
    );
  }

  Widget _buildDashboard() {
    if (_metrics == null) {
      return const Center(child: CircularProgressIndicator());
    }

    return Column(
      crossAxisAlignment: CrossAxisAlignment.stretch,
      children: [
        Row(
          children: [
            Expanded(
              child: _buildMetricCard(
                'Total Uses',
                _metrics!['totalInteractions'].toString(),
                Icons.touch_app,
              ),
            ),
            const SizedBox(width: 12),
            Expanded(
              child: _buildMetricCard(
                'Calls Practiced',
                _metrics!['callsPracticed'].toString(),
                Icons.phone,
              ),
            ),
          ],
        ),
        const SizedBox(height: 12),
        Row(
          children: [
            Expanded(
              child: _buildMetricCard(
                'Docs Explained',
                _metrics!['documentsExplained'].toString(),
                Icons.description,
              ),
            ),
            const SizedBox(width: 12),
            Expanded(
              child: _buildMetricCard(
                'Conversations',
                _metrics!['chatsCount'].toString(),
                Icons.chat,
              ),
            ),
          ],
        ),
        const SizedBox(height: 24),
        Card(
          child: Padding(
            padding: const EdgeInsets.all(16),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Row(
                  children: [
                    const Icon(Icons.download, color: AppColors.primary),
                    const SizedBox(width: 8),
                    const Text(
                      'Export Report',
                      style: TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
                    ),
                  ],
                ),
                const SizedBox(height: 12),
                Text(
                  'Generate a report for grant applications or impact assessment.',
                  style: TextStyle(color: Colors.grey[600]),
                ),
                const SizedBox(height: 12),
                ElevatedButton.icon(
                  onPressed: _exportReport,
                  icon: const Icon(Icons.file_download),
                  label: const Text('Export PDF Report'),
                ),
              ],
            ),
          ),
        ),
        const SizedBox(height: 16),
        Container(
          padding: const EdgeInsets.all(16),
          decoration: BoxDecoration(
            color: Colors.blue.withOpacity(0.1),
            borderRadius: BorderRadius.circular(12),
          ),
          child: Row(
            children: [
              const Icon(Icons.info, color: Colors.blue),
              const SizedBox(width: 12),
              Expanded(
                child: Text(
                  'Data is stored locally on user devices. These metrics are aggregated when users opt in.',
                  style: TextStyle(fontSize: 12, color: Colors.grey[600]),
                ),
              ),
            ],
          ),
        ),
      ],
    );
  }

  Widget _buildMetricCard(String label, String value, IconData icon) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          children: [
            Icon(icon, color: AppColors.primary, size: 32),
            const SizedBox(height: 8),
            Text(
              value,
              style: const TextStyle(
                fontSize: 28,
                fontWeight: FontWeight.bold,
                color: AppColors.primary,
              ),
            ),
            Text(
              label,
              style: TextStyle(fontSize: 12, color: Colors.grey[600]),
              textAlign: TextAlign.center,
            ),
          ],
        ),
      ),
    );
  }

  void _exportReport() {
    ScaffoldMessenger.of(context).showSnackBar(
      const SnackBar(
        content: Text('Report generation coming soon'),
        backgroundColor: AppColors.primary,
      ),
    );
  }
}
