import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:url_launcher/url_launcher.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/voice_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class ParentHelperPage extends StatefulWidget {
  const ParentHelperPage({super.key});

  @override
  State<ParentHelperPage> createState() => _ParentHelperPageState();
}

class _ParentHelperPageState extends State<ParentHelperPage> {
  bool _consentGiven = false;

  final String _introText = 
      "Welcome to Parent Helper. This is for parents facing tough situations. "
      "You're already doing deadly work - this is just extra tools for your toolkit. "
      "Everything stays on your device.";

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () => context.go('/'),
          tooltip: 'Back to home',
        ),
        title: const Text('Parent Helper'),
        actions: [
          IconButton(
            icon: const Icon(Icons.home),
            onPressed: () => context.go('/'),
            tooltip: 'Home',
          ),
          ListenButton(
            text: _introText,
            size: 40,
          ),
        ],
      ),
      body: ScrollableWithIndicator(
        padding: const EdgeInsets.all(16),
        child: !_consentGiven ? _buildConsentScreen() : _buildContent(),
      ),
    );
  }

  Widget _buildConsentScreen() {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.stretch,
      children: [
        const Icon(Icons.family_restroom, size: 64, color: AppColors.primary),
        const SizedBox(height: 24),
        const Text(
          'Walking with Your Kids',
          style: TextStyle(fontSize: 24, fontWeight: FontWeight.bold),
          textAlign: TextAlign.center,
        ),
        const SizedBox(height: 16),
        Container(
          padding: const EdgeInsets.all(16),
          decoration: BoxDecoration(
            color: AppColors.primary.withOpacity(0.1),
            borderRadius: BorderRadius.circular(12),
            border: Border.all(color: AppColors.primary.withOpacity(0.2)),
          ),
          child: Column(
            children: [
              const Icon(Icons.lock, color: AppColors.primary),
              const SizedBox(height: 8),
              const Text(
                'This module contains sensitive information about family situations.',
                textAlign: TextAlign.center,
              ),
              const SizedBox(height: 8),
              const Text(
                'All information stays on your device only.',
                style: TextStyle(fontWeight: FontWeight.w500),
                textAlign: TextAlign.center,
              ),
            ],
          ),
        ),
        const SizedBox(height: 24),
        ElevatedButton(
          onPressed: () => setState(() => _consentGiven = true),
          style: ElevatedButton.styleFrom(
            padding: const EdgeInsets.all(16),
          ),
          child: const Text('I understand, let me in'),
        ),
      ],
    );
  }

  Widget _buildContent() {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.stretch,
      children: [
        _buildSection(
          icon: Icons.favorite,
          title: 'Keeping Your Mob Strong',
          voiceText: "Family is everything. When things get tough, remember you're not alone. "
              "There are services run by our mob who understand what you're going through.",
          children: [
            _buildInfoCard(
              'You are doing your best',
              'Parenting is hard work. Every little thing you do matters.',
            ),
            _buildInfoCard(
              'Ask for help',
              "It's not weakness to ask for support. That's how our communities have always worked.",
            ),
          ],
        ),
        const SizedBox(height: 16),
        _buildSection(
          icon: Icons.gavel,
          title: 'Know Your Rights',
          voiceText: "You have rights when dealing with child protection. "
              "You can ask questions. You can have a support person with you. "
              "You can get legal help.",
          children: [
            _buildRightCard('You can ask for an Aboriginal support person'),
            _buildRightCard('You can request meetings at times that work for you'),
            _buildRightCard('You can ask questions about any decision'),
            _buildRightCard('You have the right to appeal decisions'),
          ],
        ),
        const SizedBox(height: 16),
        _buildSection(
          icon: Icons.phone,
          title: 'Get Support',
          voiceText: "These services are run by our mob and they understand our ways.",
          children: [
            _buildServiceCard(
              '13YARN',
              '13 92 76',
              '24/7 crisis support by Aboriginal people',
              isMobRun: true,
            ),
            _buildServiceCard(
              'Aboriginal Legal Service',
              '1800 765 767',
              'Free legal help for Aboriginal people',
              isMobRun: true,
            ),
            _buildServiceCard(
              'Link-Up',
              '1800 624 332',
              'Support for Stolen Generations',
              isMobRun: true,
            ),
          ],
        ),
      ],
    );
  }

  Widget _buildSection({
    required IconData icon,
    required String title,
    required String voiceText,
    required List<Widget> children,
  }) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                Icon(icon, color: AppColors.primary),
                const SizedBox(width: 8),
                Expanded(
                  child: Text(
                    title,
                    style: const TextStyle(
                      fontSize: 18,
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                ),
                ListenButton(text: voiceText, size: 36),
              ],
            ),
            const SizedBox(height: 12),
            ...children,
          ],
        ),
      ),
    );
  }

  Widget _buildInfoCard(String title, String description) {
    return Container(
      margin: const EdgeInsets.only(bottom: 8),
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        color: Colors.grey.withOpacity(0.1),
        borderRadius: BorderRadius.circular(8),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(title, style: const TextStyle(fontWeight: FontWeight.w600)),
          const SizedBox(height: 4),
          Text(description, style: TextStyle(color: Colors.grey[600])),
        ],
      ),
    );
  }

  Widget _buildRightCard(String right) {
    return Container(
      margin: const EdgeInsets.only(bottom: 8),
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        border: Border.all(color: AppColors.primary.withOpacity(0.3)),
        borderRadius: BorderRadius.circular(8),
      ),
      child: Row(
        children: [
          const Icon(Icons.check_circle, color: AppColors.primary, size: 20),
          const SizedBox(width: 8),
          Expanded(child: Text(right)),
        ],
      ),
    );
  }

  Widget _buildServiceCard(
    String name,
    String phone,
    String description, {
    bool isMobRun = false,
  }) {
    return Container(
      margin: const EdgeInsets.only(bottom: 8),
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        color: isMobRun ? AppColors.primary.withOpacity(0.05) : null,
        border: Border.all(
          color: isMobRun ? AppColors.primary.withOpacity(0.2) : Colors.grey.withOpacity(0.3),
        ),
        borderRadius: BorderRadius.circular(8),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            children: [
              Expanded(
                child: Text(name, style: const TextStyle(fontWeight: FontWeight.w600)),
              ),
              if (isMobRun)
                Container(
                  padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 2),
                  decoration: BoxDecoration(
                    color: AppColors.primary.withOpacity(0.1),
                    borderRadius: BorderRadius.circular(12),
                  ),
                  child: const Text(
                    'Mob-run',
                    style: TextStyle(fontSize: 12, color: AppColors.primary),
                  ),
                ),
            ],
          ),
          const SizedBox(height: 4),
          Text(description, style: TextStyle(fontSize: 12, color: Colors.grey[600])),
          const SizedBox(height: 8),
          OutlinedButton.icon(
            onPressed: () => _makeCall(phone),
            icon: const Icon(Icons.phone, size: 16),
            label: Text(phone),
          ),
        ],
      ),
    );
  }

  Future<void> _makeCall(String number) async {
    final cleanNumber = number.replaceAll(' ', '');
    final uri = Uri.parse('tel:$cleanNumber');
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri);
    }
  }
}
