import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/voice_scripts.dart';
import '../../../core/widgets/shared_widgets.dart';

class OnboardingPage extends StatefulWidget {
  const OnboardingPage({super.key});

  @override
  State<OnboardingPage> createState() => _OnboardingPageState();
}

class _OnboardingPageState extends State<OnboardingPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final VoiceScripts _scripts = VoiceScripts();
  final PageController _pageController = PageController();
  final TextEditingController _nameController = TextEditingController();
  int _currentPage = 0;
  VoicePersona _selectedPersona = VoicePersona.aunty;
  bool _isPlaying = false;

  List<OnboardingSlide> get _slides => [
    OnboardingSlide(
      icon: Icons.people_outline,
      title: "Culturally Safe AI",
      subtitle: "For mob",
      description: "New tools change the world. This artificial intelligence thing they call AI is the biggest new tool around. Mob can use it to make life easier. This app shows you how.",
      voiceText: "New tools change the world. This artificial intelligence thing they call AI is the biggest new tool around. Mob can use it to make life easier. This app shows you how.",
    ),
    OnboardingSlide(
      icon: Icons.waving_hand,
      title: "G'day!",
      subtitle: "Your helper in your pocket",
      description: "I'm like your Aunty or Uncle in your pocket—here to help sort things out. What's your name?",
      voiceText: _scripts.welcomeFirstVisit,
      isNameEntry: true,
    ),
    OnboardingSlide(
      icon: Icons.psychology,
      title: "What is AI?",
      subtitle: "A clever tool",
      description: "Like a spear extends your arm, a woomera makes your throw more powerful. And a car helps your legs travel far. AI is like a car for the system - it helps you get through it quicker.",
      voiceText: _scripts.whatIsAI,
    ),
    OnboardingSlide(
      icon: Icons.record_voice_over,
      title: "Pick a Voice",
      subtitle: "Choose who talks to you",
      description: "Aunty's got sass. Uncle's got calm. Pick whoever feels right.",
      voiceText: _scripts.voiceSetup,
      isVoiceSelection: true,
    ),
    OnboardingSlide(
      icon: Icons.lock_outline,
      title: "Your Data, Your Business",
      subtitle: "Privacy promise",
      description: "Everything stays on your phone. Your stories are your stories.",
      voiceText: _scripts.privacyPromise,
    ),
    OnboardingSlide(
      icon: Icons.favorite,
      title: "Safety Net",
      subtitle: "13YARN when you need it",
      description: "If things get heavy, I can connect you straight to mob who get it.",
      voiceText: _scripts.safetyNetIntro,
    ),
    OnboardingSlide(
      icon: Icons.check_circle_outline,
      title: "You're all set!",
      subtitle: "Let's get started",
      description: "Tap any feature on the home screen. I'll read things out and guide you through.",
      voiceText: _scripts.onboardingComplete,
      isFinalSlide: true,
    ),
  ];

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    debugPrint('OnboardingPage: Initializing voice service...');
    await _voiceService.initialize();
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playCurrentSlide();
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _pageController.dispose();
    _nameController.dispose();
    super.dispose();
  }

  void _playCurrentSlide() {
    final slide = _slides[_currentPage];
    setState(() => _isPlaying = true);
    _voiceService.speak(slide.voiceText).then((_) {
      if (mounted) setState(() => _isPlaying = false);
    });
  }

  void _nextPage() {
    _voiceService.stop();
    if (_currentPage < _slides.length - 1) {
      _pageController.nextPage(
        duration: const Duration(milliseconds: 300),
        curve: Curves.easeInOut,
      );
    } else {
      _completeOnboarding();
    }
  }

  void _previousPage() {
    _voiceService.stop();
    if (_currentPage > 0) {
      _pageController.previousPage(
        duration: const Duration(milliseconds: 300),
        curve: Curves.easeInOut,
      );
    }
  }

  void _onPageChanged(int page) {
    setState(() => _currentPage = page);
    Future.delayed(const Duration(milliseconds: 300), () {
      if (mounted) _playCurrentSlide();
    });
  }

  void _playVoiceSample() {
    setState(() => _isPlaying = true);
    final text = _scripts.voiceSample(_selectedPersona == VoicePersona.aunty);
    _voiceService.speak(text).then((_) {
      if (mounted) setState(() => _isPlaying = false);
    });
  }

  void _selectPersona(VoicePersona persona) {
    _voiceService.stop();
    setState(() {
      _selectedPersona = persona;
      _voiceService.setPersona(persona);
    });
    _playVoiceSample();
  }

  Future<void> _completeOnboarding() async {
    final name = _nameController.text.trim();
    if (name.isNotEmpty) {
      await _prefs.setUserName(name);
    }
    await _prefs.setVoicePersona(_selectedPersona);
    await _prefs.setHasSeenWelcome(true);
    await _prefs.updateLastVisit();
    _voiceService.stop();
    if (mounted) {
      context.go('/');
    }
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 24, vertical: 16),
                child: Row(
                  mainAxisAlignment: MainAxisAlignment.spaceBetween,
                  children: [
                    if (_currentPage > 0)
                      IconButton(
                        icon: const Icon(Icons.arrow_back),
                        onPressed: _previousPage,
                      )
                    else
                      const SizedBox(width: 48),
                    Row(
                      children: List.generate(
                        _slides.length,
                        (index) => Container(
                          width: index == _currentPage ? 24 : 8,
                          height: 8,
                          margin: const EdgeInsets.symmetric(horizontal: 2),
                          decoration: BoxDecoration(
                            color: index == _currentPage
                                ? (isDark ? AppColors.primaryLight : AppColors.primary)
                                : (isDark ? Colors.grey[600] : Colors.grey[300]),
                            borderRadius: BorderRadius.circular(4),
                          ),
                        ),
                      ),
                    ),
                    TextButton(
                      onPressed: _completeOnboarding,
                      child: const Text('Skip'),
                    ),
                  ],
                ),
              ),
              Expanded(
                child: PageView.builder(
                  controller: _pageController,
                  onPageChanged: _onPageChanged,
                  itemCount: _slides.length,
                  itemBuilder: (context, index) {
                    final slide = _slides[index];
                    return _buildSlide(slide, isDark);
                  },
                ),
              ),
              Padding(
                padding: const EdgeInsets.all(24),
                child: Row(
                  children: [
                    IconButton(
                      icon: Icon(_isPlaying ? Icons.volume_up : Icons.volume_off),
                      onPressed: _isPlaying ? () => _voiceService.stop() : _playCurrentSlide,
                      tooltip: _isPlaying ? 'Stop' : 'Listen',
                    ),
                    const Spacer(),
                    ElevatedButton(
                      onPressed: _nextPage,
                      style: ElevatedButton.styleFrom(
                        padding: const EdgeInsets.symmetric(horizontal: 32, vertical: 16),
                      ),
                      child: Text(_currentPage == _slides.length - 1 ? "Let's go!" : 'Next'),
                    ),
                  ],
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildSlide(OnboardingSlide slide, bool isDark) {
    final primaryColor = isDark ? AppColors.primaryLight : AppColors.primary;

    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 24),
      child: Column(
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          Container(
            padding: const EdgeInsets.all(24),
            decoration: BoxDecoration(
              color: primaryColor.withOpacity(0.15),
              shape: BoxShape.circle,
            ),
            child: Icon(
              slide.icon,
              size: 56,
              color: primaryColor,
            ),
          ),
          const SizedBox(height: 32),
          Text(
            slide.title,
            style: Theme.of(context).textTheme.headlineLarge?.copyWith(
              fontWeight: FontWeight.bold,
            ),
            textAlign: TextAlign.center,
          ),
          const SizedBox(height: 8),
          Text(
            slide.subtitle,
            style: Theme.of(context).textTheme.titleLarge?.copyWith(
              color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
            ),
            textAlign: TextAlign.center,
          ),
          const SizedBox(height: 24),
          Text(
            slide.description,
            style: Theme.of(context).textTheme.bodyLarge,
            textAlign: TextAlign.center,
          ),
          if (slide.isNameEntry) ...[
            const SizedBox(height: 32),
            TextField(
              controller: _nameController,
              textCapitalization: TextCapitalization.words,
              decoration: InputDecoration(
                hintText: 'Your first name',
                prefixIcon: Icon(Icons.person_outline, color: primaryColor),
                suffixIcon: MicButton(onResult: (t) => _nameController.text = t),
                filled: true,
                fillColor: isDark ? Colors.grey[800] : Colors.white,
                border: OutlineInputBorder(
                  borderRadius: BorderRadius.circular(16),
                  borderSide: BorderSide.none,
                ),
                focusedBorder: OutlineInputBorder(
                  borderRadius: BorderRadius.circular(16),
                  borderSide: BorderSide(color: primaryColor, width: 2),
                ),
              ),
              style: TextStyle(
                fontSize: 18,
                color: isDark ? AppColors.textPrimaryDark : AppColors.textPrimary,
              ),
              textAlign: TextAlign.center,
            ),
            const SizedBox(height: 12),
            Text(
              "Optional - just skip if you'd rather not say",
              style: TextStyle(
                fontSize: 13,
                color: isDark ? AppColors.textTertiaryDark : AppColors.textTertiary,
                fontStyle: FontStyle.italic,
              ),
            ),
          ],
          if (slide.isVoiceSelection) ...[
            const SizedBox(height: 32),
            Row(
              children: [
                Expanded(
                  child: _VoiceOption(
                    label: "Woman's voice",
                    icon: Icons.person_outline,
                    isSelected: _selectedPersona == VoicePersona.aunty,
                    isPlaying: _isPlaying && _selectedPersona == VoicePersona.aunty,
                    onTap: () => _selectPersona(VoicePersona.aunty),
                  ),
                ),
                const SizedBox(width: 16),
                Expanded(
                  child: _VoiceOption(
                    label: "Man's voice",
                    icon: Icons.person_outline,
                    isSelected: _selectedPersona == VoicePersona.uncle,
                    isPlaying: _isPlaying && _selectedPersona == VoicePersona.uncle,
                    onTap: () => _selectPersona(VoicePersona.uncle),
                  ),
                ),
              ],
            ),
          ],
        ],
      ),
    );
  }
}

class OnboardingSlide {
  final IconData icon;
  final String title;
  final String subtitle;
  final String description;
  final String voiceText;
  final bool isNameEntry;
  final bool isVoiceSelection;
  final bool isFinalSlide;

  const OnboardingSlide({
    required this.icon,
    required this.title,
    required this.subtitle,
    required this.description,
    required this.voiceText,
    this.isNameEntry = false,
    this.isVoiceSelection = false,
    this.isFinalSlide = false,
  });
}

class _VoiceOption extends StatelessWidget {
  final String label;
  final IconData icon;
  final bool isSelected;
  final bool isPlaying;
  final VoidCallback onTap;

  const _VoiceOption({
    required this.label,
    required this.icon,
    required this.isSelected,
    required this.isPlaying,
    required this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final primaryColor = isDark ? AppColors.primaryLight : AppColors.primary;

    return GestureDetector(
      onTap: onTap,
      child: AnimatedContainer(
        duration: const Duration(milliseconds: 200),
        padding: const EdgeInsets.all(20),
        decoration: BoxDecoration(
          color: isSelected
              ? primaryColor.withOpacity(0.15)
              : (isDark ? Colors.grey[800] : Colors.grey[100]),
          borderRadius: BorderRadius.circular(16),
          border: Border.all(
            color: isSelected ? primaryColor : Colors.transparent,
            width: 2,
          ),
        ),
        child: Column(
          children: [
            Icon(
              icon,
              size: 40,
              color: isSelected ? primaryColor : (isDark ? Colors.grey[400] : Colors.grey[600]),
            ),
            const SizedBox(height: 12),
            Text(
              label,
              style: TextStyle(
                fontWeight: isSelected ? FontWeight.w600 : FontWeight.normal,
                color: isSelected ? primaryColor : null,
              ),
            ),
            if (isPlaying) ...[
              const SizedBox(height: 8),
              SizedBox(
                width: 16,
                height: 16,
                child: CircularProgressIndicator(
                  strokeWidth: 2,
                  color: primaryColor,
                ),
              ),
            ],
          ],
        ),
      ),
    );
  }
}
