import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:url_launcher/url_launcher.dart';
import 'package:shared_preferences/shared_preferences.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class NumbersPage extends StatefulWidget {
  const NumbersPage({super.key});

  @override
  State<NumbersPage> createState() => _NumbersPageState();
}

class _NumbersPageState extends State<NumbersPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ScrollController _scrollController = ScrollController();
  
  bool _hasPlayedIntro = false;
  bool _privateUnlocked = false;
  bool _showingPinEntry = false;
  String _enteredPin = '';
  String? _savedPin;
  bool _isSettingPin = false;
  
  final _pinController = TextEditingController();
  final _nameController = TextEditingController();
  final _phoneController = TextEditingController();
  
  List<Map<String, String>> _privateNumbers = [];

  final List<Map<String, dynamic>> _seriousBusinessNumbers = [
    {
      'category': 'Crisis & Mental Health',
      'icon': Icons.favorite_outline,
      'color': Colors.red,
      'numbers': [
        {'name': '13YARN (Mob Crisis Line)', 'phone': '13 92 76', 'description': '24/7 - Run by and for mob'},
        {'name': 'Lifeline', 'phone': '13 11 14', 'description': '24/7 crisis support'},
        {'name': 'Suicide Call Back', 'phone': '1300 659 467', 'description': '24/7'},
        {'name': 'Beyond Blue', 'phone': '1300 22 4636', 'description': '24/7'},
        {'name': 'Kids Helpline', 'phone': '1800 55 1800', 'description': '24/7 for young people'},
      ],
    },
    {
      'category': 'Family Violence & Safety',
      'icon': Icons.shield_outlined,
      'color': Colors.purple,
      'numbers': [
        {'name': '1800RESPECT', 'phone': '1800 737 732', 'description': '24/7 DV & sexual assault'},
        {'name': 'Men\'s Referral Service', 'phone': '1300 766 491', 'description': 'For men who want to change'},
        {'name': 'Brother to Brother', 'phone': '1800 435 799', 'description': '24/7 for Aboriginal men'},
      ],
    },
    {
      'category': 'Legal Help',
      'icon': Icons.gavel,
      'color': Colors.blue,
      'numbers': [
        {'name': 'Aboriginal Legal Service NSW/ACT', 'phone': '1800 765 767', 'description': 'Free legal advice'},
        {'name': 'Victorian Aboriginal Legal', 'phone': '1800 064 865', 'description': 'Free legal advice'},
        {'name': 'Legal Aid Helpline', 'phone': '1300 888 529', 'description': 'Legal assistance'},
      ],
    },
    {
      'category': 'Housing & Money',
      'icon': Icons.home_outlined,
      'color': Colors.green,
      'numbers': [
        {'name': 'Aboriginal Housing Office', 'phone': '1800 422 322', 'description': 'NSW housing help'},
        {'name': 'Link2home (Crisis)', 'phone': '1800 152 152', 'description': '24/7 homeless support'},
        {'name': 'National Debt Helpline', 'phone': '1800 007 007', 'description': 'Free financial counselling'},
      ],
    },
    {
      'category': 'Health',
      'icon': Icons.local_hospital_outlined,
      'color': Colors.teal,
      'numbers': [
        {'name': 'Emergency', 'phone': '000', 'description': 'Police, Fire, Ambulance'},
        {'name': 'Health Direct', 'phone': '1800 022 222', 'description': '24/7 health advice'},
        {'name': 'Poisons Info', 'phone': '13 11 26', 'description': '24/7'},
        {'name': 'Quitline', 'phone': '13 78 48', 'description': 'Quit smoking support'},
      ],
    },
    {
      'category': 'Government',
      'icon': Icons.account_balance_outlined,
      'color': Colors.orange,
      'numbers': [
        {'name': 'Centrelink', 'phone': '13 63 80', 'description': 'Indigenous services line'},
        {'name': 'Medicare', 'phone': '132 011', 'description': 'Health claims'},
        {'name': 'Child Support', 'phone': '131 272', 'description': 'Payments & enquiries'},
        {'name': 'NDIS', 'phone': '1800 800 110', 'description': 'Disability services'},
      ],
    },
  ];

  @override
  void initState() {
    super.initState();
    _loadPrivateData();
    _initializeAndPlay();
  }

  Future<void> _loadPrivateData() async {
    final prefs = await SharedPreferences.getInstance();
    _savedPin = prefs.getString('private_numbers_pin');
    final savedNumbers = prefs.getStringList('private_numbers') ?? [];
    setState(() {
      _privateNumbers = savedNumbers.map((s) {
        final parts = s.split('|||');
        return {'name': parts[0], 'phone': parts.length > 1 ? parts[1] : ''};
      }).toList();
    });
  }

  Future<void> _savePrivateNumbers() async {
    final prefs = await SharedPreferences.getInstance();
    final encoded = _privateNumbers.map((n) => '${n['name']}|||${n['phone']}').toList();
    await prefs.setStringList('private_numbers', encoded);
  }

  Future<void> _savePin(String pin) async {
    final prefs = await SharedPreferences.getInstance();
    await prefs.setString('private_numbers_pin', pin);
    setState(() {
      _savedPin = pin;
      _isSettingPin = false;
      _privateUnlocked = true;
    });
  }

  Future<void> _initializeAndPlay() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    if (mounted && !_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      await Future.delayed(const Duration(milliseconds: 300));
      _voiceService.speak(
        "Important Numbers - all the serious business contacts in one safe place. "
        "Tap any number to call. The private section is locked with a PIN - "
        "handy for numbers you need but don't want in your regular contacts."
      );
    }
  }

  @override
  void dispose() {
    _scrollController.dispose();
    _voiceService.stop();
    _pinController.dispose();
    _nameController.dispose();
    _phoneController.dispose();
    super.dispose();
  }

  Future<void> _makeCall(String phone) async {
    final cleaned = phone.replaceAll(RegExp(r'[^\d]'), '');
    final uri = Uri.parse('tel:$cleaned');
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri);
    }
  }

  void _tryUnlock() {
    if (_enteredPin == _savedPin) {
      setState(() {
        _privateUnlocked = true;
        _showingPinEntry = false;
        _enteredPin = '';
      });
      if (_prefs.autoPlayVoice) {
        _voiceService.speak("Private numbers unlocked.");
      }
    } else {
      setState(() => _enteredPin = '');
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Wrong PIN. Try again.')),
      );
    }
  }

  void _addPrivateNumber() {
    showDialog(
      context: context,
      builder: (ctx) => AlertDialog(
        title: const Text('Add Private Number'),
        content: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            TextField(
              controller: _nameController,
              decoration: InputDecoration(
                labelText: 'Name',
                hintText: 'e.g., My counsellor',
                suffixIcon: MicButton(onResult: (t) => _nameController.text = t),
              ),
            ),
            const SizedBox(height: 12),
            TextField(
              controller: _phoneController,
              decoration: InputDecoration(
                labelText: 'Phone Number',
                hintText: 'e.g., 0400 123 456',
                suffixIcon: MicButton(onResult: (t) => _phoneController.text = t),
              ),
              keyboardType: TextInputType.phone,
            ),
          ],
        ),
        actions: [
          TextButton(
            onPressed: () => Navigator.pop(ctx),
            child: const Text('Cancel'),
          ),
          ElevatedButton(
            onPressed: () {
              if (_nameController.text.isNotEmpty && _phoneController.text.isNotEmpty) {
                setState(() {
                  _privateNumbers.add({
                    'name': _nameController.text,
                    'phone': _phoneController.text,
                  });
                });
                _savePrivateNumbers();
                _nameController.clear();
                _phoneController.clear();
                Navigator.pop(ctx);
              }
            },
            child: const Text('Save'),
          ),
        ],
      ),
    );
  }

  void _deletePrivateNumber(int index) {
    showDialog(
      context: context,
      builder: (ctx) => AlertDialog(
        title: const Text('Delete Number?'),
        content: Text('Remove ${_privateNumbers[index]['name']}?'),
        actions: [
          TextButton(
            onPressed: () => Navigator.pop(ctx),
            child: const Text('Keep'),
          ),
          ElevatedButton(
            style: ElevatedButton.styleFrom(backgroundColor: Colors.red),
            onPressed: () {
              setState(() => _privateNumbers.removeAt(index));
              _savePrivateNumbers();
              Navigator.pop(ctx);
            },
            child: const Text('Delete'),
          ),
        ],
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              Padding(
                padding: const EdgeInsets.all(16.0),
                child: Row(
                  children: [
                    IconButton(
                      icon: const Icon(Icons.arrow_back),
                      onPressed: () => context.go('/'),
                    ),
                    const Spacer(),
                    IconButton(
                      icon: const Icon(Icons.volume_up),
                      onPressed: () {
                        _voiceService.speak(
                          "Tap any number to call. Private numbers are locked with a PIN."
                        );
                      },
                    ),
                    IconButton(
                      icon: const Icon(Icons.settings_outlined),
                      onPressed: () => context.go('/settings'),
                    ),
                  ],
                ),
              ),
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 20.0),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(
                          Icons.phone_outlined,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                          size: 28,
                        ),
                        const SizedBox(width: 12),
                        Text(
                          'Important Numbers',
                          style: Theme.of(context).textTheme.headlineMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(
                      'Serious business contacts in one safe place',
                      style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                  ],
                ),
              ),
              const SizedBox(height: 20),
              Expanded(
                child: Stack(
                  children: [
                    ListView(
                      controller: _scrollController,
                      padding: const EdgeInsets.symmetric(horizontal: 16),
                      children: [
                        _buildPrivateSection(isDark),
                    const SizedBox(height: 24),
                    Text(
                      'Serious Business',
                      style: Theme.of(context).textTheme.titleLarge?.copyWith(
                        fontWeight: FontWeight.bold,
                      ),
                    ),
                    const SizedBox(height: 4),
                    Text(
                      'Essential services and helplines',
                      style: TextStyle(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                    const SizedBox(height: 16),
                    ..._seriousBusinessNumbers.map((category) => 
                      _buildCategorySection(category, isDark)
                    ),
                    const SizedBox(height: 24),
                      ],
                    ),
                    ScrollIndicator(scrollController: _scrollController, bottomPadding: 20),
                  ],
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildPrivateSection(bool isDark) {
    return Card(
      color: isDark ? Colors.grey[900] : Colors.grey[100],
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                Icon(
                  _privateUnlocked ? Icons.lock_open : Icons.lock,
                  color: _privateUnlocked ? Colors.green : Colors.grey,
                ),
                const SizedBox(width: 8),
                Text(
                  'Private Numbers',
                  style: Theme.of(context).textTheme.titleMedium?.copyWith(
                    fontWeight: FontWeight.bold,
                  ),
                ),
                const Spacer(),
                if (_privateUnlocked)
                  IconButton(
                    icon: const Icon(Icons.add),
                    onPressed: _addPrivateNumber,
                    tooltip: 'Add number',
                  ),
                if (_privateUnlocked)
                  IconButton(
                    icon: const Icon(Icons.lock),
                    onPressed: () => setState(() => _privateUnlocked = false),
                    tooltip: 'Lock',
                  ),
              ],
            ),
            const SizedBox(height: 8),
            Text(
              'Numbers you need but don\'t want in your regular contacts',
              style: TextStyle(
                color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                fontSize: 13,
              ),
            ),
            const SizedBox(height: 12),
            
            if (!_privateUnlocked && !_showingPinEntry) ...[
              if (_savedPin == null) ...[
                Text(
                  'Set up a 4-digit PIN to protect private numbers',
                  style: TextStyle(color: isDark ? Colors.white70 : Colors.black54),
                ),
                const SizedBox(height: 12),
                ElevatedButton.icon(
                  onPressed: () => setState(() {
                    _isSettingPin = true;
                    _showingPinEntry = true;
                  }),
                  icon: const Icon(Icons.key),
                  label: const Text('Set PIN'),
                ),
              ] else ...[
                ElevatedButton.icon(
                  onPressed: () => setState(() => _showingPinEntry = true),
                  icon: const Icon(Icons.lock_open),
                  label: const Text('Unlock'),
                ),
              ],
            ],
            
            if (_showingPinEntry) ...[
              Text(
                _isSettingPin ? 'Choose a 4-digit PIN:' : 'Enter your PIN:',
                style: const TextStyle(fontWeight: FontWeight.w500),
              ),
              const SizedBox(height: 12),
              Row(
                mainAxisAlignment: MainAxisAlignment.center,
                children: List.generate(4, (i) => Container(
                  width: 40,
                  height: 50,
                  margin: const EdgeInsets.symmetric(horizontal: 4),
                  decoration: BoxDecoration(
                    color: isDark ? Colors.black26 : Colors.white,
                    borderRadius: BorderRadius.circular(8),
                    border: Border.all(
                      color: i < _enteredPin.length 
                        ? (isDark ? AppColors.primaryLight : AppColors.primary)
                        : Colors.grey,
                      width: 2,
                    ),
                  ),
                  child: Center(
                    child: Text(
                      i < _enteredPin.length ? '•' : '',
                      style: const TextStyle(fontSize: 24),
                    ),
                  ),
                )),
              ),
              const SizedBox(height: 16),
              Wrap(
                spacing: 8,
                runSpacing: 8,
                alignment: WrapAlignment.center,
                children: [
                  ...List.generate(9, (i) => _buildPinButton('${i + 1}', isDark)),
                  _buildPinButton('Clear', isDark, isAction: true),
                  _buildPinButton('0', isDark),
                  _buildPinButton(_isSettingPin ? 'Set' : 'Go', isDark, isAction: true),
                ],
              ),
              const SizedBox(height: 8),
              TextButton(
                onPressed: () => setState(() {
                  _showingPinEntry = false;
                  _enteredPin = '';
                  _isSettingPin = false;
                }),
                child: const Text('Cancel'),
              ),
            ],
            
            if (_privateUnlocked) ...[
              if (_privateNumbers.isEmpty)
                Padding(
                  padding: const EdgeInsets.symmetric(vertical: 16),
                  child: Text(
                    'No private numbers saved yet. Tap + to add one.',
                    style: TextStyle(
                      color: isDark ? Colors.white54 : Colors.black45,
                      fontStyle: FontStyle.italic,
                    ),
                  ),
                )
              else
                ...(_privateNumbers.asMap().entries.map((entry) => ListTile(
                  contentPadding: EdgeInsets.zero,
                  leading: CircleAvatar(
                    backgroundColor: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.2),
                    child: Icon(
                      Icons.person_outline,
                      color: isDark ? AppColors.primaryLight : AppColors.primary,
                    ),
                  ),
                  title: Text(entry.value['name'] ?? ''),
                  subtitle: Text(entry.value['phone'] ?? ''),
                  trailing: Row(
                    mainAxisSize: MainAxisSize.min,
                    children: [
                      IconButton(
                        icon: Icon(Icons.phone, color: Colors.green[400]),
                        onPressed: () => _makeCall(entry.value['phone'] ?? ''),
                      ),
                      IconButton(
                        icon: Icon(Icons.delete_outline, color: Colors.red[400]),
                        onPressed: () => _deletePrivateNumber(entry.key),
                      ),
                    ],
                  ),
                ))),
            ],
          ],
        ),
      ),
    );
  }

  Widget _buildPinButton(String label, bool isDark, {bool isAction = false}) {
    return SizedBox(
      width: 70,
      height: 50,
      child: ElevatedButton(
        onPressed: () {
          if (label == 'Clear') {
            setState(() => _enteredPin = '');
          } else if (label == 'Go') {
            _tryUnlock();
          } else if (label == 'Set') {
            if (_enteredPin.length == 4) {
              _savePin(_enteredPin);
              setState(() => _enteredPin = '');
            }
          } else if (_enteredPin.length < 4) {
            setState(() => _enteredPin += label);
          }
        },
        style: ElevatedButton.styleFrom(
          backgroundColor: isAction 
            ? (isDark ? AppColors.primaryLight : AppColors.primary)
            : (isDark ? Colors.grey[800] : Colors.white),
          foregroundColor: isAction 
            ? Colors.white 
            : (isDark ? Colors.white : Colors.black),
          shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(8)),
        ),
        child: Text(label, style: TextStyle(fontSize: isAction ? 14 : 18)),
      ),
    );
  }

  Widget _buildCategorySection(Map<String, dynamic> category, bool isDark) {
    final color = category['color'] as Color;
    final numbers = category['numbers'] as List<Map<String, String>>;
    
    return ExpansionTile(
      leading: CircleAvatar(
        backgroundColor: color.withOpacity(0.2),
        child: Icon(category['icon'] as IconData, color: color, size: 20),
      ),
      title: Text(
        category['category'] as String,
        style: const TextStyle(fontWeight: FontWeight.w600),
      ),
      subtitle: Text(
        '${numbers.length} numbers',
        style: TextStyle(
          color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
          fontSize: 12,
        ),
      ),
      children: numbers.map((num) => ListTile(
        contentPadding: const EdgeInsets.only(left: 72, right: 16),
        title: Text(num['name']!),
        subtitle: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(num['phone']!),
            if (num['description']?.isNotEmpty ?? false)
              Text(
                num['description']!,
                style: TextStyle(
                  fontSize: 11,
                  color: isDark ? Colors.white54 : Colors.black45,
                ),
              ),
          ],
        ),
        trailing: IconButton(
          icon: Icon(Icons.phone, color: Colors.green[400]),
          onPressed: () => _makeCall(num['phone']!),
        ),
        onTap: () => _makeCall(num['phone']!),
      )).toList(),
    );
  }
}
