import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:hive_flutter/hive_flutter.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/services/voice_scripts.dart';
import '../../../core/widgets/shared_widgets.dart';

class MoneyPage extends StatefulWidget {
  const MoneyPage({super.key});

  @override
  State<MoneyPage> createState() => _MoneyPageState();
}

class _MoneyPageState extends State<MoneyPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _api = ApiClient();
  final VoiceScripts _scripts = VoiceScripts();
  
  final _incomeController = TextEditingController();
  final _expensesController = TextEditingController();
  final _goalsController = TextEditingController();
  
  bool _hasPlayedIntro = false;
  bool _isLoading = false;
  Map<String, dynamic>? _budgetResult;
  String? _error;
  List<Map<String, dynamic>> _savedBudgets = [];
  
  String _paymentSchedule = 'Fortnightly Centrelink';
  final List<String> _scheduleOptions = [
    'Fortnightly Centrelink',
    'Weekly',
    'Fortnightly',
    'Monthly',
  ];

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
    _loadSavedBudgets();
  }
  
  Future<void> _loadSavedBudgets() async {
    try {
      final box = await Hive.openBox('money_budgets');
      final stored = box.get('budgets', defaultValue: []);
      setState(() {
        _savedBudgets = List<Map<String, dynamic>>.from(
          (stored as List).map((e) => Map<String, dynamic>.from(e))
        );
      });
      if (_savedBudgets.isNotEmpty) {
        final last = _savedBudgets.last;
        _incomeController.text = last['income'] ?? '';
        _expensesController.text = last['expenses'] ?? '';
        _goalsController.text = last['goals'] ?? '';
        _paymentSchedule = last['schedule'] ?? 'Fortnightly Centrelink';
      }
    } catch (e) {
      debugPrint('Error loading budgets: $e');
    }
  }
  
  Future<void> _saveBudget() async {
    try {
      final box = await Hive.openBox('money_budgets');
      final budget = {
        'id': DateTime.now().millisecondsSinceEpoch.toString(),
        'date': DateTime.now().toIso8601String(),
        'income': _incomeController.text,
        'expenses': _expensesController.text,
        'goals': _goalsController.text,
        'schedule': _paymentSchedule,
        'result': _budgetResult,
      };
      _savedBudgets.add(budget);
      await box.put('budgets', _savedBudgets);
    } catch (e) {
      debugPrint('Error saving budget: $e');
    }
  }

  Future<void> _initializeAndPlay() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    if (mounted && !_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      await Future.delayed(const Duration(milliseconds: 300));
      _voiceService.speak(_scripts.moneyYarnsIntro);
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _incomeController.dispose();
    _expensesController.dispose();
    _goalsController.dispose();
    super.dispose();
  }

  Future<void> _calculateBudget() async {
    if (_incomeController.text.isEmpty) {
      setState(() => _error = 'Please tell me about your income');
      return;
    }
    
    setState(() {
      _isLoading = true;
      _error = null;
    });
    
    try {
      final result = await _api.post('/api/money/budget', {
        'income': _incomeController.text,
        'expenses': _expensesController.text,
        'paymentSchedule': _paymentSchedule,
        'goals': _goalsController.text,
      });
      
      setState(() {
        _budgetResult = result;
        _isLoading = false;
      });
      
      await _saveBudget();
      
      if (_prefs.autoPlayVoice && result['summary'] != null) {
        _voiceService.speak(result['summary']);
      }
    } catch (e) {
      setState(() {
        _error = 'Having trouble working that out. Try again?';
        _isLoading = false;
      });
    }
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              Padding(
                padding: const EdgeInsets.all(16.0),
                child: Row(
                  children: [
                    IconButton(
                      icon: const Icon(Icons.arrow_back),
                      onPressed: () => context.go('/'),
                    ),
                    const Spacer(),
                    IconButton(
                      icon: const Icon(Icons.volume_up),
                      onPressed: () {
                        _voiceService.speak(
                          "Tell me what money comes in each payment, "
                          "and what bills and expenses you have. "
                          "I'll help work out a budget that makes sense."
                        );
                      },
                    ),
                    IconButton(
                      icon: const Icon(Icons.settings_outlined),
                      onPressed: () => context.go('/settings'),
                    ),
                  ],
                ),
              ),
              Expanded(
                child: ScrollableWithIndicator(
                  padding: const EdgeInsets.symmetric(horizontal: 20),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Row(
                        children: [
                          Icon(
                            Icons.account_balance_wallet,
                            color: isDark ? AppColors.primaryLight : AppColors.primary,
                            size: 28,
                          ),
                          const SizedBox(width: 12),
                          Text(
                            'Money Yarns',
                            style: Theme.of(context).textTheme.headlineMedium?.copyWith(
                              fontWeight: FontWeight.bold,
                            ),
                          ),
                        ],
                      ),
                      const SizedBox(height: 8),
                      Text(
                        "Let's work out your budget together",
                        style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                          color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                        ),
                      ),
                      const SizedBox(height: 24),
                      
                      if (_budgetResult == null) ...[
                        _buildInputSection(isDark),
                      ] else ...[
                        _buildResultSection(isDark),
                      ],
                      
                      if (_error != null)
                        Padding(
                          padding: const EdgeInsets.only(top: 16),
                          child: Text(
                            _error!,
                            style: TextStyle(color: Colors.red[400]),
                          ),
                        ),
                      
                      const SizedBox(height: 24),
                    ],
                  ),
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildInputSection(bool isDark) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          'Your Income',
          style: Theme.of(context).textTheme.titleMedium?.copyWith(
            fontWeight: FontWeight.bold,
          ),
        ),
        const SizedBox(height: 8),
        TextField(
          controller: _incomeController,
          decoration: InputDecoration(
            hintText: 'e.g., \$800 per fortnight from JobSeeker',
            border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
            prefixIcon: const Icon(Icons.attach_money),
            suffixIcon: MicButton(onResult: (t) => _incomeController.text = t),
          ),
          maxLines: 2,
        ),
        const SizedBox(height: 16),
        
        Text(
          'Payment Schedule',
          style: Theme.of(context).textTheme.titleMedium?.copyWith(
            fontWeight: FontWeight.bold,
          ),
        ),
        const SizedBox(height: 8),
        DropdownButtonFormField<String>(
          value: _paymentSchedule,
          decoration: InputDecoration(
            border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
            prefixIcon: const Icon(Icons.calendar_today),
          ),
          items: _scheduleOptions.map((s) => DropdownMenuItem(value: s, child: Text(s))).toList(),
          onChanged: (v) => setState(() => _paymentSchedule = v!),
        ),
        const SizedBox(height: 16),
        
        Text(
          'Regular Expenses',
          style: Theme.of(context).textTheme.titleMedium?.copyWith(
            fontWeight: FontWeight.bold,
          ),
        ),
        const SizedBox(height: 8),
        TextField(
          controller: _expensesController,
          decoration: InputDecoration(
            hintText: 'e.g., Rent \$200, power \$50, phone \$30, food \$150',
            border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
            prefixIcon: const Icon(Icons.receipt_long),
            suffixIcon: MicButton(onResult: (t) => _expensesController.text = t),
          ),
          maxLines: 3,
        ),
        const SizedBox(height: 16),
        
        Text(
          'Your Goals (optional)',
          style: Theme.of(context).textTheme.titleMedium?.copyWith(
            fontWeight: FontWeight.bold,
          ),
        ),
        const SizedBox(height: 8),
        TextField(
          controller: _goalsController,
          decoration: InputDecoration(
            hintText: 'e.g., Save for kids school stuff, pay off debt',
            border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
            prefixIcon: const Icon(Icons.star_outline),
            suffixIcon: MicButton(onResult: (t) => _goalsController.text = t),
          ),
          maxLines: 2,
        ),
        const SizedBox(height: 24),
        
        SizedBox(
          width: double.infinity,
          child: ElevatedButton.icon(
            onPressed: _isLoading ? null : _calculateBudget,
            icon: _isLoading 
              ? const SizedBox(
                  width: 20,
                  height: 20,
                  child: CircularProgressIndicator(strokeWidth: 2),
                )
              : const Icon(Icons.calculate),
            label: Text(_isLoading ? 'Working it out...' : 'Work Out My Budget'),
            style: ElevatedButton.styleFrom(
              padding: const EdgeInsets.symmetric(vertical: 16),
              shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
            ),
          ),
        ),
      ],
    );
  }

  Widget _buildResultSection(bool isDark) {
    final result = _budgetResult!;
    
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Card(
          child: Padding(
            padding: const EdgeInsets.all(16),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Row(
                  children: [
                    Icon(Icons.lightbulb_outline, color: isDark ? AppColors.primaryLight : AppColors.primary),
                    const SizedBox(width: 8),
                    Text(
                      'Your Budget Summary',
                      style: Theme.of(context).textTheme.titleMedium?.copyWith(
                        fontWeight: FontWeight.bold,
                      ),
                    ),
                    const Spacer(),
                    IconButton(
                      icon: const Icon(Icons.volume_up, size: 20),
                      onPressed: () => _voiceService.speak(result['summary'] ?? ''),
                    ),
                  ],
                ),
                const SizedBox(height: 12),
                Text(result['summary'] ?? 'No summary available'),
              ],
            ),
          ),
        ),
        const SizedBox(height: 16),
        
        if (result['priorityBills'] != null && (result['priorityBills'] as List).isNotEmpty) ...[
          Text(
            'Pay These First',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          ...((result['priorityBills'] as List).map((bill) => Padding(
            padding: const EdgeInsets.only(bottom: 8),
            child: Row(
              children: [
                Icon(Icons.priority_high, color: Colors.orange[400], size: 20),
                const SizedBox(width: 8),
                Expanded(child: Text(bill.toString())),
              ],
            ),
          ))),
          const SizedBox(height: 16),
        ],
        
        if (result['savingsTips'] != null && (result['savingsTips'] as List).isNotEmpty) ...[
          Text(
            'Saving Tips',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          ...((result['savingsTips'] as List).map((tip) => Padding(
            padding: const EdgeInsets.only(bottom: 8),
            child: Row(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Icon(Icons.check_circle_outline, color: Colors.green[400], size: 20),
                const SizedBox(width: 8),
                Expanded(child: Text(tip.toString())),
              ],
            ),
          ))),
          const SizedBox(height: 16),
        ],
        
        if (result['warnings'] != null && (result['warnings'] as List).isNotEmpty) ...[
          Card(
            color: Colors.orange.withOpacity(0.1),
            child: Padding(
              padding: const EdgeInsets.all(12),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Row(
                    children: [
                      Icon(Icons.warning_amber, color: Colors.orange[400]),
                      const SizedBox(width: 8),
                      Text(
                        'Watch Out For',
                        style: Theme.of(context).textTheme.titleSmall?.copyWith(
                          fontWeight: FontWeight.bold,
                        ),
                      ),
                    ],
                  ),
                  const SizedBox(height: 8),
                  ...((result['warnings'] as List).map((w) => Padding(
                    padding: const EdgeInsets.only(bottom: 4),
                    child: Text('• ${w.toString()}'),
                  ))),
                ],
              ),
            ),
          ),
          const SizedBox(height: 16),
        ],
        
        SizedBox(
          width: double.infinity,
          child: OutlinedButton.icon(
            onPressed: () {
              setState(() => _budgetResult = null);
            },
            icon: const Icon(Icons.refresh),
            label: const Text('Start Again'),
            style: OutlinedButton.styleFrom(
              padding: const EdgeInsets.symmetric(vertical: 16),
              shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
            ),
          ),
        ),
      ],
    );
  }
}
