import 'package:flutter/material.dart';
import 'package:url_launcher/url_launcher.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/voice_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/voice_scripts.dart';
import '../../../core/widgets/shared_widgets.dart';
import '../models/document_type.dart';
import '../models/document_analysis.dart';

class ResultsScreen extends StatefulWidget {
  final DocumentAnalysis analysis;

  const ResultsScreen({super.key, required this.analysis});

  @override
  State<ResultsScreen> createState() => _ResultsScreenState();
}

class _ResultsScreenState extends State<ResultsScreen> {
  final VoiceService _voiceService = VoiceService();
  final PreferencesService _prefs = PreferencesService();
  final VoiceScripts _scripts = VoiceScripts();
  final ScrollController _scrollController = ScrollController();
  bool _hasPlayedSummary = false;

  @override
  void initState() {
    super.initState();
    _initializeAndSpeak();
  }

  Future<void> _initializeAndSpeak() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted && !_hasPlayedSummary && _prefs.autoPlayVoice) {
      await Future.delayed(const Duration(milliseconds: 500));
      _speakSummary();
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _scrollController.dispose();
    super.dispose();
  }

  void _speakSummary() {
    if (_hasPlayedSummary) return;
    _hasPlayedSummary = true;

    final summary = widget.analysis.plainLanguageSummary;
    String message;
    
    switch (widget.analysis.urgency) {
      case UrgencyLevel.urgent:
        message = _scripts.letterHelperUrgent(summary);
        break;
      case UrgencyLevel.important:
        message = _scripts.letterHelperImportant(summary);
        break;
      case UrgencyLevel.normal:
        message = _scripts.letterHelperNormal(summary);
        break;
    }
    
    _voiceService.speak(message);
  }

  Future<void> _callNumber(String number) async {
    final uri = Uri.parse('tel:${number.replaceAll(' ', '')}');
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri);
    }
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final analysis = widget.analysis;

    return Scaffold(
      appBar: AppBar(
        title: Text(analysis.type.displayName),
        backgroundColor: Colors.transparent,
        elevation: 0,
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () => Navigator.pop(context),
        ),
        actions: [
          ListenButton(
            onPressed: () {
              _voiceService.stop();
              _speakSummary();
            },
          ),
        ],
      ),
      extendBodyBehindAppBar: true,
      body: CulturalBackground(
        child: SafeArea(
          child: Stack(
            children: [
              SingleChildScrollView(
                controller: _scrollController,
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.stretch,
                  children: [
                    _buildUrgencyBanner(isDark),
                    const SizedBox(height: 16),
                    _buildSummaryCard(isDark),
                    if (analysis.keyInfo.isNotEmpty) ...[
                      const SizedBox(height: 16),
                      _buildKeyInfoCard(isDark),
                    ],
                    if (analysis.actionSteps.isNotEmpty) ...[
                      const SizedBox(height: 16),
                      _buildActionStepsCard(isDark),
                    ],
                    if (analysis.warnings.isNotEmpty) ...[
                      const SizedBox(height: 16),
                      _buildWarningsCard(isDark),
                    ],
                    const SizedBox(height: 100),
                  ],
                ),
              ),
              ScrollIndicator(scrollController: _scrollController, bottomPadding: 20),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildUrgencyBanner(bool isDark) {
    final urgency = widget.analysis.urgency;
    Color bgColor;
    Color textColor = Colors.white;
    
    switch (urgency) {
      case UrgencyLevel.urgent:
        bgColor = AppColors.terracotta;
        break;
      case UrgencyLevel.important:
        bgColor = AppColors.ochre;
        textColor = Colors.black87;
        break;
      case UrgencyLevel.normal:
        bgColor = AppColors.eucalyptus;
        break;
    }

    return Container(
      padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
      decoration: BoxDecoration(
        color: bgColor,
        borderRadius: BorderRadius.circular(12),
      ),
      child: Row(
        children: [
          Icon(
            urgency == UrgencyLevel.urgent
                ? Icons.warning
                : urgency == UrgencyLevel.important
                    ? Icons.info
                    : Icons.check_circle,
            color: textColor,
          ),
          const SizedBox(width: 12),
          Expanded(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(
                  urgency.displayName,
                  style: TextStyle(
                    color: textColor,
                    fontWeight: FontWeight.bold,
                    fontSize: 16,
                  ),
                ),
                if (widget.analysis.deadline != null)
                  Text(
                    'Deadline: ${_formatDate(widget.analysis.deadline!)}',
                    style: TextStyle(color: textColor.withOpacity(0.9)),
                  ),
              ],
            ),
          ),
          IconButton(
            icon: Icon(Icons.volume_up, color: textColor),
            onPressed: () {
              _hasPlayedSummary = false;
              _speakSummary();
            },
          ),
        ],
      ),
    );
  }

  Widget _buildSummaryCard(bool isDark) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(20),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                Icon(
                  Icons.summarize,
                  color: isDark ? AppColors.primaryLight : AppColors.primary,
                ),
                const SizedBox(width: 12),
                Text(
                  'What This Means',
                  style: Theme.of(context).textTheme.titleMedium?.copyWith(
                    fontWeight: FontWeight.bold,
                  ),
                ),
                const Spacer(),
                IconButton(
                  icon: Icon(
                    Icons.volume_up,
                    color: isDark ? AppColors.primaryLight : AppColors.primary,
                  ),
                  onPressed: () {
                    _voiceService.stop();
                    _voiceService.speak(widget.analysis.plainLanguageSummary);
                  },
                ),
              ],
            ),
            const SizedBox(height: 12),
            Text(
              widget.analysis.plainLanguageSummary,
              style: Theme.of(context).textTheme.bodyLarge,
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildKeyInfoCard(bool isDark) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(20),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                Icon(
                  Icons.info_outline,
                  color: isDark ? AppColors.primaryLight : AppColors.primary,
                ),
                const SizedBox(width: 12),
                Text(
                  'Key Information',
                  style: Theme.of(context).textTheme.titleMedium?.copyWith(
                    fontWeight: FontWeight.bold,
                  ),
                ),
              ],
            ),
            const SizedBox(height: 16),
            ...widget.analysis.keyInfo.map((info) => Padding(
              padding: const EdgeInsets.only(bottom: 12),
              child: Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Container(
                    padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 4),
                    decoration: BoxDecoration(
                      color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.1),
                      borderRadius: BorderRadius.circular(6),
                    ),
                    child: Text(
                      info.category,
                      style: TextStyle(
                        fontSize: 12,
                        fontWeight: FontWeight.bold,
                        color: isDark ? AppColors.primaryLight : AppColors.primary,
                      ),
                    ),
                  ),
                  const SizedBox(width: 12),
                  Expanded(
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Text(
                          info.label,
                          style: const TextStyle(fontWeight: FontWeight.w500),
                        ),
                        Text(
                          info.value,
                          style: TextStyle(
                            color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                          ),
                        ),
                        if (info.explanation != null)
                          Text(
                            info.explanation!,
                            style: TextStyle(
                              fontSize: 12,
                              fontStyle: FontStyle.italic,
                              color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                            ),
                          ),
                      ],
                    ),
                  ),
                ],
              ),
            )),
          ],
        ),
      ),
    );
  }

  Widget _buildActionStepsCard(bool isDark) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(20),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                Icon(
                  Icons.checklist,
                  color: isDark ? AppColors.primaryLight : AppColors.primary,
                ),
                const SizedBox(width: 12),
                Text(
                  'What To Do',
                  style: Theme.of(context).textTheme.titleMedium?.copyWith(
                    fontWeight: FontWeight.bold,
                  ),
                ),
                const Spacer(),
                IconButton(
                  icon: Icon(
                    Icons.volume_up,
                    color: isDark ? AppColors.primaryLight : AppColors.primary,
                  ),
                  onPressed: () {
                    _voiceService.stop();
                    final steps = widget.analysis.actionSteps
                        .map((s) => 'Step ${s.stepNumber}: ${s.title}. ${s.description}')
                        .join('. ');
                    _voiceService.speak('Here\'s what you need to do. $steps');
                  },
                ),
              ],
            ),
            const SizedBox(height: 16),
            ...widget.analysis.actionSteps.map((step) => Padding(
              padding: const EdgeInsets.only(bottom: 16),
              child: Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Container(
                    width: 32,
                    height: 32,
                    decoration: BoxDecoration(
                      color: isDark ? AppColors.primaryLight : AppColors.primary,
                      shape: BoxShape.circle,
                    ),
                    child: Center(
                      child: Text(
                        '${step.stepNumber}',
                        style: const TextStyle(
                          color: Colors.white,
                          fontWeight: FontWeight.bold,
                        ),
                      ),
                    ),
                  ),
                  const SizedBox(width: 12),
                  Expanded(
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Row(
                          children: [
                            Expanded(
                              child: Text(
                                step.title,
                                style: const TextStyle(fontWeight: FontWeight.bold),
                              ),
                            ),
                            Container(
                              padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 2),
                              decoration: BoxDecoration(
                                color: step.timing.contains('TODAY')
                                    ? Colors.red.withOpacity(0.1)
                                    : Colors.grey.withOpacity(0.1),
                                borderRadius: BorderRadius.circular(4),
                              ),
                              child: Text(
                                step.timing,
                                style: TextStyle(
                                  fontSize: 11,
                                  fontWeight: FontWeight.bold,
                                  color: step.timing.contains('TODAY') ? Colors.red : Colors.grey,
                                ),
                              ),
                            ),
                          ],
                        ),
                        const SizedBox(height: 4),
                        Text(step.description),
                        if (step.phoneNumber != null) ...[
                          const SizedBox(height: 8),
                          OutlinedButton.icon(
                            onPressed: () => _callNumber(step.phoneNumber!),
                            icon: const Icon(Icons.call, size: 16),
                            label: Text('Call ${step.phoneNumber}'),
                            style: OutlinedButton.styleFrom(
                              padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 8),
                            ),
                          ),
                        ],
                      ],
                    ),
                  ),
                ],
              ),
            )),
          ],
        ),
      ),
    );
  }

  Widget _buildWarningsCard(bool isDark) {
    return Card(
      color: Colors.orange.withOpacity(0.1),
      child: Padding(
        padding: const EdgeInsets.all(20),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                const Icon(Icons.warning_amber, color: Colors.orange),
                const SizedBox(width: 12),
                Text(
                  'Watch Out For',
                  style: Theme.of(context).textTheme.titleMedium?.copyWith(
                    fontWeight: FontWeight.bold,
                    color: Colors.orange,
                  ),
                ),
              ],
            ),
            const SizedBox(height: 12),
            ...widget.analysis.warnings.map((warning) => Padding(
              padding: const EdgeInsets.only(bottom: 8),
              child: Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  const Text('\u2022 ', style: TextStyle(color: Colors.orange)),
                  Expanded(child: Text(warning)),
                ],
              ),
            )),
          ],
        ),
      ),
    );
  }

  String _formatDate(DateTime date) {
    return '${date.day}/${date.month}/${date.year}';
  }
}
