import 'dart:io';
import 'package:flutter/material.dart';
import 'package:image_picker/image_picker.dart';
import '../../core/theme/app_theme.dart';
import '../../core/services/voice_service.dart';
import '../../core/services/preferences_service.dart';
import '../../core/services/ocr_service.dart';
import '../../core/services/document_analysis_service.dart';
import '../../core/services/voice_scripts.dart';
import '../../core/widgets/shared_widgets.dart';
import 'models/document_analysis.dart';
import 'screens/results_screen.dart';

class LetterHelperPage extends StatefulWidget {
  const LetterHelperPage({super.key});

  @override
  State<LetterHelperPage> createState() => _LetterHelperPageState();
}

class _LetterHelperPageState extends State<LetterHelperPage> {
  final VoiceService _voiceService = VoiceService();
  final PreferencesService _prefs = PreferencesService();
  final VoiceScripts _scripts = VoiceScripts();
  final OCRService _ocrService = OCRService();
  final DocumentAnalysisService _analysisService = DocumentAnalysisService();
  final TextEditingController _textController = TextEditingController();
  final ScrollController _scrollController = ScrollController();
  
  bool _hasPlayedIntro = false;
  bool _isProcessing = false;
  String _processingStatus = '';
  File? _selectedImage;

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playIntro();
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _textController.dispose();
    _scrollController.dispose();
    super.dispose();
  }

  void _playIntro() {
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      _voiceService.speak(_scripts.letterHelperIntro);
    }
  }

  Future<void> _takePhoto() async {
    final picker = ImagePicker();
    final image = await picker.pickImage(
      source: ImageSource.camera,
      imageQuality: 85,
    );
    
    if (image != null) {
      setState(() {
        _selectedImage = File(image.path);
      });
      _processImage(File(image.path));
    }
  }

  Future<void> _pickFromGallery() async {
    final picker = ImagePicker();
    final image = await picker.pickImage(
      source: ImageSource.gallery,
      imageQuality: 85,
    );
    
    if (image != null) {
      setState(() {
        _selectedImage = File(image.path);
      });
      _processImage(File(image.path));
    }
  }

  Future<void> _processImage(File imageFile) async {
    setState(() {
      _isProcessing = true;
      _processingStatus = 'Reading your letter...';
    });

    await _voiceService.speak(_scripts.letterHelperProcessing);

    try {
      final extractedText = await _ocrService.extractTextFromImage(imageFile);
      
      if (extractedText.isEmpty) {
        setState(() {
          _isProcessing = false;
          _processingStatus = '';
        });
        await _voiceService.speak(_scripts.isAunty
            ? "Hmm, I couldn't read the text in that photo. Try taking another one with better lighting, or type in what it says."
            : "Couldn't make out the text. Try a clearer photo or type it in.");
        return;
      }

      setState(() {
        _processingStatus = 'Working out what it means...';
      });

      final analysis = await _analysisService.analyzeDocument(extractedText);
      
      setState(() {
        _isProcessing = false;
        _processingStatus = '';
      });

      if (mounted) {
        Navigator.push(
          context,
          MaterialPageRoute(
            builder: (context) => ResultsScreen(analysis: analysis),
          ),
        );
      }
    } catch (e) {
      setState(() {
        _isProcessing = false;
        _processingStatus = '';
      });
      
      await _voiceService.speak(_scripts.errorMessage);

      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(
            content: Text(_getErrorMessage(e)),
            action: SnackBarAction(label: 'Try Again', onPressed: () => _processImage(imageFile)),
          ),
        );
      }
    }
  }

  String _getErrorMessage(dynamic error) {
    final errorStr = error.toString().toLowerCase();
    if (errorStr.contains('api key')) {
      return 'API key not configured. Please check settings.';
    } else if (errorStr.contains('network') || errorStr.contains('socket') || errorStr.contains('timeout')) {
      return 'No internet connection. Please try again when online.';
    } else {
      return 'Something went wrong. Please try again.';
    }
  }

  Future<void> _processText() async {
    final text = _textController.text.trim();
    if (text.isEmpty) {
      await _voiceService.speak(_scripts.isAunty
          ? "You haven't typed anything yet."
          : "Nothing to analyze. Type or paste some text first.");
      return;
    }

    setState(() {
      _isProcessing = true;
      _processingStatus = 'Working out what it means...';
    });

    await _voiceService.speak(_scripts.letterHelperProcessing);

    try {
      final analysis = await _analysisService.analyzeDocument(text);
      
      setState(() {
        _isProcessing = false;
        _processingStatus = '';
      });

      if (mounted) {
        Navigator.push(
          context,
          MaterialPageRoute(
            builder: (context) => ResultsScreen(analysis: analysis),
          ),
        );
      }
    } catch (e) {
      setState(() {
        _isProcessing = false;
        _processingStatus = '';
      });
      
      await _voiceService.speak(_scripts.errorMessage);

      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(
            content: Text(_getErrorMessage(e)),
            action: SnackBarAction(label: 'Try Again', onPressed: _processText),
          ),
        );
      }
    }
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      appBar: AppBar(
        title: const Text('Letter Helper'),
        leading: const YarnHomeButton(),
        backgroundColor: Colors.transparent,
        elevation: 0,
        actions: [
          ListenButton(
            onPressed: () {
              _voiceService.speak(_scripts.letterHelperIntro);
            },
          ),
        ],
      ),
      extendBodyBehindAppBar: true,
      body: CulturalBackground(
        child: SafeArea(
          child: _isProcessing
              ? _buildProcessingView(isDark)
              : _buildInputView(isDark),
        ),
      ),
    );
  }

  Widget _buildProcessingView(bool isDark) {
    return Center(
      child: Padding(
        padding: const EdgeInsets.all(32),
        child: Column(
          mainAxisAlignment: MainAxisAlignment.center,
          children: [
            const CircularProgressIndicator(),
            const SizedBox(height: 24),
            Text(
              _processingStatus,
              style: Theme.of(context).textTheme.titleMedium,
              textAlign: TextAlign.center,
            ),
            const SizedBox(height: 16),
            Text(
              'This might take a moment...',
              style: TextStyle(
                color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
              ),
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildInputView(bool isDark) {
    return Stack(
      children: [
        SingleChildScrollView(
          controller: _scrollController,
          padding: const EdgeInsets.all(16),
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.stretch,
            children: [
              Card(
                child: Padding(
                  padding: const EdgeInsets.all(20),
                  child: Column(
                    children: [
                      Icon(
                        Icons.camera_alt,
                        size: 48,
                        color: isDark ? AppColors.primaryLight : AppColors.primary,
                      ),
                      const SizedBox(height: 16),
                      Text(
                        'Take a Photo',
                        style: Theme.of(context).textTheme.titleLarge,
                      ),
                      const SizedBox(height: 8),
                      Text(
                        'Point your camera at the letter and take a clear photo',
                        textAlign: TextAlign.center,
                        style: TextStyle(
                          color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                        ),
                      ),
                      const SizedBox(height: 20),
                      Row(
                        children: [
                          Expanded(
                            child: ElevatedButton.icon(
                              onPressed: _takePhoto,
                              icon: const Icon(Icons.camera_alt),
                              label: const Text('Camera'),
                              style: ElevatedButton.styleFrom(
                                backgroundColor: isDark ? AppColors.primaryLight : AppColors.primary,
                                foregroundColor: Colors.white,
                                padding: const EdgeInsets.symmetric(vertical: 16),
                              ),
                            ),
                          ),
                          const SizedBox(width: 12),
                          Expanded(
                            child: OutlinedButton.icon(
                              onPressed: _pickFromGallery,
                              icon: const Icon(Icons.photo_library),
                              label: const Text('Gallery'),
                              style: OutlinedButton.styleFrom(
                                padding: const EdgeInsets.symmetric(vertical: 16),
                              ),
                            ),
                          ),
                        ],
                      ),
                    ],
                  ),
                ),
              ),
              const SizedBox(height: 24),
              Row(
                children: [
                  Expanded(child: Divider(color: isDark ? Colors.white24 : Colors.black12)),
                  Padding(
                    padding: const EdgeInsets.symmetric(horizontal: 16),
                    child: Text(
                      'OR',
                      style: TextStyle(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                        fontWeight: FontWeight.bold,
                      ),
                    ),
                  ),
                  Expanded(child: Divider(color: isDark ? Colors.white24 : Colors.black12)),
                ],
              ),
              const SizedBox(height: 24),
              Card(
                child: Padding(
                  padding: const EdgeInsets.all(20),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Row(
                        children: [
                          Icon(
                            Icons.edit_document,
                            color: isDark ? AppColors.primaryLight : AppColors.primary,
                          ),
                          const SizedBox(width: 12),
                          Text(
                            'Type or Paste Text',
                            style: Theme.of(context).textTheme.titleMedium,
                          ),
                        ],
                      ),
                      const SizedBox(height: 16),
                      TextField(
                        controller: _textController,
                        maxLines: 8,
                        decoration: InputDecoration(
                          hintText: 'Paste or type the letter text here...',
                          border: OutlineInputBorder(
                            borderRadius: BorderRadius.circular(12),
                          ),
                          filled: true,
                          fillColor: isDark ? Colors.black26 : Colors.grey[50],
                        ),
                      ),
                      const SizedBox(height: 16),
                      SizedBox(
                        width: double.infinity,
                        child: ElevatedButton.icon(
                          onPressed: _processText,
                          icon: const Icon(Icons.translate),
                          label: const Text('Explain This'),
                          style: ElevatedButton.styleFrom(
                            backgroundColor: isDark ? AppColors.primaryLight : AppColors.primary,
                            foregroundColor: Colors.white,
                            padding: const EdgeInsets.symmetric(vertical: 16),
                          ),
                        ),
                      ),
                    ],
                  ),
                ),
              ),
              const SizedBox(height: 100),
            ],
          ),
        ),
        ScrollIndicator(scrollController: _scrollController, bottomPadding: 20),
      ],
    );
  }
}
