import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class LearnPage extends StatefulWidget {
  const LearnPage({super.key});

  @override
  State<LearnPage> createState() => _LearnPageState();
}

class _LearnPageState extends State<LearnPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ScrollController _scrollController = ScrollController();
  final Set<String> _completedSteps = {};
  bool _hasPlayedIntro = false;

  final List<Map<String, dynamic>> _lessons = [
    {
      'id': 'what-is-ai',
      'title': 'What is AI?',
      'description': 'Learn the basics in simple terms',
      'icon': Icons.lightbulb_outline,
      'steps': [
        {'title': 'AI is a smart helper', 'content': 'AI is like a very smart computer program that can understand what you say and help you with tasks. Think of it as a helpful assistant that never gets tired.'},
        {'title': 'AI learns from examples', 'content': 'AI learned by reading millions of books, websites, and conversations. This is why it can answer questions about many different topics.'},
        {'title': 'AI makes mistakes sometimes', 'content': 'AI is helpful but not perfect. Always double-check important information, especially for medical, legal, or financial matters.'},
      ],
    },
    {
      'id': 'talking-to-ai',
      'title': 'How to Talk to AI',
      'description': 'Get the best answers',
      'icon': Icons.chat_bubble_outline,
      'steps': [
        {'title': 'Be clear and specific', 'content': 'Instead of asking "Help me with my letter", try "Help me write a letter to my landlord asking for more time to pay rent".'},
        {'title': 'Ask follow-up questions', 'content': 'If the first answer isn\'t quite right, you can ask AI to try again. Say things like "Can you make it simpler?"'},
        {'title': 'Give context', 'content': 'Tell AI who you are and what you\'re trying to do. For example: "I\'m a single parent trying to understand this school form."'},
      ],
    },
    {
      'id': 'document-help',
      'title': 'Getting Help with Documents',
      'description': 'Understand and respond to paperwork',
      'icon': Icons.description_outlined,
      'steps': [
        {'title': 'Copy and paste your document', 'content': 'Open your document and copy all the text. Then paste it into the document helper. Ask "Can you explain this in simple words?"'},
        {'title': 'Ask specific questions', 'content': 'After pasting your document, ask specific questions like: "What does this letter want me to do?" or "When is the deadline?"'},
        {'title': 'Get an action plan', 'content': 'Ask AI "What steps do I need to take?" and it will give you a clear list of things to do.'},
      ],
    },
    {
      'id': 'staying-safe',
      'title': 'Staying Safe with AI',
      'description': 'Important tips for using AI wisely',
      'icon': Icons.shield_outlined,
      'steps': [
        {'title': 'Don\'t share personal details', 'content': 'Never share your passwords, bank details, Medicare number, or address with AI. AI doesn\'t need this information to help you.'},
        {'title': 'Check important information', 'content': 'For legal, medical, or financial matters, use AI to help you understand things, but always check with a real professional.'},
        {'title': 'AI is a helper, not a decision maker', 'content': 'AI can give you information and options, but the final decision should always be yours.'},
      ],
    },
    {
      'id': 'getting-help',
      'title': 'Getting Free Help',
      'description': 'Legal aid, Centrelink, housing services',
      'icon': Icons.people_outline,
      'steps': [
        {'title': 'Aboriginal Legal Services', 'content': 'ATSILS provides free legal help specifically for Aboriginal and Torres Strait Islander mob. Call 1800 765 767 (free call).'},
        {'title': 'Centrelink Indigenous Line', 'content': 'There\'s a special Centrelink line just for mob. Call 1800 136 380, Monday to Friday, 8am to 5pm.'},
        {'title': 'Housing help', 'content': 'If you\'re having problems with your landlord or facing eviction, there are free tenant advocacy services. Call Legal Aid in your state.'},
      ],
    },
  ];

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    debugPrint('LearnPage: Initializing voice service...');
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playIntro();
    }
  }

  @override
  void dispose() {
    _scrollController.dispose();
    _voiceService.stop();
    super.dispose();
  }

  void _playIntro() {
    debugPrint('LearnPage: _playIntro called, autoPlay=${_prefs.autoPlayVoice}');
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      _voiceService.speak(
        "This page has guides to help you with everyday stuff. "
        "Like how to talk to AI, how to get help with documents, and where to find free services. "
        "Tap any topic to learn more, or tap the speaker icon to hear it read aloud."
      );
    }
  }

  void _toggleStep(String lessonId, int stepIndex) {
    final key = '$lessonId-$stepIndex';
    setState(() {
      if (_completedSteps.contains(key)) {
        _completedSteps.remove(key);
      } else {
        _completedSteps.add(key);
      }
    });
  }

  int _getLessonProgress(String lessonId, int totalSteps) {
    int completed = 0;
    for (int i = 0; i < totalSteps; i++) {
      if (_completedSteps.contains('$lessonId-$i')) {
        completed++;
      }
    }
    return completed;
  }

  void _speakStep(Map<String, dynamic> step) {
    _voiceService.stop();
    final title = step['title'] as String;
    final content = step['content'] as String;
    _voiceService.speak('$title. $content');
  }

  void _speakLesson(Map<String, dynamic> lesson) {
    _voiceService.stop();
    final title = lesson['title'] as String;
    final description = lesson['description'] as String;
    final steps = lesson['steps'] as List;
    final stepTitles = steps.map((s) => (s as Map<String, dynamic>)['title'] as String).join(', ');
    _voiceService.speak('$title. $description. This lesson covers: $stepTitles');
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () {
            _voiceService.stop();
            context.go('/');
          },
        ),
        title: const Text('Everyday Help'),
        actions: [
          IconButton(
            icon: const Icon(Icons.volume_up),
            tooltip: 'Read page aloud',
            onPressed: () {
              _voiceService.speak(
                "This page has guides to help you learn. "
                "Tap any topic to expand it, then tap the speaker icon to hear each step."
              );
            },
          ),
          const SettingsButton(),
        ],
      ),
      body: CulturalBackground(
        child: Stack(
          children: [
            ListView.builder(
              controller: _scrollController,
              padding: const EdgeInsets.all(16),
              itemCount: _lessons.length + 1,
              itemBuilder: (context, index) {
            if (index == 0) {
              return Card(
              margin: const EdgeInsets.only(bottom: 16),
              color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.1),
              child: InkWell(
                onTap: () {
                  _voiceService.stop();
                  context.go('/welcome-yarn');
                },
                borderRadius: BorderRadius.circular(12),
                child: Padding(
                  padding: const EdgeInsets.all(16),
                  child: Row(
                    children: [
                      Container(
                        padding: const EdgeInsets.all(12),
                        decoration: BoxDecoration(
                          color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.2),
                          shape: BoxShape.circle,
                        ),
                        child: Icon(
                          Icons.auto_stories,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                        ),
                      ),
                      const SizedBox(width: 16),
                      Expanded(
                        child: Column(
                          crossAxisAlignment: CrossAxisAlignment.start,
                          children: [
                            Text(
                              'New to AI? Start here',
                              style: Theme.of(context).textTheme.titleMedium?.copyWith(
                                fontWeight: FontWeight.bold,
                              ),
                            ),
                            const SizedBox(height: 4),
                            Text(
                              'A yarn about what AI is and how it can help',
                              style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                                color: isDark ? Colors.grey[400] : Colors.grey[600],
                              ),
                            ),
                          ],
                        ),
                      ),
                      Icon(
                        Icons.arrow_forward_ios,
                        color: isDark ? AppColors.primaryLight : AppColors.primary,
                        size: 16,
                      ),
                    ],
                  ),
                ),
              ),
            );
          }
          
          final lesson = _lessons[index - 1];
          final steps = lesson['steps'] as List;
          final progress = _getLessonProgress(lesson['id'], steps.length);
          final isComplete = progress == steps.length;
          
          return Card(
            margin: const EdgeInsets.only(bottom: 12),
            child: ExpansionTile(
              leading: Container(
                padding: const EdgeInsets.all(10),
                decoration: BoxDecoration(
                  color: (isComplete ? Colors.green : (isDark ? AppColors.primaryLight : AppColors.primary)).withOpacity(0.15),
                  borderRadius: BorderRadius.circular(10),
                ),
                child: Icon(
                  lesson['icon'] as IconData,
                  color: isComplete ? Colors.green : (isDark ? AppColors.primaryLight : AppColors.primary),
                ),
              ),
              title: Row(
                children: [
                  Expanded(child: Text(lesson['title'] as String)),
                  if (isComplete)
                    const Icon(Icons.check_circle, color: Colors.green, size: 20),
                  IconButton(
                    icon: const Icon(Icons.volume_up, size: 20),
                    tooltip: 'Listen to lesson overview',
                    onPressed: () => _speakLesson(lesson),
                  ),
                ],
              ),
              subtitle: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(lesson['description'] as String),
                  if (progress > 0 && !isComplete) ...[
                    const SizedBox(height: 4),
                    LinearProgressIndicator(
                      value: progress / steps.length,
                      backgroundColor: isDark ? Colors.grey[800] : Colors.grey[300],
                      color: isDark ? AppColors.primaryLight : AppColors.primary,
                    ),
                  ],
                ],
              ),
              children: [
                ...steps.asMap().entries.map((entry) {
                  final stepIndex = entry.key;
                  final step = entry.value as Map<String, dynamic>;
                  final isStepComplete = _completedSteps.contains('${lesson['id']}-$stepIndex');
                  
                  return ListTile(
                    leading: GestureDetector(
                      onTap: () => _toggleStep(lesson['id'], stepIndex),
                      child: Icon(
                        isStepComplete ? Icons.check_circle : Icons.circle_outlined,
                        color: isStepComplete ? Colors.green : Colors.grey,
                      ),
                    ),
                    title: Text(
                      step['title'] as String,
                      style: TextStyle(
                        fontWeight: FontWeight.w600,
                        decoration: isStepComplete ? TextDecoration.lineThrough : null,
                      ),
                    ),
                    subtitle: Padding(
                      padding: const EdgeInsets.only(top: 4),
                      child: Text(step['content'] as String),
                    ),
                    trailing: IconButton(
                      icon: Icon(
                        Icons.volume_up,
                        size: 20,
                        color: isDark ? AppColors.primaryLight : AppColors.primary,
                      ),
                      tooltip: 'Listen to this step',
                      onPressed: () => _speakStep(step),
                    ),
                  );
                }),
              ],
            ),
          );
          },
        ),
            ScrollIndicator(scrollController: _scrollController, bottomPadding: 20),
          ],
        ),
      ),
    );
  }
}
