import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:hive_flutter/hive_flutter.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/widgets/shared_widgets.dart';

class JournalPage extends StatefulWidget {
  const JournalPage({super.key});

  @override
  State<JournalPage> createState() => _JournalPageState();
}

class _JournalPageState extends State<JournalPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _api = ApiClient();
  final _entryController = TextEditingController();
  final ScrollController _scrollController = ScrollController();
  
  bool _hasPlayedIntro = false;
  List<Map<String, dynamic>> _entries = [];
  Map<String, dynamic>? _currentPrompt;
  bool _isLoadingPrompt = false;

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
    _loadEntries();
    _loadPrompt();
  }

  Future<void> _initializeAndPlay() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    if (mounted && !_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      await Future.delayed(const Duration(milliseconds: 300));
      _voiceService.speak(
        "Hey friend, welcome to Story Keeper - your quiet yarn spot. "
        "This is just for you. Write down whatever's on your mind, or use the prompt I give you to get started. "
        "Tap Save to keep today's yarn. Everything stays on your phone only - no one else can ever see it. "
        "This is your safe space to let it out."
      );
    }
  }

  Future<void> _loadEntries() async {
    try {
      final box = await Hive.openBox('journal_entries');
      final stored = box.get('entries', defaultValue: []);
      setState(() {
        _entries = List<Map<String, dynamic>>.from(
          (stored as List).map((e) => Map<String, dynamic>.from(e))
        );
        _entries.sort((a, b) => (b['date'] ?? '').compareTo(a['date'] ?? ''));
      });
    } catch (e) {
      debugPrint('Error loading entries: $e');
    }
  }

  Future<void> _saveEntries() async {
    try {
      final box = await Hive.openBox('journal_entries');
      await box.put('entries', _entries);
    } catch (e) {
      debugPrint('Error saving entries: $e');
    }
  }

  Future<void> _loadPrompt() async {
    setState(() => _isLoadingPrompt = true);
    try {
      final result = await _api.post('/api/journal/prompt', {
        'mood': 'neutral',
        'topic': 'general reflection',
      });
      setState(() {
        _currentPrompt = result;
        _isLoadingPrompt = false;
      });
    } catch (e) {
      setState(() {
        _currentPrompt = {'prompt': 'What made you smile today?'};
        _isLoadingPrompt = false;
      });
    }
  }

  @override
  void dispose() {
    _scrollController.dispose();
    _voiceService.stop();
    _entryController.dispose();
    super.dispose();
  }

  void _addEntry() {
    if (_entryController.text.isEmpty) return;

    final newEntry = {
      'id': DateTime.now().millisecondsSinceEpoch.toString(),
      'date': DateTime.now().toIso8601String(),
      'content': _entryController.text,
      'prompt': _currentPrompt?['prompt'] ?? '',
    };

    setState(() {
      _entries.insert(0, newEntry);
    });
    _saveEntries();
    _entryController.clear();
    
    if (_prefs.autoPlayVoice) {
      _voiceService.speak(_currentPrompt?['affirmation'] ?? 'Thanks for sharing that with me.');
    }
    
    _loadPrompt();
  }

  void _deleteEntry(String id) {
    setState(() {
      _entries.removeWhere((e) => e['id'] == id);
    });
    _saveEntries();
  }

  String _formatDate(String? isoDate) {
    if (isoDate == null) return '';
    try {
      final date = DateTime.parse(isoDate);
      final now = DateTime.now();
      final diff = now.difference(date);
      
      if (diff.inDays == 0) return 'Today';
      if (diff.inDays == 1) return 'Yesterday';
      if (diff.inDays < 7) return '${diff.inDays} days ago';
      return '${date.day}/${date.month}/${date.year}';
    } catch (e) {
      return '';
    }
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              Padding(
                padding: const EdgeInsets.all(16.0),
                child: Row(
                  children: [
                    IconButton(
                      icon: const Icon(Icons.arrow_back),
                      onPressed: () => context.go('/'),
                    ),
                    const Spacer(),
                    IconButton(
                      icon: const Icon(Icons.volume_up),
                      onPressed: () {
                        if (_currentPrompt != null) {
                          _voiceService.speak(_currentPrompt!['prompt'] ?? '');
                        }
                      },
                    ),
                    IconButton(
                      icon: const Icon(Icons.settings_outlined),
                      onPressed: () => context.go('/settings'),
                    ),
                  ],
                ),
              ),
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 20.0),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(
                          Icons.auto_stories,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                          size: 28,
                        ),
                        const SizedBox(width: 12),
                        Text(
                          'Story Keeper',
                          style: Theme.of(context).textTheme.headlineMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(
                      'Your private journal - stays on your phone',
                      style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                  ],
                ),
              ),
              const SizedBox(height: 20),
              
              // Prompt Card
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 16),
                child: Card(
                  child: Padding(
                    padding: const EdgeInsets.all(16),
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Row(
                          children: [
                            Icon(
                              Icons.lightbulb_outline,
                              color: Colors.amber[600],
                              size: 20,
                            ),
                            const SizedBox(width: 8),
                            Text(
                              'Today\'s Prompt',
                              style: Theme.of(context).textTheme.titleSmall?.copyWith(
                                color: Colors.amber[600],
                                fontWeight: FontWeight.bold,
                              ),
                            ),
                            const Spacer(),
                            IconButton(
                              icon: const Icon(Icons.refresh, size: 20),
                              onPressed: _loadPrompt,
                              tooltip: 'New prompt',
                            ),
                          ],
                        ),
                        const SizedBox(height: 8),
                        if (_isLoadingPrompt)
                          const Center(child: CircularProgressIndicator())
                        else
                          Text(
                            _currentPrompt?['prompt'] ?? 'What made you smile today?',
                            style: Theme.of(context).textTheme.titleMedium,
                          ),
                        const SizedBox(height: 16),
                        TextField(
                          controller: _entryController,
                          decoration: InputDecoration(
                            hintText: 'Share your thoughts...',
                            border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
                            suffixIcon: MicButton(
                              onResult: (text) {
                                _entryController.text = text;
                              },
                            ),
                          ),
                          maxLines: 3,
                        ),
                        const SizedBox(height: 12),
                        SizedBox(
                          width: double.infinity,
                          child: ElevatedButton.icon(
                            onPressed: _addEntry,
                            icon: const Icon(Icons.save),
                            label: const Text('Save Entry'),
                          ),
                        ),
                      ],
                    ),
                  ),
                ),
              ),
              const SizedBox(height: 16),
              
              Expanded(
                child: _entries.isEmpty
                  ? _buildEmptyState(isDark)
                  : Stack(
                      children: [
                        _buildEntriesList(isDark),
                        ScrollIndicator(scrollController: _scrollController, bottomPadding: 20),
                      ],
                    ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildEmptyState(bool isDark) {
    return Center(
      child: Padding(
        padding: const EdgeInsets.all(40),
        child: Column(
          mainAxisAlignment: MainAxisAlignment.center,
          children: [
            Icon(
              Icons.book_outlined,
              size: 60,
              color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.5),
            ),
            const SizedBox(height: 16),
            Text(
              'Your story starts here',
              style: Theme.of(context).textTheme.titleMedium,
            ),
            const SizedBox(height: 8),
            Text(
              'Write your first entry above',
              textAlign: TextAlign.center,
              style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
              ),
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildEntriesList(bool isDark) {
    return ListView.builder(
      controller: _scrollController,
      padding: const EdgeInsets.symmetric(horizontal: 16),
      itemCount: _entries.length,
      itemBuilder: (context, index) {
        final entry = _entries[index];
        
        return Dismissible(
          key: Key(entry['id']),
          direction: DismissDirection.endToStart,
          background: Container(
            alignment: Alignment.centerRight,
            padding: const EdgeInsets.only(right: 20),
            color: Colors.red,
            child: const Icon(Icons.delete, color: Colors.white),
          ),
          onDismissed: (_) => _deleteEntry(entry['id']),
          child: Card(
            margin: const EdgeInsets.only(bottom: 12),
            child: InkWell(
              onTap: () {
                _voiceService.speak(entry['content'] ?? '');
              },
              borderRadius: BorderRadius.circular(12),
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(
                          Icons.calendar_today,
                          size: 14,
                          color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                        ),
                        const SizedBox(width: 6),
                        Text(
                          _formatDate(entry['date']),
                          style: Theme.of(context).textTheme.bodySmall?.copyWith(
                            color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                          ),
                        ),
                        const Spacer(),
                        IconButton(
                          icon: const Icon(Icons.volume_up, size: 18),
                          onPressed: () => _voiceService.speak(entry['content'] ?? ''),
                          padding: EdgeInsets.zero,
                          constraints: const BoxConstraints(),
                        ),
                      ],
                    ),
                    if (entry['prompt']?.isNotEmpty ?? false) ...[
                      const SizedBox(height: 8),
                      Text(
                        entry['prompt'],
                        style: Theme.of(context).textTheme.bodySmall?.copyWith(
                          fontStyle: FontStyle.italic,
                          color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                        ),
                      ),
                    ],
                    const SizedBox(height: 8),
                    Text(
                      entry['content'] ?? '',
                      style: Theme.of(context).textTheme.bodyMedium,
                    ),
                  ],
                ),
              ),
            ),
          ),
        );
      },
    );
  }
}
