import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/widgets/shared_widgets.dart';

class JobsPage extends StatefulWidget {
  const JobsPage({super.key});

  @override
  State<JobsPage> createState() => _JobsPageState();
}

class _JobsPageState extends State<JobsPage> with SingleTickerProviderStateMixin {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _api = ApiClient();
  
  late TabController _tabController;
  final _skillsController = TextEditingController();
  final _locationController = TextEditingController();
  final _preferencesController = TextEditingController();
  final _artsSkillsController = TextEditingController();
  final _artsExperienceController = TextEditingController();
  
  bool _hasPlayedIntro = false;
  bool _isLoading = false;
  bool _isArtsSearch = false;
  Map<String, dynamic>? _jobResults;
  Map<String, dynamic>? _artsResults;
  String? _error;

  static const List<Map<String, String>> artsOrganizations = [
    {'name': 'Australia Council for the Arts', 'desc': 'National arts funding - First Nations grants', 'phone': '02 9215 9000', 'website': 'australiacouncil.gov.au', 'isMob': 'false'},
    {'name': 'Desart', 'desc': 'Central Australian Aboriginal art centres', 'phone': '08 8953 4736', 'website': 'desart.com.au', 'isMob': 'true'},
    {'name': 'ANKAAA', 'desc': 'Association of Northern, Kimberley and Arnhem Aboriginal Artists', 'phone': '08 8981 6134', 'website': 'ankaaa.org.au', 'isMob': 'true'},
    {'name': 'Koorie Heritage Trust', 'desc': 'Victorian Aboriginal arts & culture', 'phone': '03 8662 6300', 'website': 'koorieheritagetrust.com.au', 'isMob': 'true'},
    {'name': 'BlakDance', 'desc': 'National First Nations dance organization', 'phone': '', 'website': 'blakdance.org.au', 'isMob': 'true'},
    {'name': 'First Nations Media Australia', 'desc': 'Aboriginal & Torres Strait Islander media network', 'phone': '08 8952 6465', 'website': 'firstnationsmedia.org.au', 'isMob': 'true'},
    {'name': 'Music Australia', 'desc': 'First Nations music industry support', 'phone': '', 'website': 'musicaustralia.org.au', 'isMob': 'false'},
    {'name': 'Create NSW', 'desc': 'NSW arts funding - Aboriginal Arts Development', 'phone': '02 8289 6500', 'website': 'create.nsw.gov.au', 'isMob': 'false'},
    {'name': 'Creative Victoria', 'desc': 'Vic arts funding - First Peoples programs', 'phone': '03 8683 3100', 'website': 'creative.vic.gov.au', 'isMob': 'false'},
  ];

  @override
  void initState() {
    super.initState();
    _tabController = TabController(length: 2, vsync: this);
    _tabController.addListener(_onTabChanged);
    _initializeAndPlay();
  }

  void _onTabChanged() {
    if (!_tabController.indexIsChanging) {
      setState(() {
        _isArtsSearch = _tabController.index == 1;
        _jobResults = null;
      });
      if (_prefs.autoPlayVoice) {
        _voiceService.stop();
        if (_isArtsSearch) {
          _voiceService.speak(
            "Arts and culture! Your painting, dance, music, weaving, storytelling - it's all valuable. "
            "Tell me about your cultural practice and I'll help you find opportunities."
          );
        }
      }
    }
  }

  Future<void> _initializeAndPlay() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    if (mounted && !_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      await Future.delayed(const Duration(milliseconds: 300));
      _voiceService.speak(
        "Hey cuz, let's look for work that suits you. "
        "Tell me what you're good at - cooking, driving, building, caring for people, whatever your skills are. "
        "Pop in where you live and tap Search for Jobs. "
        "I'll find jobs and also tell you about Indigenous employment programs like IESC and IEP that can support you."
      );
    }
  }

  @override
  void dispose() {
    _tabController.removeListener(_onTabChanged);
    _tabController.dispose();
    _voiceService.stop();
    _skillsController.dispose();
    _locationController.dispose();
    _preferencesController.dispose();
    _artsSkillsController.dispose();
    _artsExperienceController.dispose();
    super.dispose();
  }

  Future<void> _searchJobs({bool forArts = false}) async {
    setState(() {
      _isLoading = true;
      _error = null;
    });
    
    try {
      final skills = forArts ? _artsSkillsController.text : _skillsController.text;
      final experience = forArts ? _artsExperienceController.text : _preferencesController.text;
      
      final result = await _api.post('/api/jobs/search', {
        'skills': skills,
        'query': skills,
        'location': forArts ? '' : _locationController.text,
        'preferences': experience,
        'experience': experience,
        'isArts': forArts,
      });
      
      setState(() {
        if (forArts) {
          _artsResults = result;
        } else {
          _jobResults = result;
        }
        _isLoading = false;
      });
      
      if (_prefs.autoPlayVoice) {
        final advice = result['advice'] as String? ?? '';
        if (advice.isNotEmpty) {
          final shortAdvice = advice.length > 200 ? '${advice.substring(0, 200)}...' : advice;
          _voiceService.speak(
            forArts 
              ? "I've found some arts opportunities for you. $shortAdvice"
              : "Based on your skills, here's some job advice. $shortAdvice"
          );
        }
      }
    } catch (e) {
      setState(() {
        _error = 'Having trouble searching. Try again?';
        _isLoading = false;
      });
    }
  }

  void _reset() {
    setState(() {
      if (_isArtsSearch) {
        _artsResults = null;
      } else {
        _jobResults = null;
      }
    });
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              Padding(
                padding: const EdgeInsets.all(16.0),
                child: Row(
                  children: [
                    IconButton(
                      icon: const Icon(Icons.arrow_back),
                      onPressed: () {
                        if (_jobResults != null) {
                          _reset();
                        } else {
                          context.go('/');
                        }
                      },
                    ),
                    const Spacer(),
                    IconButton(
                      icon: const Icon(Icons.volume_up),
                      onPressed: () {
                        _voiceService.speak(
                          _isArtsSearch 
                            ? "Tell me about your art, dance, music, or cultural practice. I'll find opportunities for you."
                            : "Tell me what you're good at and where you live, and I'll help find jobs and programs."
                        );
                      },
                    ),
                    IconButton(
                      icon: const Icon(Icons.settings_outlined),
                      onPressed: () => context.go('/settings'),
                    ),
                  ],
                ),
              ),
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 20.0),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(
                          _isArtsSearch ? Icons.palette : Icons.work_outline,
                          color: _isArtsSearch 
                            ? Colors.amber[700] 
                            : (isDark ? AppColors.primaryLight : AppColors.primary),
                          size: 28,
                        ),
                        const SizedBox(width: 12),
                        Text(
                          'Job Spotter',
                          style: Theme.of(context).textTheme.headlineMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(
                      _isArtsSearch ? 'Find arts & culture opportunities' : 'Find work that suits you',
                      style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                  ],
                ),
              ),
              const SizedBox(height: 16),
              TabBar(
                controller: _tabController,
                tabs: const [
                  Tab(icon: Icon(Icons.work_outline), text: 'All Jobs'),
                  Tab(icon: Icon(Icons.palette), text: 'Arts & Culture'),
                ],
                labelColor: isDark ? AppColors.primaryLight : AppColors.primary,
                indicatorColor: isDark ? AppColors.primaryLight : AppColors.primary,
              ),
              const SizedBox(height: 8),
              Expanded(
                child: TabBarView(
                  controller: _tabController,
                  children: [
                    _jobResults == null
                      ? _buildInputSection(isDark)
                      : _buildResultsSection(isDark),
                    _artsResults == null
                      ? _buildArtsInputSection(isDark)
                      : _buildArtsResultsSection(isDark),
                  ],
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildInputSection(bool isDark) {
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(
            'What are you good at?',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          TextField(
            controller: _skillsController,
            decoration: InputDecoration(
              hintText: 'e.g., Driving, customer service, cooking, cleaning',
              border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
              prefixIcon: const Icon(Icons.star_outline),
              suffixIcon: MicButton(onResult: (t) => _skillsController.text = t),
            ),
            maxLines: 2,
          ),
          const SizedBox(height: 16),
          
          Text(
            'Where are you located?',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          TextField(
            controller: _locationController,
            decoration: InputDecoration(
              hintText: 'e.g., Sydney, Darwin, Alice Springs',
              border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
              prefixIcon: const Icon(Icons.location_on_outlined),
              suffixIcon: MicButton(onResult: (t) => _locationController.text = t),
            ),
          ),
          const SizedBox(height: 16),
          
          Text(
            'What kind of work are you looking for?',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          TextField(
            controller: _preferencesController,
            decoration: InputDecoration(
              hintText: 'e.g., Part-time, full-time, flexible hours',
              border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
              prefixIcon: const Icon(Icons.search),
              suffixIcon: MicButton(onResult: (t) => _preferencesController.text = t),
            ),
            maxLines: 2,
          ),
          const SizedBox(height: 24),
          
          if (_error != null)
            Padding(
              padding: const EdgeInsets.only(bottom: 16),
              child: Text(_error!, style: TextStyle(color: Colors.red[400])),
            ),
          
          SizedBox(
            width: double.infinity,
            child: ElevatedButton.icon(
              onPressed: _isLoading ? null : _searchJobs,
              icon: _isLoading 
                ? const SizedBox(
                    width: 20, height: 20,
                    child: CircularProgressIndicator(strokeWidth: 2),
                  )
                : const Icon(Icons.search),
              label: Text(_isLoading ? 'Searching...' : 'Find Jobs'),
              style: ElevatedButton.styleFrom(
                padding: const EdgeInsets.symmetric(vertical: 16),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
            ),
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }

  Widget _buildResultsSection(bool isDark) {
    final results = _jobResults!;
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          if (results['suggestedRoles'] != null) ...[
            _buildResultCard(
              'Jobs That Might Suit You',
              Icons.work,
              results['suggestedRoles'] as List,
              isDark,
            ),
          ],
          
          if (results['indigenousPrograms'] != null) ...[
            _buildResultCard(
              'Indigenous Employment Programs',
              Icons.group,
              results['indigenousPrograms'] as List,
              isDark,
              color: Colors.green,
            ),
          ],
          
          if (results['whereToLook'] != null) ...[
            _buildResultCard(
              'Where to Look',
              Icons.search,
              results['whereToLook'] as List,
              isDark,
              color: Colors.blue,
            ),
          ],
          
          if (results['skillsToHighlight'] != null) ...[
            _buildResultCard(
              'Skills to Highlight',
              Icons.star,
              results['skillsToHighlight'] as List,
              isDark,
              color: Colors.amber,
            ),
          ],
          
          if (results['trainingOptions'] != null) ...[
            _buildResultCard(
              'Training Options',
              Icons.school,
              results['trainingOptions'] as List,
              isDark,
              color: Colors.purple,
            ),
          ],
          
          if (results['applicationTips'] != null) ...[
            Card(
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.lightbulb_outline, color: Colors.orange[400]),
                        const SizedBox(width: 8),
                        Text(
                          'Application Tips',
                          style: Theme.of(context).textTheme.titleMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                        const Spacer(),
                        IconButton(
                          icon: const Icon(Icons.volume_up, size: 20),
                          onPressed: () {
                            final tips = results['applicationTips'] as List;
                            _voiceService.speak(tips.join('. '));
                          },
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    ...((results['applicationTips'] as List).map((tip) => Padding(
                      padding: const EdgeInsets.only(bottom: 8),
                      child: Row(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Icon(Icons.check, color: Colors.green[400], size: 18),
                          const SizedBox(width: 8),
                          Expanded(child: Text(tip.toString())),
                        ],
                      ),
                    ))),
                  ],
                ),
              ),
            ),
            const SizedBox(height: 16),
          ],
          
          Row(
            children: [
              Expanded(
                child: OutlinedButton.icon(
                  onPressed: _reset,
                  icon: const Icon(Icons.refresh),
                  label: const Text('New Search'),
                  style: OutlinedButton.styleFrom(
                    padding: const EdgeInsets.symmetric(vertical: 16),
                    shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
                  ),
                ),
              ),
              const SizedBox(width: 12),
              Expanded(
                child: ElevatedButton.icon(
                  onPressed: () => context.go('/skills'),
                  icon: const Icon(Icons.description),
                  label: const Text('Build Resume'),
                  style: ElevatedButton.styleFrom(
                    padding: const EdgeInsets.symmetric(vertical: 16),
                    shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
                  ),
                ),
              ),
            ],
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }

  Widget _buildResultCard(String title, IconData icon, List items, bool isDark, {Color? color}) {
    if (items.isEmpty) return const SizedBox();
    
    final iconColor = color ?? (isDark ? AppColors.primaryLight : AppColors.primary);
    
    return Card(
      margin: const EdgeInsets.only(bottom: 16),
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                Icon(icon, color: iconColor),
                const SizedBox(width: 8),
                Expanded(
                  child: Text(
                    title,
                    style: Theme.of(context).textTheme.titleMedium?.copyWith(
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                ),
              ],
            ),
            const SizedBox(height: 12),
            ...items.map((item) => Padding(
              padding: const EdgeInsets.only(bottom: 8),
              child: Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Icon(Icons.arrow_right, color: iconColor, size: 20),
                  const SizedBox(width: 4),
                  Expanded(child: Text(item.toString())),
                ],
              ),
            )),
          ],
        ),
      ),
    );
  }

  Widget _buildArtsInputSection(bool isDark) {
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Card(
            color: Colors.amber[50],
            child: Padding(
              padding: const EdgeInsets.all(12),
              child: Row(
                children: [
                  Icon(Icons.palette, color: Colors.amber[800]),
                  const SizedBox(width: 12),
                  Expanded(
                    child: Text(
                      'Aboriginal art, dance, music, and storytelling are valued worldwide. Let\'s find opportunities for your cultural practice.',
                      style: TextStyle(color: Colors.amber[900], fontSize: 13),
                    ),
                  ),
                ],
              ),
            ),
          ),
          const SizedBox(height: 16),
          
          Text(
            'What\'s your art or cultural practice?',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          TextField(
            controller: _artsSkillsController,
            decoration: InputDecoration(
              hintText: 'e.g., painting, dance, music, weaving, storytelling',
              border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
              prefixIcon: const Icon(Icons.palette),
              suffixIcon: MicButton(onResult: (t) => _artsSkillsController.text = t),
            ),
            maxLines: 2,
          ),
          const SizedBox(height: 8),
          Wrap(
            spacing: 8,
            runSpacing: 8,
            children: ['painting', 'dance', 'music', 'weaving', 'storytelling'].map((skill) {
              return ActionChip(
                label: Text(skill),
                onPressed: () => setState(() => _artsSkillsController.text = skill),
              );
            }).toList(),
          ),
          const SizedBox(height: 16),
          
          Text(
            'Tell us about your experience',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          TextField(
            controller: _artsExperienceController,
            decoration: InputDecoration(
              hintText: 'e.g., learned from family, sell at markets, teach workshops',
              border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
              prefixIcon: const Icon(Icons.history_edu),
              suffixIcon: MicButton(onResult: (t) => _artsExperienceController.text = t),
            ),
            maxLines: 3,
          ),
          const SizedBox(height: 24),
          
          if (_error != null)
            Padding(
              padding: const EdgeInsets.only(bottom: 16),
              child: Text(_error!, style: TextStyle(color: Colors.red[400])),
            ),
          
          SizedBox(
            width: double.infinity,
            child: ElevatedButton.icon(
              onPressed: _isLoading ? null : () => _searchJobs(forArts: true),
              icon: _isLoading 
                ? const SizedBox(
                    width: 20, height: 20,
                    child: CircularProgressIndicator(strokeWidth: 2),
                  )
                : const Icon(Icons.search),
              label: Text(_isLoading ? 'Finding opportunities...' : 'Find Arts Opportunities'),
              style: ElevatedButton.styleFrom(
                backgroundColor: Colors.amber[700],
                padding: const EdgeInsets.symmetric(vertical: 16),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
            ),
          ),
          const SizedBox(height: 24),
          
          Text(
            'Arts Organizations',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 12),
          ...artsOrganizations.map((org) {
            final isMob = org['isMob'] == 'true';
            return Card(
              margin: const EdgeInsets.only(bottom: 8),
              color: isMob ? Colors.orange.withOpacity(0.05) : null,
              shape: RoundedRectangleBorder(
                borderRadius: BorderRadius.circular(12),
                side: isMob ? BorderSide(color: Colors.orange.withOpacity(0.3)) : BorderSide.none,
              ),
              child: Padding(
                padding: const EdgeInsets.all(12),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Expanded(
                          child: Column(
                            crossAxisAlignment: CrossAxisAlignment.start,
                            children: [
                              Row(
                                children: [
                                  Flexible(child: Text(org['name']!, style: const TextStyle(fontWeight: FontWeight.w600))),
                                  if (isMob) ...[
                                    const SizedBox(width: 8),
                                    Container(
                                      padding: const EdgeInsets.symmetric(horizontal: 6, vertical: 2),
                                      decoration: BoxDecoration(
                                        color: Colors.orange.withOpacity(0.2),
                                        borderRadius: BorderRadius.circular(4),
                                      ),
                                      child: Text('Mob', style: TextStyle(fontSize: 10, color: Colors.orange[800], fontWeight: FontWeight.w500)),
                                    ),
                                  ],
                                ],
                              ),
                              const SizedBox(height: 4),
                              Text(org['desc']!, style: TextStyle(fontSize: 12, color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary)),
                            ],
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Row(
                      children: [
                        if (org['phone']!.isNotEmpty)
                          OutlinedButton.icon(
                            onPressed: () => launchPhone(org['phone']!.replaceAll(' ', '')),
                            icon: const Icon(Icons.phone, size: 16),
                            label: const Text('Call'),
                            style: OutlinedButton.styleFrom(
                              padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 4),
                              minimumSize: Size.zero,
                            ),
                          ),
                        if (org['phone']!.isNotEmpty) const SizedBox(width: 8),
                        if (org['website']!.isNotEmpty)
                          ElevatedButton.icon(
                            onPressed: () => launchWebsite(org['website']!),
                            icon: const Icon(Icons.language, size: 16),
                            label: const Text('Visit'),
                            style: ElevatedButton.styleFrom(
                              padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 4),
                              minimumSize: Size.zero,
                            ),
                          ),
                      ],
                    ),
                  ],
                ),
              ),
            );
          }),
          const SizedBox(height: 16),
          
          Card(
            color: isDark ? Colors.grey[800] : Colors.grey[100],
            child: Padding(
              padding: const EdgeInsets.all(12),
              child: Row(
                children: [
                  Icon(Icons.info_outline, color: isDark ? Colors.grey[400] : Colors.grey[600]),
                  const SizedBox(width: 12),
                  Expanded(
                    child: Text(
                      'Only share what you have the right to share. Your cultural knowledge is precious.',
                      style: TextStyle(color: isDark ? Colors.grey[300] : Colors.grey[700], fontSize: 13, fontStyle: FontStyle.italic),
                    ),
                  ),
                ],
              ),
            ),
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }

  Widget _buildArtsResultsSection(bool isDark) {
    if (_artsResults == null) return _buildArtsInputSection(isDark);
    
    final results = _artsResults!;
    final advice = results['advice'] as String? ?? '';
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          if (advice.isNotEmpty) ...[
            Card(
              color: Colors.amber[50],
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.palette, color: Colors.amber[800]),
                        const SizedBox(width: 8),
                        Text(
                          'Arts Opportunities',
                          style: Theme.of(context).textTheme.titleMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                            color: Colors.amber[900],
                          ),
                        ),
                        const Spacer(),
                        IconButton(
                          icon: Icon(Icons.volume_up, color: Colors.amber[800]),
                          onPressed: () => _voiceService.speak(advice),
                        ),
                      ],
                    ),
                    const SizedBox(height: 12),
                    Text(advice, style: TextStyle(color: Colors.amber[900])),
                  ],
                ),
              ),
            ),
            const SizedBox(height: 16),
          ],
          
          if (results['indigenousPrograms'] != null) ...[
            _buildResultCard(
              'Arts Programs & Funding',
              Icons.palette,
              results['indigenousPrograms'] as List,
              isDark,
              color: Colors.amber,
            ),
          ],
          
          if (results['whereToLook'] != null) ...[
            _buildResultCard(
              'Where to Connect',
              Icons.language,
              results['whereToLook'] as List,
              isDark,
              color: Colors.blue,
            ),
          ],
          
          Row(
            children: [
              Expanded(
                child: OutlinedButton.icon(
                  onPressed: _reset,
                  icon: const Icon(Icons.refresh),
                  label: const Text('New Search'),
                  style: OutlinedButton.styleFrom(
                    padding: const EdgeInsets.symmetric(vertical: 16),
                    shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
                  ),
                ),
              ),
              const SizedBox(width: 12),
              Expanded(
                child: ElevatedButton.icon(
                  onPressed: () => context.go('/skills'),
                  icon: const Icon(Icons.attach_money),
                  label: const Text('Price My Work'),
                  style: ElevatedButton.styleFrom(
                    backgroundColor: Colors.amber[700],
                    padding: const EdgeInsets.symmetric(vertical: 16),
                    shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
                  ),
                ),
              ),
            ],
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }

  void launchPhone(String phone) {
    // URL launcher would be used here in production
    debugPrint('Calling: $phone');
  }

  void launchWebsite(String website) {
    // URL launcher would be used here in production
    debugPrint('Opening: https://$website');
  }
}
