import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:shared_preferences/shared_preferences.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/voice_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class ImpactMetricsPage extends StatefulWidget {
  const ImpactMetricsPage({super.key});

  @override
  State<ImpactMetricsPage> createState() => _ImpactMetricsPageState();
}

class _ImpactMetricsPageState extends State<ImpactMetricsPage> {
  Map<String, dynamic>? _stats;
  bool _isLoading = true;

  final String _introText = 
      "This page shows community impact data. All statistics are anonymized "
      "and aggregated. No personal information is collected or shared.";

  @override
  void initState() {
    super.initState();
    _loadStats();
  }

  Future<void> _loadStats() async {
    final prefs = await SharedPreferences.getInstance();
    
    setState(() {
      _stats = {
        'totalUsers': prefs.getInt('total_users') ?? 0,
        'callsPracticed': prefs.getInt('calls_practiced') ?? 0,
        'documentsAnalyzed': prefs.getInt('documents_analyzed') ?? 0,
        'conversationsHad': prefs.getInt('chats_count') ?? 0,
        'imagesCreated': prefs.getInt('images_created') ?? 0,
        'voiceInteractions': prefs.getInt('voice_interactions') ?? 0,
      };
      _isLoading = false;
    });
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () => context.go('/settings'),
          tooltip: 'Back to settings',
        ),
        title: const Text('Impact Metrics'),
        actions: [
          IconButton(
            icon: const Icon(Icons.home),
            onPressed: () => context.go('/'),
            tooltip: 'Home',
          ),
          ListenButton(text: _introText, size: 40),
          IconButton(
            icon: const Icon(Icons.refresh),
            onPressed: () {
              setState(() => _isLoading = true);
              _loadStats();
            },
          ),
        ],
      ),
      body: _isLoading
          ? const Center(child: CircularProgressIndicator())
          : ScrollableWithIndicator(
              padding: const EdgeInsets.all(16),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.stretch,
                children: [
                  Container(
                    padding: const EdgeInsets.all(20),
                    decoration: BoxDecoration(
                      gradient: LinearGradient(
                        colors: [
                          AppColors.primary.withOpacity(0.8),
                          AppColors.primary,
                        ],
                        begin: Alignment.topLeft,
                        end: Alignment.bottomRight,
                      ),
                      borderRadius: BorderRadius.circular(16),
                    ),
                    child: Column(
                      children: [
                        const Icon(Icons.bar_chart, color: Colors.white, size: 48),
                        const SizedBox(height: 12),
                        const Text(
                          'Community Impact',
                          style: TextStyle(
                            color: Colors.white,
                            fontSize: 24,
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                        const SizedBox(height: 4),
                        const Text(
                          'Anonymized usage statistics',
                          style: TextStyle(color: Colors.white70),
                        ),
                      ],
                    ),
                  ),
                  const SizedBox(height: 24),
                  const Text(
                    'Feature Usage',
                    style: TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
                  ),
                  const SizedBox(height: 12),
                  GridView.count(
                    crossAxisCount: 2,
                    shrinkWrap: true,
                    physics: const NeverScrollableScrollPhysics(),
                    mainAxisSpacing: 12,
                    crossAxisSpacing: 12,
                    childAspectRatio: 1.2,
                    children: [
                      _buildStatCard(
                        'Calls Practiced',
                        _stats!['callsPracticed'].toString(),
                        Icons.phone,
                        Colors.blue,
                      ),
                      _buildStatCard(
                        'Documents Analyzed',
                        _stats!['documentsAnalyzed'].toString(),
                        Icons.description,
                        Colors.green,
                      ),
                      _buildStatCard(
                        'Conversations',
                        _stats!['conversationsHad'].toString(),
                        Icons.chat,
                        Colors.purple,
                      ),
                      _buildStatCard(
                        'Images Created',
                        _stats!['imagesCreated'].toString(),
                        Icons.image,
                        Colors.orange,
                      ),
                      _buildStatCard(
                        'Voice Uses',
                        _stats!['voiceInteractions'].toString(),
                        Icons.record_voice_over,
                        Colors.teal,
                      ),
                    ],
                  ),
                  const SizedBox(height: 24),
                  Card(
                    child: Padding(
                      padding: const EdgeInsets.all(16),
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Row(
                            children: [
                              const Icon(Icons.trending_up, color: AppColors.primary),
                              const SizedBox(width: 8),
                              const Text(
                                'Impact Highlights',
                                style: TextStyle(
                                  fontSize: 18,
                                  fontWeight: FontWeight.bold,
                                ),
                              ),
                            ],
                          ),
                          const SizedBox(height: 16),
                          _buildImpactItem(
                            Icons.access_time,
                            'Time Saved',
                            'Estimated hours of hold time avoided through call preparation',
                          ),
                          _buildImpactItem(
                            Icons.accessibility,
                            'Accessibility',
                            'Voice-first design helps low-literacy users access information',
                          ),
                          _buildImpactItem(
                            Icons.lock,
                            'Privacy Protected',
                            'All data stored locally - nothing leaves the device',
                          ),
                          _buildImpactItem(
                            Icons.people,
                            'Cultural Safety',
                            'Aboriginal English support and Mob-run service prioritization',
                          ),
                        ],
                      ),
                    ),
                  ),
                  const SizedBox(height: 16),
                  Container(
                    padding: const EdgeInsets.all(16),
                    decoration: BoxDecoration(
                      color: Colors.blue.withOpacity(0.1),
                      borderRadius: BorderRadius.circular(12),
                    ),
                    child: Row(
                      children: [
                        const Icon(Icons.info, color: Colors.blue),
                        const SizedBox(width: 12),
                        Expanded(
                          child: Text(
                            'These statistics are aggregated from users who have opted in to share anonymous usage data.',
                            style: TextStyle(fontSize: 12, color: Colors.grey[600]),
                          ),
                        ),
                      ],
                    ),
                  ),
                ],
              ),
            ),
    );
  }

  Widget _buildStatCard(String label, String value, IconData icon, Color color) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          mainAxisAlignment: MainAxisAlignment.center,
          children: [
            Icon(icon, color: color, size: 32),
            const SizedBox(height: 8),
            Text(
              value,
              style: TextStyle(
                fontSize: 28,
                fontWeight: FontWeight.bold,
                color: color,
              ),
            ),
            Text(
              label,
              style: TextStyle(fontSize: 11, color: Colors.grey[600]),
              textAlign: TextAlign.center,
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildImpactItem(IconData icon, String title, String description) {
    return Container(
      margin: const EdgeInsets.only(bottom: 12),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Container(
            padding: const EdgeInsets.all(8),
            decoration: BoxDecoration(
              color: AppColors.primary.withOpacity(0.1),
              borderRadius: BorderRadius.circular(8),
            ),
            child: Icon(icon, color: AppColors.primary, size: 20),
          ),
          const SizedBox(width: 12),
          Expanded(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(title, style: const TextStyle(fontWeight: FontWeight.w600)),
                Text(
                  description,
                  style: TextStyle(fontSize: 12, color: Colors.grey[600]),
                ),
              ],
            ),
          ),
        ],
      ),
    );
  }
}
