import 'dart:convert';
import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/widgets/shared_widgets.dart';

class ImagesPage extends StatefulWidget {
  const ImagesPage({super.key});

  @override
  State<ImagesPage> createState() => _ImagesPageState();
}

class _ImagesPageState extends State<ImagesPage> {
  final TextEditingController _promptController = TextEditingController();
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _apiClient = ApiClient();
  
  bool _isLoading = false;
  bool _isListening = false;
  bool _hasPlayedIntro = false;
  String? _generatedImageUrl;
  String? _generatedImageBase64;
  String? _icipWarning;

  final List<String> _suggestions = [
    'A sunset over the ocean',
    'Kangaroos in the outback',
    'A beautiful garden',
    'The stars at night',
    'A cozy campfire',
  ];

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    debugPrint('ImagesPage: Initializing voice service...');
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playIntro();
    }
  }

  @override
  void dispose() {
    _promptController.dispose();
    _voiceService.stop();
    super.dispose();
  }

  void _playIntro() {
    debugPrint('ImagesPage: _playIntro called, autoPlay=${_prefs.autoPlayVoice}');
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      _voiceService.speak(
        "Let's make a picture! "
        "Just describe what you want to see, like a sunset over the outback with kangaroos. "
        "Tap the mic to speak, or type it in. Then hit Make Picture."
      );
    }
  }

  Future<void> _generateImage() async {
    if (_promptController.text.trim().isEmpty) {
      if (_prefs.autoPlayVoice) {
        _voiceService.speak("Tell me what picture you'd like first. Tap the mic to speak, or type it in.");
      }
      return;
    }

    _voiceService.stop();
    setState(() {
      _isLoading = true;
      _icipWarning = null;
    });

    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Creating your picture. This might take a moment...");
    }

    try {
      final data = await _apiClient.post('/api/generate-image', {
        'prompt': _promptController.text,
      });
      
      setState(() {
        _generatedImageUrl = data['url'];
        _generatedImageBase64 = data['b64_json'];
        _isLoading = false;
        
        if (data['icipWarning'] != null && data['icipWarning']['detected'] == true) {
          _icipWarning = data['icipWarning']['message'];
        }
      });

      if (_icipWarning != null && _prefs.autoPlayVoice) {
        await _voiceService.speak("A note about your request: $_icipWarning");
        await Future.delayed(const Duration(seconds: 2));
      }

      if (_prefs.autoPlayVoice) {
        await _voiceService.speak("Your picture is ready! Have a look. If you want to make another one, just describe something new.");
      }
    } catch (e) {
      setState(() => _isLoading = false);
      if (_prefs.autoPlayVoice) {
        _voiceService.speak("Sorry, I couldn't make that picture. Want to try again with a different description?");
      }
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(content: Text('Couldn\'t make that picture. Try again?')),
        );
      }
    }
  }

  Future<void> _toggleListening() async {
    _voiceService.stop();
    
    if (_isListening) {
      await _voiceService.stopListening();
      setState(() => _isListening = false);
    } else {
      setState(() => _isListening = true);
      
      await _voiceService.speakThenListen(
        prompt: "I'm listening. Tell me what picture you'd like to make.",
        onResult: (text) {
          setState(() {
            _isListening = false;
            _promptController.text = text;
          });
          if (text.isNotEmpty && _prefs.autoPlayVoice) {
            _voiceService.speak("Got it. Tap Make Picture when you're ready.");
          }
        },
        onError: (error) {
          setState(() => _isListening = false);
          if (_prefs.autoPlayVoice) {
            _voiceService.speak("I couldn't hear that clearly. Tap the mic to try again.");
          }
        },
      );
    }
  }

  void _useSuggestion(String suggestion) {
    _promptController.text = suggestion;
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("$suggestion. Good choice! Tap Make Picture when you're ready.");
    }
  }

  void _makeAnother() {
    setState(() {
      _promptController.clear();
      _generatedImageUrl = null;
      _generatedImageBase64 = null;
      _icipWarning = null;
    });
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Ready for a new picture. What would you like to see?");
    }
  }

  bool get _hasImage => _generatedImageUrl != null || _generatedImageBase64 != null;

  Widget _buildGeneratedImage() {
    if (_generatedImageBase64 != null) {
      try {
        return Image.memory(
          base64Decode(_generatedImageBase64!),
          fit: BoxFit.cover,
          width: double.infinity,
          errorBuilder: (context, error, stackTrace) => _buildImageError(),
        );
      } catch (e) {
        return _buildImageError();
      }
    } else if (_generatedImageUrl != null) {
      return Image.network(
        _generatedImageUrl!,
        fit: BoxFit.cover,
        width: double.infinity,
        loadingBuilder: (context, child, loadingProgress) {
          if (loadingProgress == null) return child;
          return Container(
            height: 300,
            color: Theme.of(context).cardColor,
            child: const Center(child: CircularProgressIndicator()),
          );
        },
        errorBuilder: (context, error, stackTrace) => _buildImageError(),
      );
    }
    return const SizedBox.shrink();
  }

  Widget _buildImageError() {
    return Container(
      height: 300,
      color: Theme.of(context).cardColor,
      child: Column(
        mainAxisAlignment: MainAxisAlignment.center,
        children: const [
          Icon(Icons.error, size: 48),
          SizedBox(height: 8),
          Text('Couldn\'t load the picture'),
        ],
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () {
            _voiceService.stop();
            context.go('/');
          },
        ),
        title: const Text('Make a Picture'),
        actions: [
          IconButton(
            icon: const Icon(Icons.volume_up),
            tooltip: 'Read page aloud',
            onPressed: () {
              _voiceService.speak(
                _hasImage
                    ? "Your picture is showing. Tap Make Another to create something new."
                    : "Describe what you want to see. Tap the mic to speak, or type it in the box."
              );
            },
          ),
          const SettingsButton(),
        ],
      ),
      body: CulturalBackground(
        child: ScrollableWithIndicator(
          padding: const EdgeInsets.all(16),
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Text(
                'Describe your picture',
                style: Theme.of(context).textTheme.headlineSmall,
              ),
              const SizedBox(height: 8),
              Text(
                'Tell me what you want to see and I\'ll create it for you.',
                style: Theme.of(context).textTheme.bodyMedium,
              ),
              const SizedBox(height: 16),
              Stack(
                children: [
                  TextField(
                    controller: _promptController,
                    maxLines: 4,
                    decoration: const InputDecoration(
                      hintText: 'A sunset over the outback with kangaroos...',
                    ),
                  ),
                  Positioned(
                    right: 8,
                    top: 8,
                    child: FloatingActionButton.small(
                      onPressed: _toggleListening,
                      backgroundColor: _isListening ? Colors.red : (isDark ? AppColors.primaryLight : AppColors.primary),
                      child: Icon(
                        _isListening ? Icons.stop : Icons.mic,
                        color: Colors.white,
                      ),
                    ),
                  ),
                ],
              ),
              if (_isListening)
                Padding(
                  padding: const EdgeInsets.only(top: 8),
                  child: Text(
                    'Listening... speak now',
                    style: TextStyle(
                      color: Colors.red,
                      fontStyle: FontStyle.italic,
                    ),
                  ),
                ),
              if (_promptController.text.isEmpty && !_hasImage) ...[
                const SizedBox(height: 16),
                Text(
                  'Or try one of these:',
                  style: Theme.of(context).textTheme.bodySmall,
                ),
                const SizedBox(height: 8),
                Wrap(
                  spacing: 8,
                  runSpacing: 8,
                  children: _suggestions.map((s) => ActionChip(
                    label: Text(s, style: const TextStyle(fontSize: 12)),
                    onPressed: () => _useSuggestion(s),
                  )).toList(),
                ),
              ],
              const SizedBox(height: 16),
              SizedBox(
                width: double.infinity,
                child: ElevatedButton(
                  onPressed: _isLoading ? null : _generateImage,
                  child: _isLoading
                      ? Row(
                          mainAxisAlignment: MainAxisAlignment.center,
                          children: const [
                            SizedBox(
                              height: 20,
                              width: 20,
                              child: CircularProgressIndicator(
                                strokeWidth: 2,
                                color: Colors.white,
                              ),
                            ),
                            SizedBox(width: 12),
                            Text('Creating...'),
                          ],
                        )
                      : const Text('Make Picture'),
                ),
              ),
              if (_icipWarning != null) ...[
                const SizedBox(height: 16),
                Card(
                  color: Colors.orange.withOpacity(0.1),
                  child: Padding(
                    padding: const EdgeInsets.all(12),
                    child: Row(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        const Icon(Icons.info_outline, color: Colors.orange, size: 20),
                        const SizedBox(width: 8),
                        Expanded(
                          child: Text(
                            _icipWarning!,
                            style: Theme.of(context).textTheme.bodySmall,
                          ),
                        ),
                      ],
                    ),
                  ),
                ),
              ],
              if (_hasImage) ...[
                const SizedBox(height: 24),
                ClipRRect(
                  borderRadius: BorderRadius.circular(12),
                  child: _buildGeneratedImage(),
                ),
                const SizedBox(height: 16),
                SizedBox(
                  width: double.infinity,
                  child: OutlinedButton.icon(
                    onPressed: _makeAnother,
                    icon: const Icon(Icons.add),
                    label: const Text('Make Another Picture'),
                  ),
                ),
              ],
            ],
          ),
        ),
      ),
    );
  }
}
