import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:url_launcher/url_launcher.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/widgets/shared_widgets.dart';

class HousingPage extends StatefulWidget {
  const HousingPage({super.key});

  @override
  State<HousingPage> createState() => _HousingPageState();
}

class _HousingPageState extends State<HousingPage> with SingleTickerProviderStateMixin {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _api = ApiClient();
  final _issueController = TextEditingController();
  final _applicationController = TextEditingController();
  
  late TabController _tabController;
  bool _hasPlayedIntro = false;
  bool _isLoading = false;
  String _selectedState = 'NSW';
  String _selectedHousingType = 'Private rental';
  Map<String, dynamic>? _helpResult;
  Map<String, dynamic>? _applicationResult;
  String? _error;

  final List<String> _states = ['NSW', 'VIC', 'QLD', 'WA', 'SA', 'TAS', 'NT', 'ACT'];
  final List<String> _housingTypes = [
    'Private rental',
    'Public housing',
    'Community housing',
    'Aboriginal housing',
    'Boarding house',
    'Other',
  ];

  final List<Map<String, String>> _commonIssues = [
    {'title': 'Repairs not done', 'description': 'Landlord won\'t fix things'},
    {'title': 'Rent increase', 'description': 'Being asked to pay more'},
    {'title': 'Eviction notice', 'description': 'Being told to leave'},
    {'title': 'Bond problems', 'description': 'Getting bond back'},
    {'title': 'Routine inspection coming up', 'description': 'Preparing for inspection'},
    {'title': 'Entry inspection (moving in)', 'description': 'What to check and document'},
    {'title': 'Exit inspection (moving out)', 'description': 'Getting bond back'},
    {'title': 'Lease questions', 'description': 'Understanding my lease'},
  ];

  final Map<String, Map<String, String>> _stateResources = {
    'NSW': {
      'aboriginal_housing': 'Aboriginal Housing Office',
      'aboriginal_phone': '1800 422 322',
      'tenancy_service': 'NSW Fair Trading Aboriginal Line',
      'tenancy_phone': '1800 500 330',
      'legal_service': 'Aboriginal Legal Service NSW/ACT',
      'legal_phone': '1800 765 767',
      'crisis': 'Link2home (24/7)',
      'crisis_phone': '1800 152 152',
    },
    'VIC': {
      'aboriginal_housing': 'Aboriginal Housing Victoria',
      'aboriginal_phone': '1800 248 842',
      'tenancy_service': 'Yarrka Barring (Consumer Affairs VIC)',
      'tenancy_phone': '1300 661 511',
      'legal_service': 'Victorian Aboriginal Legal Service',
      'legal_phone': '1800 064 865',
      'crisis': 'Housing Vic Crisis Line',
      'crisis_phone': '1800 825 955',
    },
    'QLD': {
      'aboriginal_housing': 'Aboriginal & Torres Strait Islander Housing',
      'aboriginal_phone': '13 74 68',
      'tenancy_service': 'Tenants Queensland',
      'tenancy_phone': '1300 744 263',
      'legal_service': 'Aboriginal & Torres Strait Islander Legal Service QLD',
      'legal_phone': '1800 012 255',
      'crisis': 'Housing Service Centre',
      'crisis_phone': '1300 474 753',
    },
    'WA': {
      'aboriginal_housing': 'Housing Authority WA',
      'aboriginal_phone': '1800 093 325',
      'tenancy_service': 'Tenancy WA',
      'tenancy_phone': '9221 0088',
      'legal_service': 'Aboriginal Legal Service WA',
      'legal_phone': '1800 019 900',
      'crisis': 'Entrypoint Perth',
      'crisis_phone': '1800 124 684',
    },
    'SA': {
      'aboriginal_housing': 'SA Housing Authority',
      'aboriginal_phone': '131 299',
      'tenancy_service': 'Tenants Information Service',
      'tenancy_phone': '1800 060 462',
      'legal_service': 'Aboriginal Legal Rights Movement',
      'legal_phone': '1800 643 222',
      'crisis': 'Housing SA Crisis',
      'crisis_phone': '1800 003 308',
    },
    'TAS': {
      'aboriginal_housing': 'Housing Tasmania',
      'aboriginal_phone': '1300 135 513',
      'tenancy_service': 'Tenants Union Tasmania',
      'tenancy_phone': '1300 652 641',
      'legal_service': 'Tasmanian Aboriginal Legal Service',
      'legal_phone': '1800 595 162',
      'crisis': 'Housing Connect',
      'crisis_phone': '1800 800 588',
    },
    'NT': {
      'aboriginal_housing': 'Territory Housing',
      'aboriginal_phone': '1800 681 700',
      'tenancy_service': 'Darwin Community Legal Service',
      'tenancy_phone': '8982 1111',
      'legal_service': 'North Australian Aboriginal Justice Agency',
      'legal_phone': '1800 898 251',
      'crisis': 'NT Shelter',
      'crisis_phone': '1800 681 700',
    },
    'ACT': {
      'aboriginal_housing': 'Housing ACT',
      'aboriginal_phone': '13 34 27',
      'tenancy_service': 'Tenants ACT',
      'tenancy_phone': '6207 1077',
      'legal_service': 'Aboriginal Legal Service NSW/ACT',
      'legal_phone': '1800 765 767',
      'crisis': 'OneLink',
      'crisis_phone': '1800 176 468',
    },
  };

  @override
  void initState() {
    super.initState();
    _tabController = TabController(length: 2, vsync: this);
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    if (mounted && !_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      await Future.delayed(const Duration(milliseconds: 300));
      _voiceService.speak(
        "G'day! Housing Helper's got your back - whether you're looking for a place or need help with a problem. "
        "Use the tabs up top to switch between finding a rental or sorting out issues. "
        "I've got mob-run services and Aboriginal housing contacts for your state too."
      );
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _tabController.dispose();
    _issueController.dispose();
    _applicationController.dispose();
    super.dispose();
  }

  Future<void> _makeCall(String phone) async {
    final cleaned = phone.replaceAll(RegExp(r'[^\d]'), '');
    final uri = Uri.parse('tel:$cleaned');
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri);
    }
  }

  Future<void> _getIssueHelp() async {
    if (_issueController.text.isEmpty) {
      setState(() => _error = 'Please describe your housing issue');
      return;
    }
    
    setState(() {
      _isLoading = true;
      _error = null;
    });
    
    try {
      final result = await _api.post('/api/housing/help', {
        'issue': _issueController.text,
        'housingType': _selectedHousingType,
        'state': _selectedState,
      });
      
      setState(() {
        _helpResult = result;
        _isLoading = false;
      });
      
      if (_prefs.autoPlayVoice && result['theirRights'] != null) {
        final rights = result['theirRights'] as List;
        if (rights.isNotEmpty) {
          _voiceService.speak(
            "Here's what you need to know. ${rights.first}. "
            "I've also got steps for what to do next."
          );
        }
      }
    } catch (e) {
      setState(() {
        _error = 'Having trouble getting help. Try again?';
        _isLoading = false;
      });
    }
  }

  Future<void> _getApplicationHelp() async {
    if (_applicationController.text.isEmpty) {
      setState(() => _error = 'Tell me a bit about your situation');
      return;
    }
    
    setState(() {
      _isLoading = true;
      _error = null;
    });
    
    try {
      final result = await _api.post('/api/housing/application-help', {
        'situation': _applicationController.text,
        'state': _selectedState,
      });
      
      setState(() {
        _applicationResult = result;
        _isLoading = false;
      });
      
      if (_prefs.autoPlayVoice && result['summary'] != null) {
        _voiceService.speak(result['summary']);
      }
    } catch (e) {
      setState(() {
        _error = 'Having trouble. Try again?';
        _isLoading = false;
      });
    }
  }

  void _selectIssue(String issue) {
    _issueController.text = issue;
  }

  void _resetIssue() {
    setState(() {
      _helpResult = null;
      _issueController.clear();
    });
  }

  void _resetApplication() {
    setState(() {
      _applicationResult = null;
      _applicationController.clear();
    });
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              Padding(
                padding: const EdgeInsets.all(16.0),
                child: Row(
                  children: [
                    IconButton(
                      icon: const Icon(Icons.arrow_back),
                      onPressed: () {
                        if (_helpResult != null) {
                          _resetIssue();
                        } else if (_applicationResult != null) {
                          _resetApplication();
                        } else {
                          context.go('/');
                        }
                      },
                    ),
                    const Spacer(),
                    IconButton(
                      icon: const Icon(Icons.volume_up),
                      onPressed: () {
                        _voiceService.speak(
                          "Use the tabs to switch between finding a rental or getting help with housing issues."
                        );
                      },
                    ),
                    IconButton(
                      icon: const Icon(Icons.settings_outlined),
                      onPressed: () => context.go('/settings'),
                    ),
                  ],
                ),
              ),
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 20.0),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(
                          Icons.home_outlined,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                          size: 28,
                        ),
                        const SizedBox(width: 12),
                        Text(
                          'Housing Helper',
                          style: Theme.of(context).textTheme.headlineMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(
                      'Find a place or sort out issues',
                      style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                  ],
                ),
              ),
              const SizedBox(height: 16),
              Container(
                margin: const EdgeInsets.symmetric(horizontal: 16),
                decoration: BoxDecoration(
                  color: isDark ? Colors.white10 : Colors.black.withOpacity(0.05),
                  borderRadius: BorderRadius.circular(12),
                ),
                child: TabBar(
                  controller: _tabController,
                  indicator: BoxDecoration(
                    color: isDark ? AppColors.primaryLight : AppColors.primary,
                    borderRadius: BorderRadius.circular(10),
                  ),
                  indicatorSize: TabBarIndicatorSize.tab,
                  labelColor: isDark ? Colors.black : Colors.white,
                  unselectedLabelColor: isDark ? Colors.white70 : Colors.black87,
                  dividerColor: Colors.transparent,
                  labelStyle: const TextStyle(fontWeight: FontWeight.bold, fontSize: 14),
                  tabs: const [
                    Tab(
                      child: Row(
                        mainAxisAlignment: MainAxisAlignment.center,
                        children: [
                          Icon(Icons.search, size: 18),
                          SizedBox(width: 8),
                          Text('Find a Rental'),
                        ],
                      ),
                    ),
                    Tab(
                      child: Row(
                        mainAxisAlignment: MainAxisAlignment.center,
                        children: [
                          Icon(Icons.help_outline, size: 18),
                          SizedBox(width: 8),
                          Text('Got an Issue'),
                        ],
                      ),
                    ),
                  ],
                ),
              ),
              const SizedBox(height: 16),
              Expanded(
                child: TabBarView(
                  controller: _tabController,
                  children: [
                    _buildFindRentalTab(isDark),
                    _buildIssueTab(isDark),
                  ],
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildFindRentalTab(bool isDark) {
    if (_applicationResult != null) {
      return _buildApplicationResult(isDark);
    }
    
    final resources = _stateResources[_selectedState]!;
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Card(
            color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.1),
            child: Padding(
              padding: const EdgeInsets.all(16),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Row(
                    children: [
                      Icon(Icons.lightbulb_outline, 
                        color: isDark ? AppColors.primaryLight : AppColors.primary),
                      const SizedBox(width: 8),
                      Text(
                        'Mob-Run Housing Services',
                        style: Theme.of(context).textTheme.titleMedium?.copyWith(
                          fontWeight: FontWeight.bold,
                        ),
                      ),
                    ],
                  ),
                  const SizedBox(height: 12),
                  Text(
                    'These services are run by and for our mob. They understand cultural needs like extended family staying over.',
                    style: TextStyle(
                      color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                    ),
                  ),
                  const SizedBox(height: 12),
                  _buildContactCard(
                    resources['aboriginal_housing']!,
                    resources['aboriginal_phone']!,
                    Icons.home,
                    isDark,
                  ),
                  const SizedBox(height: 8),
                  _buildContactCard(
                    resources['legal_service']!,
                    resources['legal_phone']!,
                    Icons.gavel,
                    isDark,
                  ),
                ],
              ),
            ),
          ),
          const SizedBox(height: 20),
          
          Text(
            'Tips for Winning a Rental',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 12),
          _buildTipCard(
            '1',
            'Get Support First',
            'Contact Aboriginal housing services - they can attend inspections with you, help with applications, and even help with bond money.',
            Icons.people_outline,
            isDark,
          ),
          _buildTipCard(
            '2',
            'Gather Your Docs',
            'ID, payslips or Centrelink letter, rental history if you have it. Don\'t stress if you don\'t have everything - support services can help.',
            Icons.folder_outlined,
            isDark,
          ),
          _buildTipCard(
            '3',
            'Apply for Multiple Places',
            'It\'s normal to apply for lots of places. Keep copies of everything you send.',
            Icons.content_copy_outlined,
            isDark,
          ),
          _buildTipCard(
            '4',
            'Ace the Inspections',
            'At open inspections, arrive early and be friendly. When you move in, take photos of EVERYTHING - walls, floors, appliances. This protects your bond later.',
            Icons.camera_alt_outlined,
            isDark,
          ),
          _buildTipCard(
            '5',
            'Know Your Rights',
            'Landlords can\'t discriminate. If you feel you\'re being treated unfairly because you\'re Aboriginal, contact the legal service.',
            Icons.shield_outlined,
            isDark,
          ),
          const SizedBox(height: 20),
          
          Row(
            children: [
              Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      'Your State',
                      style: Theme.of(context).textTheme.titleSmall,
                    ),
                    const SizedBox(height: 4),
                    DropdownButtonFormField<String>(
                      value: _selectedState,
                      decoration: InputDecoration(
                        border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
                        contentPadding: const EdgeInsets.symmetric(horizontal: 12),
                      ),
                      items: _states.map((s) => DropdownMenuItem(value: s, child: Text(s))).toList(),
                      onChanged: (v) => setState(() => _selectedState = v!),
                    ),
                  ],
                ),
              ),
            ],
          ),
          const SizedBox(height: 20),
          
          Text(
            'Tell me your situation',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          TextField(
            controller: _applicationController,
            decoration: InputDecoration(
              hintText: 'e.g., Looking for a 2-bedroom for me and my kids, currently staying with family',
              border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
              suffixIcon: MicButton(onResult: (t) => _applicationController.text = t),
            ),
            maxLines: 3,
          ),
          const SizedBox(height: 16),
          
          if (_error != null)
            Padding(
              padding: const EdgeInsets.only(bottom: 16),
              child: Text(_error!, style: TextStyle(color: Colors.red[400])),
            ),
          
          SizedBox(
            width: double.infinity,
            child: ElevatedButton.icon(
              onPressed: _isLoading ? null : _getApplicationHelp,
              icon: _isLoading 
                ? const SizedBox(width: 20, height: 20, child: CircularProgressIndicator(strokeWidth: 2))
                : const Icon(Icons.search),
              label: Text(_isLoading ? 'Getting advice...' : 'Get Personalised Advice'),
              style: ElevatedButton.styleFrom(
                padding: const EdgeInsets.symmetric(vertical: 16),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
            ),
          ),
          const SizedBox(height: 16),
          
          Card(
            color: Colors.orange.withOpacity(0.1),
            child: Padding(
              padding: const EdgeInsets.all(16),
              child: Row(
                children: [
                  Icon(Icons.warning_amber, color: Colors.orange[700]),
                  const SizedBox(width: 12),
                  Expanded(
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Text(
                          'Need a place urgently?',
                          style: Theme.of(context).textTheme.titleSmall?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                        const SizedBox(height: 4),
                        Text(
                          'Call ${resources['crisis']}',
                          style: TextStyle(color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary),
                        ),
                      ],
                    ),
                  ),
                  IconButton(
                    icon: const Icon(Icons.phone),
                    onPressed: () => _makeCall(resources['crisis_phone']!),
                  ),
                ],
              ),
            ),
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }

  Widget _buildContactCard(String name, String phone, IconData icon, bool isDark) {
    return InkWell(
      onTap: () => _makeCall(phone),
      borderRadius: BorderRadius.circular(8),
      child: Container(
        padding: const EdgeInsets.all(12),
        decoration: BoxDecoration(
          color: isDark ? Colors.white10 : Colors.white,
          borderRadius: BorderRadius.circular(8),
        ),
        child: Row(
          children: [
            Icon(icon, size: 20, color: isDark ? AppColors.primaryLight : AppColors.primary),
            const SizedBox(width: 12),
            Expanded(
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(name, style: const TextStyle(fontWeight: FontWeight.w500)),
                  Text(phone, style: TextStyle(
                    color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                    fontSize: 13,
                  )),
                ],
              ),
            ),
            Icon(Icons.phone, size: 20, color: Colors.green[400]),
          ],
        ),
      ),
    );
  }

  Widget _buildTipCard(String number, String title, String description, IconData icon, bool isDark) {
    return Padding(
      padding: const EdgeInsets.only(bottom: 12),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Container(
            width: 28,
            height: 28,
            decoration: BoxDecoration(
              gradient: LinearGradient(
                colors: [
                  isDark ? AppColors.primaryLight : AppColors.primary,
                  AppColors.ochre,
                ],
              ),
              shape: BoxShape.circle,
            ),
            child: Center(
              child: Text(
                number,
                style: const TextStyle(
                  color: Colors.white,
                  fontWeight: FontWeight.bold,
                  fontSize: 14,
                ),
              ),
            ),
          ),
          const SizedBox(width: 12),
          Expanded(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(
                  title,
                  style: Theme.of(context).textTheme.titleSmall?.copyWith(
                    fontWeight: FontWeight.bold,
                  ),
                ),
                const SizedBox(height: 4),
                Text(
                  description,
                  style: TextStyle(
                    color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                    fontSize: 13,
                  ),
                ),
              ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildApplicationResult(bool isDark) {
    final result = _applicationResult!;
    final resources = _stateResources[_selectedState]!;
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          if (result['summary'] != null) ...[
            Card(
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.lightbulb_outline, 
                          color: isDark ? AppColors.primaryLight : AppColors.primary),
                        const SizedBox(width: 8),
                        Text(
                          'Your Plan',
                          style: Theme.of(context).textTheme.titleMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                        const Spacer(),
                        IconButton(
                          icon: const Icon(Icons.volume_up, size: 20),
                          onPressed: () => _voiceService.speak(result['summary']),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(result['summary']),
                  ],
                ),
              ),
            ),
            const SizedBox(height: 16),
          ],
          
          if (result['steps'] != null && (result['steps'] as List).isNotEmpty) ...[
            Text(
              'Next Steps',
              style: Theme.of(context).textTheme.titleMedium?.copyWith(
                fontWeight: FontWeight.bold,
              ),
            ),
            const SizedBox(height: 12),
            ...((result['steps'] as List).asMap().entries.map((entry) => _buildTipCard(
              '${entry.key + 1}',
              '',
              entry.value.toString(),
              Icons.check,
              isDark,
            ))),
            const SizedBox(height: 16),
          ],
          
          if (result['eligiblePrograms'] != null && (result['eligiblePrograms'] as List).isNotEmpty) ...[
            Text(
              'Programs You May Be Eligible For',
              style: Theme.of(context).textTheme.titleMedium?.copyWith(
                fontWeight: FontWeight.bold,
              ),
            ),
            const SizedBox(height: 8),
            ...((result['eligiblePrograms'] as List).map((program) => Card(
              margin: const EdgeInsets.only(bottom: 8),
              child: ListTile(
                leading: Icon(Icons.check_circle, color: Colors.green[400]),
                title: Text(program.toString()),
              ),
            ))),
            const SizedBox(height: 16),
          ],
          
          Text(
            'Who Can Help',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          _buildContactCard(
            resources['aboriginal_housing']!,
            resources['aboriginal_phone']!,
            Icons.home,
            isDark,
          ),
          const SizedBox(height: 8),
          _buildContactCard(
            resources['tenancy_service']!,
            resources['tenancy_phone']!,
            Icons.support_agent,
            isDark,
          ),
          const SizedBox(height: 8),
          _buildContactCard(
            resources['legal_service']!,
            resources['legal_phone']!,
            Icons.gavel,
            isDark,
          ),
          const SizedBox(height: 24),
          
          SizedBox(
            width: double.infinity,
            child: OutlinedButton.icon(
              onPressed: _resetApplication,
              icon: const Icon(Icons.refresh),
              label: const Text('Ask About Something Else'),
              style: OutlinedButton.styleFrom(
                padding: const EdgeInsets.symmetric(vertical: 16),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
            ),
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }

  Widget _buildIssueTab(bool isDark) {
    if (_helpResult != null) {
      return _buildIssueResult(isDark);
    }
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(
            'Common Issues',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          Wrap(
            spacing: 8,
            runSpacing: 8,
            children: _commonIssues.map((issue) => ActionChip(
              label: Text(issue['title']!),
              onPressed: () => _selectIssue(issue['title']!),
            )).toList(),
          ),
          const SizedBox(height: 20),
          
          Text(
            'Describe your issue',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          TextField(
            controller: _issueController,
            decoration: InputDecoration(
              hintText: 'e.g., My landlord won\'t fix the leaking roof',
              border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
              suffixIcon: MicButton(onResult: (t) => _issueController.text = t),
            ),
            maxLines: 3,
          ),
          const SizedBox(height: 16),
          
          Row(
            children: [
              Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      'State',
                      style: Theme.of(context).textTheme.titleSmall,
                    ),
                    const SizedBox(height: 4),
                    DropdownButtonFormField<String>(
                      value: _selectedState,
                      decoration: InputDecoration(
                        border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
                        contentPadding: const EdgeInsets.symmetric(horizontal: 12),
                      ),
                      items: _states.map((s) => DropdownMenuItem(value: s, child: Text(s))).toList(),
                      onChanged: (v) => setState(() => _selectedState = v!),
                    ),
                  ],
                ),
              ),
              const SizedBox(width: 16),
              Expanded(
                flex: 2,
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      'Housing Type',
                      style: Theme.of(context).textTheme.titleSmall,
                    ),
                    const SizedBox(height: 4),
                    DropdownButtonFormField<String>(
                      value: _selectedHousingType,
                      decoration: InputDecoration(
                        border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
                        contentPadding: const EdgeInsets.symmetric(horizontal: 12),
                      ),
                      items: _housingTypes.map((h) => DropdownMenuItem(value: h, child: Text(h))).toList(),
                      onChanged: (v) => setState(() => _selectedHousingType = v!),
                    ),
                  ],
                ),
              ),
            ],
          ),
          const SizedBox(height: 24),
          
          if (_error != null)
            Padding(
              padding: const EdgeInsets.only(bottom: 16),
              child: Text(_error!, style: TextStyle(color: Colors.red[400])),
            ),
          
          SizedBox(
            width: double.infinity,
            child: ElevatedButton.icon(
              onPressed: _isLoading ? null : _getIssueHelp,
              icon: _isLoading 
                ? const SizedBox(width: 20, height: 20, child: CircularProgressIndicator(strokeWidth: 2))
                : const Icon(Icons.help_outline),
              label: Text(_isLoading ? 'Getting help...' : 'Get My Rights'),
              style: ElevatedButton.styleFrom(
                padding: const EdgeInsets.symmetric(vertical: 16),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
            ),
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }

  Widget _buildIssueResult(bool isDark) {
    final result = _helpResult!;
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          if (result['theirRights'] != null && (result['theirRights'] as List).isNotEmpty) ...[
            Card(
              color: Colors.green.withOpacity(0.1),
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.shield, color: Colors.green[400]),
                        const SizedBox(width: 8),
                        Text(
                          'Your Rights',
                          style: Theme.of(context).textTheme.titleMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                        const Spacer(),
                        IconButton(
                          icon: const Icon(Icons.volume_up, size: 20),
                          onPressed: () {
                            final rights = result['theirRights'] as List;
                            _voiceService.speak(rights.join('. '));
                          },
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    ...((result['theirRights'] as List).map((right) => Padding(
                      padding: const EdgeInsets.only(bottom: 8),
                      child: Row(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Icon(Icons.check_circle, color: Colors.green[400], size: 18),
                          const SizedBox(width: 8),
                          Expanded(child: Text(right.toString())),
                        ],
                      ),
                    ))),
                  ],
                ),
              ),
            ),
            const SizedBox(height: 16),
          ],
          
          if (result['steps'] != null && (result['steps'] as List).isNotEmpty) ...[
            Text(
              'What To Do',
              style: Theme.of(context).textTheme.titleMedium?.copyWith(
                fontWeight: FontWeight.bold,
              ),
            ),
            const SizedBox(height: 8),
            ...((result['steps'] as List).asMap().entries.map((entry) => Padding(
              padding: const EdgeInsets.only(bottom: 12),
              child: Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Container(
                    width: 24,
                    height: 24,
                    decoration: BoxDecoration(
                      color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.2),
                      shape: BoxShape.circle,
                    ),
                    child: Center(
                      child: Text(
                        '${entry.key + 1}',
                        style: TextStyle(
                          fontWeight: FontWeight.bold,
                          fontSize: 12,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                        ),
                      ),
                    ),
                  ),
                  const SizedBox(width: 12),
                  Expanded(child: Text(entry.value.toString())),
                ],
              ),
            ))),
            const SizedBox(height: 16),
          ],
          
          if (result['letterTemplate'] != null && result['letterTemplate'].toString().isNotEmpty) ...[
            Card(
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        const Icon(Icons.mail_outline),
                        const SizedBox(width: 8),
                        Text(
                          'Letter Template',
                          style: Theme.of(context).textTheme.titleMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 12),
                    Container(
                      padding: const EdgeInsets.all(12),
                      decoration: BoxDecoration(
                        color: isDark ? Colors.black26 : Colors.grey[100],
                        borderRadius: BorderRadius.circular(8),
                      ),
                      child: Text(
                        result['letterTemplate'].toString(),
                        style: const TextStyle(fontFamily: 'monospace', fontSize: 12),
                      ),
                    ),
                  ],
                ),
              ),
            ),
            const SizedBox(height: 16),
          ],
          
          if (result['helpContacts'] != null && (result['helpContacts'] as List).isNotEmpty) ...[
            Text(
              'Get More Help',
              style: Theme.of(context).textTheme.titleMedium?.copyWith(
                fontWeight: FontWeight.bold,
              ),
            ),
            const SizedBox(height: 8),
            ...((result['helpContacts'] as List).map((contact) => Card(
              margin: const EdgeInsets.only(bottom: 8),
              child: ListTile(
                leading: const Icon(Icons.phone),
                title: Text(contact.toString()),
              ),
            ))),
            const SizedBox(height: 16),
          ],
          
          SizedBox(
            width: double.infinity,
            child: OutlinedButton.icon(
              onPressed: _resetIssue,
              icon: const Icon(Icons.refresh),
              label: const Text('Ask About Another Issue'),
              style: OutlinedButton.styleFrom(
                padding: const EdgeInsets.symmetric(vertical: 16),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
            ),
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }
}
