import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/voice_scripts.dart';
import '../../../core/widgets/ceremonial_fire.dart';
import '../../../core/widgets/shared_widgets.dart';

class HomePage extends StatefulWidget {
  const HomePage({super.key});

  @override
  State<HomePage> createState() => _HomePageState();
}

class _HomePageState extends State<HomePage> with SingleTickerProviderStateMixin {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final VoiceScripts _scripts = VoiceScripts();
  bool _hasPlayedWelcome = false;
  bool _isReturningUser = false;
  String _greeting = "G'day!";
  String _subGreeting = "What's happening today?";
  
  static bool _hasVisitedHomeThisSession = false;
  
  late AnimationController _fadeController;
  late Animation<double> _fadeAnimation;

  @override
  void initState() {
    super.initState();
    _fadeController = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 1200),
    );
    _fadeAnimation = CurvedAnimation(
      parent: _fadeController,
      curve: Curves.easeOut,
    );
    
    _checkReturningUser();
    _initializeAndPlay();
    _fadeController.forward();
  }

  void _checkReturningUser() {
    _isReturningUser = _prefs.isReturningUser;
    if (_isReturningUser) {
      _greeting = _prefs.getWelcomeGreetingWithRegion();
      final regionInfo = _prefs.regionInfo;
      if (_prefs.region.index > 0) {
        _subGreeting = regionInfo.acknowledgment;
      } else {
        _subGreeting = "Good to see you again";
      }
    }
    _prefs.updateLastVisit();
  }

  Future<void> _initializeAndPlay() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 800));
      _playHomeWelcome();
    }
  }

  @override
  void dispose() {
    _fadeController.dispose();
    _voiceService.stop();
    super.dispose();
  }

  void _playHomeWelcome() {
    if (!_hasPlayedWelcome && _prefs.autoPlayVoice) {
      _hasPlayedWelcome = true;
      
      String message;
      
      if (_hasVisitedHomeThisSession) {
        message = _scripts.backAtFire;
      } else if (_isReturningUser) {
        message = _scripts.welcomeReturning(_prefs.userName);
      } else {
        message = _scripts.welcomeFirstVisit;
      }
      
      _hasVisitedHomeThisSession = true;
      _voiceService.speak(message);
    }
  }

  void _navigateWithVoice(String route, String announcement) {
    _voiceService.stop();
    if (_prefs.autoPlayVoice) {
      _voiceService.speak(announcement);
    }
    Future.delayed(const Duration(milliseconds: 100), () {
      context.go(route);
    });
  }

  void _startYarn() {
    _voiceService.stop();
    if (_prefs.autoPlayVoice) {
      _voiceService.speak(_scripts.startListening);
    }
    context.go('/yarn-flow/story-catcher');
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final screenHeight = MediaQuery.of(context).size.height;
    final isLandscape = MediaQuery.of(context).orientation == Orientation.landscape;
    
    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: FadeTransition(
            opacity: _fadeAnimation,
            child: isLandscape 
              ? _buildLandscapeLayout(isDark, screenHeight)
              : _buildPortraitLayout(isDark, screenHeight),
          ),
        ),
      ),
    );
  }
  
  Widget _buildPortraitLayout(bool isDark, double screenHeight) {
    return Column(
      children: [
        _buildHeader(isDark),
        
        Expanded(
          child: SingleChildScrollView(
            child: Column(
              mainAxisAlignment: MainAxisAlignment.center,
              children: [
                SizedBox(height: screenHeight * 0.02),
                _buildGreeting(isDark),
                SizedBox(height: screenHeight * 0.02),
                _buildCeremonialFire(),
                SizedBox(height: screenHeight * 0.02),
                _buildYarnPrompt(isDark),
                SizedBox(height: screenHeight * 0.03),
                _buildQuickAccess(isDark),
                const SizedBox(height: 16),
              ],
            ),
          ),
        ),
      ],
    );
  }
  
  Widget _buildLandscapeLayout(bool isDark, double screenHeight) {
    return Column(
      children: [
        _buildHeader(isDark),
        
        Expanded(
          child: Row(
            children: [
              Expanded(
                flex: 1,
                child: SingleChildScrollView(
                  child: Column(
                    mainAxisAlignment: MainAxisAlignment.center,
                    children: [
                      _buildGreeting(isDark),
                      const SizedBox(height: 16),
                      _buildCeremonialFire(),
                      const SizedBox(height: 16),
                      _buildYarnPrompt(isDark),
                    ],
                  ),
                ),
              ),
              Expanded(
                flex: 1,
                child: SingleChildScrollView(
                  padding: const EdgeInsets.symmetric(horizontal: 8),
                  child: _buildQuickAccess(isDark),
                ),
              ),
            ],
          ),
        ),
      ],
    );
  }
  
  Widget _buildHeader(bool isDark) {
    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 8),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          IconButton(
            icon: const Icon(Icons.volume_up),
            onPressed: () {
              _voiceService.speak(
                "You're at the Welcome Fire. "
                "Tap the fire in the middle to start a yarn about whatever's on your mind. "
                "Or use the quick access buttons at the bottom for specific help."
              );
            },
            tooltip: 'Read page aloud',
          ),
          Row(
            children: [
              IconButton(
                icon: Icon(isDark ? Icons.light_mode : Icons.dark_mode),
                onPressed: () {
                  final prefs = _prefs;
                  prefs.setDarkMode(!isDark);
                  if (mounted) setState(() {});
                },
                tooltip: isDark ? 'Light mode' : 'Dark mode',
              ),
              IconButton(
                icon: const Icon(Icons.phone_outlined),
                onPressed: () => _navigateWithVoice('/numbers', "Your important numbers"),
                tooltip: 'Important numbers',
              ),
              IconButton(
                icon: const Icon(Icons.settings_outlined),
                onPressed: () => _navigateWithVoice('/settings', "Opening settings"),
                tooltip: 'Settings',
              ),
            ],
          ),
        ],
      ),
    );
  }
  
  Widget _buildGreeting(bool isDark) {
    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 24),
      child: Column(
        children: [
          Text(
            _greeting,
            style: Theme.of(context).textTheme.headlineLarge?.copyWith(
              fontSize: 32,
            ),
            textAlign: TextAlign.center,
          ),
          const SizedBox(height: 4),
          Text(
            _subGreeting,
            style: Theme.of(context).textTheme.bodyLarge?.copyWith(
              color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
            ),
            textAlign: TextAlign.center,
          ),
        ],
      ),
    );
  }
  
  Widget _buildCeremonialFire() {
    return CeremonialFire(
      size: 220,
      onTap: _startYarn,
    );
  }
  
  Widget _buildYarnPrompt(bool isDark) {
    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 32),
      child: Column(
        children: [
          FunkyMicButton(
            onTap: _startYarn,
            size: 72,
            label: "Tap to speak",
          ),
          const SizedBox(height: 16),
          Text(
            "Tell me what's on your mind",
            style: Theme.of(context).textTheme.bodyMedium?.copyWith(
              color: isDark ? AppColors.textTertiaryDark : AppColors.textTertiary,
            ),
            textAlign: TextAlign.center,
          ),
        ],
      ),
    );
  }
  
  Widget _buildQuickAccess(bool isDark) {
    return Container(
      margin: const EdgeInsets.symmetric(horizontal: 16),
      padding: const EdgeInsets.all(16),
      decoration: BoxDecoration(
        color: (isDark ? AppColors.cardDark : AppColors.cardLight).withOpacity(0.7),
        borderRadius: BorderRadius.circular(16),
        border: Border.all(
          color: (isDark ? AppColors.textTertiaryDark : AppColors.textTertiary).withOpacity(0.2),
        ),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(
            "Quick help",
            style: Theme.of(context).textTheme.labelSmall?.copyWith(
              color: isDark ? AppColors.textTertiaryDark : AppColors.textTertiary,
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 12),
          // Top row - 3 buttons symmetrical
          Row(
            children: [
              Expanded(
                child: _QuickAccessButton(
                  icon: Icons.phone_outlined,
                  label: "Call help",
                  onTap: () => _navigateWithVoice('/call-coach', "Let's get you ready for that call"),
                ),
              ),
              Expanded(
                child: _QuickAccessButton(
                  icon: Icons.description_outlined,
                  label: "Got a letter",
                  onTap: () => _navigateWithVoice('/documents', "Bring your letter, I'll explain it"),
                ),
              ),
              Expanded(
                child: _QuickAccessButton(
                  icon: Icons.favorite_outline,
                  label: "Need support",
                  color: Colors.red[400],
                  onTap: () => _navigateWithVoice('/safety', "Your safe space with crisis contacts"),
                ),
              ),
            ],
          ),
          const SizedBox(height: 8),
          // Bottom row - 3 buttons symmetrical
          Row(
            children: [
              Expanded(
                child: _QuickAccessButton(
                  icon: Icons.chat_outlined,
                  label: "Ask anything",
                  onTap: () => _navigateWithVoice('/chat', "I'm listening, what's on your mind?"),
                ),
              ),
              Expanded(
                child: _QuickAccessButton(
                  icon: Icons.account_balance_wallet,
                  label: "Money help",
                  onTap: () => _navigateWithVoice('/money', "Let's yarn about your money"),
                ),
              ),
              Expanded(
                child: _QuickAccessButton(
                  icon: Icons.more_horiz,
                  label: "More",
                  onTap: () => _showMoreSheet(context),
                ),
              ),
            ],
          ),
        ],
      ),
    );
  }

  void _showMoreSheet(BuildContext context) {
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Here's more ways I can help. Pick what feels right.");
    }
    
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    showModalBottomSheet(
      context: context,
      backgroundColor: isDark ? AppColors.cardDark : AppColors.cardLight,
      isScrollControlled: true,
      shape: const RoundedRectangleBorder(
        borderRadius: BorderRadius.vertical(top: Radius.circular(20)),
      ),
      builder: (context) => DraggableScrollableSheet(
        initialChildSize: 0.7,
        minChildSize: 0.5,
        maxChildSize: 0.9,
        expand: false,
        builder: (context, scrollController) => SingleChildScrollView(
          controller: scrollController,
          padding: const EdgeInsets.all(24.0),
          child: Column(
            mainAxisSize: MainAxisSize.min,
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Center(
                child: Container(
                  width: 40,
                  height: 4,
                  margin: const EdgeInsets.only(bottom: 16),
                  decoration: BoxDecoration(
                    color: Colors.grey[400],
                    borderRadius: BorderRadius.circular(2),
                  ),
                ),
              ),
              
              Center(
                child: Text(
                  "What kind of help do you need?",
                  style: Theme.of(context).textTheme.titleLarge,
                ),
              ),
              const SizedBox(height: 8),
              Center(
                child: Text(
                  "Or just tap the fire and tell me your story",
                  style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                    color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                  ),
                ),
              ),
              const SizedBox(height: 24),
              
              _MoreSheetSection(
                title: "CORE FEATURES",
                items: [
                  _MoreSheetItem(
                    icon: Icons.phone_outlined,
                    title: "Call Coach",
                    subtitle: "Practice before a tough call",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/call-coach', "Let's get you ready for that call");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.mail_outlined,
                    title: "Letter Helper",
                    subtitle: "Photo a letter, I'll explain it",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/letter-helper', "Bring your letter, I'll help explain it");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.chat_outlined,
                    title: "Ask Anything",
                    subtitle: "Chat about whatever's on your mind",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/chat', "I'm listening, what's on your mind?");
                    },
                  ),
                ],
              ),
              
              const SizedBox(height: 16),
              _MoreSheetSection(
                title: "URGENT HELP",
                items: [
                  _MoreSheetItem(
                    icon: Icons.favorite,
                    iconColor: Colors.red[400],
                    title: "Safety Net",
                    subtitle: "Crisis support, 13YARN, safe space",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/safety', "Your safe space");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.restaurant,
                    iconColor: AppColors.ochre,
                    title: "Food & Essentials",
                    subtitle: "Emergency food, clothing, bills",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/food', "Let's find you some help");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.account_balance,
                    iconColor: AppColors.terracotta,
                    title: "Centrelink Navigator",
                    subtitle: "Payments, breaches, appeals",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/centrelink', "Let's sort out your Centrelink");
                    },
                  ),
                ],
              ),
              
              const SizedBox(height: 16),
              _MoreSheetSection(
                title: "EVERYDAY STUFF",
                items: [
                  _MoreSheetItem(
                    icon: Icons.account_balance_wallet,
                    title: "Money Yarns",
                    subtitle: "Budget & Centrelink help",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/money', "Let's yarn about your money");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.home_outlined,
                    title: "Housing Helper",
                    subtitle: "Renting, landlords, tenancy rights",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/housing', "Know your rights as a renter");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.school_outlined,
                    title: "School Helper",
                    subtitle: "Kids, teachers, school stuff",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/school', "Help with school stuff");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.work_outline,
                    title: "Job Spotter",
                    subtitle: "Find work, training, opportunities",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/jobs', "Let's find you some work");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.description_outlined,
                    title: "Form Filler",
                    subtitle: "Help with government forms",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/forms', "Help you fill out those forms");
                    },
                  ),
                ],
              ),
              
              const SizedBox(height: 16),
              _MoreSheetSection(
                title: "TRACK & REMEMBER",
                items: [
                  _MoreSheetItem(
                    icon: Icons.calendar_today,
                    title: "Appointments",
                    subtitle: "Prepare for meetings",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/appointments', "Get ready for your appointment");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.medication_outlined,
                    title: "Health & Meds",
                    subtitle: "Medication reminders",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/health', "Set up your health reminders");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.auto_stories,
                    title: "Story Keeper",
                    subtitle: "Your private journal",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/journal', "Your private story space");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.folder_outlined,
                    title: "Keep a Note",
                    subtitle: "Track your cases & documents",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/advocate', "Your private advocate space");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.phone_outlined,
                    title: "Important Numbers",
                    subtitle: "Saved contact numbers",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/numbers', "Your important numbers");
                    },
                  ),
                ],
              ),
              
              const SizedBox(height: 16),
              _MoreSheetSection(
                title: "HANDY BITS",
                items: [
                  _MoreSheetItem(
                    icon: Icons.image_outlined,
                    title: "Make a Picture",
                    subtitle: "Create images from words",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/images', "What picture would you like?");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.edit_document,
                    title: "Work Stuff",
                    subtitle: "Resumes, cover letters",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/skills', "Let's work on your job stuff");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.help_outline,
                    title: "What is AI?",
                    subtitle: "Learn about AI",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/welcome-yarn', "Let me tell you about AI");
                    },
                  ),
                ],
              ),
              
              const SizedBox(height: 16),
              _MoreSheetSection(
                title: "RESEARCH MODULES",
                items: [
                  _MoreSheetItem(
                    icon: Icons.family_restroom,
                    iconColor: AppColors.ochre,
                    title: "Parent Helper",
                    subtitle: "Walking with your kids",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/parent-helper', "Support for parents facing tough situations");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.shield_outlined,
                    iconColor: Colors.red[400],
                    title: "Family Safety",
                    subtitle: "Staying safe",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/family-safety', "Your safe space for family safety");
                    },
                  ),
                ],
              ),
              
              const SizedBox(height: 16),
              _MoreSheetSection(
                title: "FOR SUPPORT WORKERS",
                items: [
                  _MoreSheetItem(
                    icon: Icons.business_outlined,
                    title: "Partner Console",
                    subtitle: "Community impact data",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/partner', "Partner console for support workers");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.school,
                    title: "Trainer Hub",
                    subtitle: "Training resources",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/trainer', "Welcome to the Trainer Hub");
                    },
                  ),
                  _MoreSheetItem(
                    icon: Icons.bar_chart,
                    title: "Impact Metrics",
                    subtitle: "Usage statistics",
                    onTap: () {
                      Navigator.pop(context);
                      _navigateWithVoice('/impact', "Community impact metrics");
                    },
                  ),
                ],
              ),
              
              const SizedBox(height: 24),
              Center(
                child: TextButton.icon(
                  onPressed: () {
                    Navigator.pop(context);
                    _navigateWithVoice('/privacy', "Your privacy matters to us");
                  },
                  icon: const Icon(Icons.shield_outlined, size: 18),
                  label: const Text("Privacy"),
                ),
              ),
              const SizedBox(height: 16),
            ],
          ),
        ),
      ),
    );
  }
}

class _QuickAccessButton extends StatelessWidget {
  final IconData icon;
  final String label;
  final VoidCallback onTap;
  final Color? color;

  const _QuickAccessButton({
    required this.icon,
    required this.label,
    required this.onTap,
    this.color,
  });

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final iconColor = color ?? (isDark ? AppColors.primaryLight : AppColors.primary);
    
    return InkWell(
      onTap: onTap,
      borderRadius: BorderRadius.circular(12),
      child: Padding(
        padding: const EdgeInsets.all(8),
        child: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            Container(
              padding: const EdgeInsets.all(12),
              decoration: BoxDecoration(
                color: iconColor.withOpacity(0.15),
                borderRadius: BorderRadius.circular(12),
              ),
              child: Icon(icon, color: iconColor, size: 24),
            ),
            const SizedBox(height: 6),
            Text(
              label,
              style: Theme.of(context).textTheme.bodySmall?.copyWith(
                fontSize: 11,
              ),
              textAlign: TextAlign.center,
            ),
          ],
        ),
      ),
    );
  }
}

class _MoreSheetSection extends StatelessWidget {
  final String title;
  final List<_MoreSheetItem> items;

  const _MoreSheetSection({
    required this.title,
    required this.items,
  });

  @override
  Widget build(BuildContext context) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Padding(
          padding: const EdgeInsets.only(left: 16, bottom: 8),
          child: Text(
            title,
            style: Theme.of(context).textTheme.labelSmall?.copyWith(
              color: Colors.grey,
              fontWeight: FontWeight.bold,
            ),
          ),
        ),
        ...items,
      ],
    );
  }
}

class _MoreSheetItem extends StatelessWidget {
  final IconData icon;
  final Color? iconColor;
  final String title;
  final String subtitle;
  final VoidCallback onTap;

  const _MoreSheetItem({
    required this.icon,
    this.iconColor,
    required this.title,
    required this.subtitle,
    required this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    return ListTile(
      leading: Icon(icon, color: iconColor),
      title: Text(title),
      subtitle: Text(subtitle),
      onTap: onTap,
    );
  }
}
