import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:hive_flutter/hive_flutter.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class HealthPage extends StatefulWidget {
  const HealthPage({super.key});

  @override
  State<HealthPage> createState() => _HealthPageState();
}

class _HealthPageState extends State<HealthPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ScrollController _scrollController = ScrollController();
  
  final _medicationController = TextEditingController();
  final _dosageController = TextEditingController();
  final _notesController = TextEditingController();
  
  bool _hasPlayedIntro = false;
  List<Map<String, dynamic>> _reminders = [];
  String _selectedFrequency = 'Daily';
  TimeOfDay _selectedTime = const TimeOfDay(hour: 8, minute: 0);

  final List<String> _frequencies = ['Daily', 'Twice daily', 'Three times daily', 'Weekly', 'As needed'];

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
    _loadReminders();
  }

  Future<void> _initializeAndPlay() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    if (mounted && !_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      await Future.delayed(const Duration(milliseconds: 300));
      _voiceService.speak(
        "Hey love, this is Med Reminder - keeps your tablets on time. "
        "Tap the plus button to add your medication. Tell me what you take, when, and how often. "
        "I'll save it on your phone and nudge you when it's time. "
        "Everything stays private - just between us."
      );
    }
  }

  Future<void> _loadReminders() async {
    try {
      final box = await Hive.openBox('health_reminders');
      final stored = box.get('reminders', defaultValue: []);
      setState(() {
        _reminders = List<Map<String, dynamic>>.from(
          (stored as List).map((e) => Map<String, dynamic>.from(e))
        );
      });
    } catch (e) {
      debugPrint('Error loading reminders: $e');
    }
  }

  Future<void> _saveReminders() async {
    try {
      final box = await Hive.openBox('health_reminders');
      await box.put('reminders', _reminders);
    } catch (e) {
      debugPrint('Error saving reminders: $e');
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _scrollController.dispose();
    _medicationController.dispose();
    _dosageController.dispose();
    _notesController.dispose();
    super.dispose();
  }

  void _addReminder() {
    if (_medicationController.text.isEmpty) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Please enter medication name')),
      );
      return;
    }

    final newReminder = {
      'id': DateTime.now().millisecondsSinceEpoch.toString(),
      'medication': _medicationController.text,
      'dosage': _dosageController.text,
      'frequency': _selectedFrequency,
      'time': '${_selectedTime.hour}:${_selectedTime.minute.toString().padLeft(2, '0')}',
      'notes': _notesController.text,
      'active': true,
    };

    setState(() {
      _reminders.add(newReminder);
    });
    _saveReminders();

    _medicationController.clear();
    _dosageController.clear();
    _notesController.clear();
    
    if (_prefs.autoPlayVoice) {
      _voiceService.speak(
        "Reminder set for ${newReminder['medication']}. "
        "I'll remind you ${_selectedFrequency.toLowerCase()}."
      );
    }

    Navigator.pop(context);
  }

  void _deleteReminder(String id) {
    setState(() {
      _reminders.removeWhere((r) => r['id'] == id);
    });
    _saveReminders();
  }

  void _toggleReminder(String id) {
    setState(() {
      final index = _reminders.indexWhere((r) => r['id'] == id);
      if (index != -1) {
        _reminders[index]['active'] = !(_reminders[index]['active'] ?? true);
      }
    });
    _saveReminders();
  }

  void _showAddReminderSheet() {
    showModalBottomSheet(
      context: context,
      isScrollControlled: true,
      backgroundColor: Theme.of(context).cardColor,
      shape: const RoundedRectangleBorder(
        borderRadius: BorderRadius.vertical(top: Radius.circular(20)),
      ),
      builder: (context) => Padding(
        padding: EdgeInsets.only(
          left: 20,
          right: 20,
          top: 20,
          bottom: MediaQuery.of(context).viewInsets.bottom + 20,
        ),
        child: SingleChildScrollView(
          child: Column(
            mainAxisSize: MainAxisSize.min,
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Text(
                'Add Medication Reminder',
                style: Theme.of(context).textTheme.titleLarge?.copyWith(
                  fontWeight: FontWeight.bold,
                ),
              ),
              const SizedBox(height: 20),
              
              TextField(
                controller: _medicationController,
                decoration: InputDecoration(
                  labelText: 'Medication Name',
                  hintText: 'e.g., Blood pressure tablet',
                  border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
                  prefixIcon: const Icon(Icons.medication_outlined),
                  suffixIcon: MicButton(onResult: (t) => _medicationController.text = t),
                ),
              ),
              const SizedBox(height: 16),
              
              TextField(
                controller: _dosageController,
                decoration: InputDecoration(
                  labelText: 'Dosage (optional)',
                  hintText: 'e.g., 1 tablet, 5ml',
                  border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
                  prefixIcon: const Icon(Icons.format_list_numbered),
                  suffixIcon: MicButton(onResult: (t) => _dosageController.text = t),
                ),
              ),
              const SizedBox(height: 16),
              
              DropdownButtonFormField<String>(
                value: _selectedFrequency,
                decoration: InputDecoration(
                  labelText: 'How Often',
                  border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
                  prefixIcon: const Icon(Icons.repeat),
                ),
                items: _frequencies.map((f) => 
                  DropdownMenuItem(value: f, child: Text(f))
                ).toList(),
                onChanged: (v) => setState(() => _selectedFrequency = v!),
              ),
              const SizedBox(height: 16),
              
              ListTile(
                contentPadding: EdgeInsets.zero,
                leading: const Icon(Icons.access_time),
                title: const Text('Reminder Time'),
                trailing: TextButton(
                  onPressed: () async {
                    final time = await showTimePicker(
                      context: context,
                      initialTime: _selectedTime,
                    );
                    if (time != null) {
                      setState(() => _selectedTime = time);
                    }
                  },
                  child: Text(
                    _selectedTime.format(context),
                    style: const TextStyle(fontSize: 18),
                  ),
                ),
              ),
              const SizedBox(height: 16),
              
              TextField(
                controller: _notesController,
                decoration: InputDecoration(
                  labelText: 'Notes (optional)',
                  hintText: 'e.g., Take with food',
                  border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
                  prefixIcon: const Icon(Icons.notes),
                  suffixIcon: MicButton(onResult: (t) => _notesController.text = t),
                ),
                maxLines: 2,
              ),
              const SizedBox(height: 24),
              
              SizedBox(
                width: double.infinity,
                child: ElevatedButton.icon(
                  onPressed: _addReminder,
                  icon: const Icon(Icons.add_alarm),
                  label: const Text('Add Reminder'),
                  style: ElevatedButton.styleFrom(
                    padding: const EdgeInsets.symmetric(vertical: 16),
                    shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
                  ),
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              Padding(
                padding: const EdgeInsets.all(16.0),
                child: Row(
                  children: [
                    IconButton(
                      icon: const Icon(Icons.arrow_back),
                      onPressed: () => context.go('/'),
                    ),
                    const Spacer(),
                    IconButton(
                      icon: const Icon(Icons.volume_up),
                      onPressed: () {
                        if (_reminders.isEmpty) {
                          _voiceService.speak(
                            "You haven't set any reminders yet. "
                            "Tap the plus button to add your first medication reminder."
                          );
                        } else {
                          _voiceService.speak(
                            "You have ${_reminders.length} medication reminders set up. "
                            "Tap any reminder to hear more details."
                          );
                        }
                      },
                    ),
                    IconButton(
                      icon: const Icon(Icons.settings_outlined),
                      onPressed: () => context.go('/settings'),
                    ),
                  ],
                ),
              ),
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 20.0),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(
                          Icons.medication_outlined,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                          size: 28,
                        ),
                        const SizedBox(width: 12),
                        Text(
                          'Med Reminder',
                          style: Theme.of(context).textTheme.headlineMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(
                      'Never miss your medications',
                      style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                  ],
                ),
              ),
              const SizedBox(height: 20),
              Expanded(
                child: _reminders.isEmpty
                  ? _buildEmptyState(isDark)
                  : Stack(
                      children: [
                        _buildRemindersList(isDark),
                        ScrollIndicator(scrollController: _scrollController, bottomPadding: 20),
                      ],
                    ),
              ),
            ],
          ),
        ),
      ),
      floatingActionButton: FloatingActionButton.extended(
        onPressed: _showAddReminderSheet,
        icon: const Icon(Icons.add),
        label: const Text('Add Reminder'),
      ),
    );
  }

  Widget _buildEmptyState(bool isDark) {
    return Center(
      child: Padding(
        padding: const EdgeInsets.all(40),
        child: Column(
          mainAxisAlignment: MainAxisAlignment.center,
          children: [
            Icon(
              Icons.medication_liquid_outlined,
              size: 80,
              color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.5),
            ),
            const SizedBox(height: 24),
            Text(
              'No reminders yet',
              style: Theme.of(context).textTheme.titleLarge,
            ),
            const SizedBox(height: 8),
            Text(
              'Add your medications and I\'ll help you remember to take them',
              textAlign: TextAlign.center,
              style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
              ),
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildRemindersList(bool isDark) {
    return ListView.builder(
      controller: _scrollController,
      padding: const EdgeInsets.symmetric(horizontal: 16),
      itemCount: _reminders.length,
      itemBuilder: (context, index) {
        final reminder = _reminders[index];
        final isActive = reminder['active'] ?? true;
        
        return Dismissible(
          key: Key(reminder['id']),
          direction: DismissDirection.endToStart,
          background: Container(
            alignment: Alignment.centerRight,
            padding: const EdgeInsets.only(right: 20),
            color: Colors.red,
            child: const Icon(Icons.delete, color: Colors.white),
          ),
          onDismissed: (_) => _deleteReminder(reminder['id']),
          child: Card(
            margin: const EdgeInsets.only(bottom: 12),
            child: ListTile(
              leading: Container(
                padding: const EdgeInsets.all(10),
                decoration: BoxDecoration(
                  color: (isActive 
                    ? (isDark ? AppColors.primaryLight : AppColors.primary)
                    : Colors.grey
                  ).withOpacity(0.15),
                  borderRadius: BorderRadius.circular(10),
                ),
                child: Icon(
                  Icons.medication,
                  color: isActive 
                    ? (isDark ? AppColors.primaryLight : AppColors.primary)
                    : Colors.grey,
                ),
              ),
              title: Text(
                reminder['medication'] ?? 'Medication',
                style: TextStyle(
                  fontWeight: FontWeight.bold,
                  decoration: isActive ? null : TextDecoration.lineThrough,
                ),
              ),
              subtitle: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  if (reminder['dosage']?.isNotEmpty ?? false)
                    Text(reminder['dosage']),
                  Text(
                    '${reminder['frequency']} at ${reminder['time']}',
                    style: TextStyle(
                      color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                    ),
                  ),
                ],
              ),
              trailing: Switch(
                value: isActive,
                onChanged: (_) => _toggleReminder(reminder['id']),
              ),
              onTap: () {
                _voiceService.speak(
                  "Reminder for ${reminder['medication']}. "
                  "${reminder['dosage']?.isNotEmpty == true ? 'Take ${reminder['dosage']}.' : ''} "
                  "${reminder['frequency']} at ${reminder['time']}. "
                  "${reminder['notes']?.isNotEmpty == true ? reminder['notes'] : ''}"
                );
              },
            ),
          ),
        );
      },
    );
  }
}
