import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/services/voice_scripts.dart';
import '../../../core/widgets/shared_widgets.dart';

class FormsPage extends StatefulWidget {
  const FormsPage({super.key});

  @override
  State<FormsPage> createState() => _FormsPageState();
}

class _FormsPageState extends State<FormsPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _api = ApiClient();
  final VoiceScripts _scripts = VoiceScripts();
  final _answerController = TextEditingController();
  
  bool _hasPlayedIntro = false;
  bool _isLoading = false;
  String? _selectedFormType;
  Map<String, dynamic>? _currentQuestion;
  Map<String, String> _answers = {};
  String? _error;

  final List<Map<String, String>> _formTypes = [
    {'id': 'centrelink_claim', 'name': 'Centrelink New Claim', 'icon': 'government'},
    {'id': 'centrelink_update', 'name': 'Centrelink Update', 'icon': 'update'},
    {'id': 'housing_application', 'name': 'Housing Application', 'icon': 'home'},
    {'id': 'rental_application', 'name': 'Rental Application', 'icon': 'key'},
    {'id': 'jobseeker', 'name': 'Job Seeker Form', 'icon': 'work'},
    {'id': 'medicare', 'name': 'Medicare/Health Form', 'icon': 'health'},
    {'id': 'school_enrolment', 'name': 'School Form', 'icon': 'school'},
    {'id': 'legal_aid', 'name': 'Legal Aid Form', 'icon': 'gavel'},
    {'id': 'other', 'name': 'Other Form', 'icon': 'document'},
  ];

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    if (mounted && !_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      await Future.delayed(const Duration(milliseconds: 300));
      _voiceService.speak(_scripts.formFillerIntro);
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _answerController.dispose();
    super.dispose();
  }

  IconData _getIconForType(String iconType) {
    switch (iconType) {
      case 'government': return Icons.account_balance;
      case 'update': return Icons.sync;
      case 'home': return Icons.home_outlined;
      case 'key': return Icons.vpn_key_outlined;
      case 'work': return Icons.work_outline;
      case 'health': return Icons.local_hospital_outlined;
      case 'school': return Icons.school_outlined;
      case 'gavel': return Icons.gavel;
      default: return Icons.description_outlined;
    }
  }

  Future<void> _selectFormType(String formType) async {
    setState(() {
      _selectedFormType = formType;
      _isLoading = true;
      _error = null;
    });
    
    await _getNextQuestion();
  }

  Future<void> _getNextQuestion() async {
    try {
      final result = await _api.post('/api/forms/fill', {
        'formType': _selectedFormType,
        'answers': _answers,
        'context': _answerController.text,
      });
      
      setState(() {
        _currentQuestion = result;
        _isLoading = false;
        _answerController.clear();
      });
      
      if (_prefs.autoPlayVoice && result['question'] != null) {
        _voiceService.speak(result['question']);
      }
    } catch (e) {
      setState(() {
        _error = 'Having trouble getting the next question. Try again?';
        _isLoading = false;
      });
    }
  }

  Future<void> _submitAnswer() async {
    if (_answerController.text.isEmpty) return;
    
    final currentSection = _currentQuestion?['currentSection'] ?? 'answer';
    _answers[currentSection] = _answerController.text;
    
    setState(() => _isLoading = true);
    await _getNextQuestion();
  }

  void _resetForm() {
    setState(() {
      _selectedFormType = null;
      _currentQuestion = null;
      _answers = {};
      _answerController.clear();
    });
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              Padding(
                padding: const EdgeInsets.all(16.0),
                child: Row(
                  children: [
                    IconButton(
                      icon: const Icon(Icons.arrow_back),
                      onPressed: () {
                        if (_selectedFormType != null) {
                          _resetForm();
                        } else {
                          context.go('/');
                        }
                      },
                    ),
                    const Spacer(),
                    IconButton(
                      icon: const Icon(Icons.volume_up),
                      onPressed: () {
                        if (_currentQuestion != null) {
                          _voiceService.speak(_currentQuestion!['question'] ?? 'Pick a form type to get started');
                        } else {
                          _voiceService.speak("Pick which form you need help filling out");
                        }
                      },
                    ),
                    IconButton(
                      icon: const Icon(Icons.settings_outlined),
                      onPressed: () => context.go('/settings'),
                    ),
                  ],
                ),
              ),
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 20.0),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(
                          Icons.edit_document,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                          size: 28,
                        ),
                        const SizedBox(width: 12),
                        Text(
                          'Form Filler',
                          style: Theme.of(context).textTheme.headlineMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(
                      _selectedFormType == null 
                        ? 'Pick a form to get started'
                        : 'Answer the questions below',
                      style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                  ],
                ),
              ),
              const SizedBox(height: 20),
              Expanded(
                child: _selectedFormType == null
                  ? _buildFormTypeSelector(isDark)
                  : _buildQuestionSection(isDark),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildFormTypeSelector(bool isDark) {
    return ListView.builder(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      itemCount: _formTypes.length,
      itemBuilder: (context, index) {
        final form = _formTypes[index];
        return Card(
          margin: const EdgeInsets.only(bottom: 12),
          child: ListTile(
            leading: Container(
              padding: const EdgeInsets.all(10),
              decoration: BoxDecoration(
                color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.15),
                borderRadius: BorderRadius.circular(10),
              ),
              child: Icon(
                _getIconForType(form['icon']!),
                color: isDark ? AppColors.primaryLight : AppColors.primary,
              ),
            ),
            title: Text(
              form['name']!,
              style: const TextStyle(fontWeight: FontWeight.w600),
            ),
            trailing: const Icon(Icons.chevron_right),
            onTap: () => _selectFormType(form['id']!),
          ),
        );
      },
    );
  }

  Widget _buildQuestionSection(bool isDark) {
    if (_isLoading) {
      return const Center(child: CircularProgressIndicator());
    }

    if (_error != null) {
      return Center(
        child: Padding(
          padding: const EdgeInsets.all(20),
          child: Column(
            mainAxisAlignment: MainAxisAlignment.center,
            children: [
              Icon(Icons.error_outline, size: 48, color: Colors.red[400]),
              const SizedBox(height: 16),
              Text(_error!, textAlign: TextAlign.center),
              const SizedBox(height: 16),
              ElevatedButton(
                onPressed: () => _getNextQuestion(),
                child: const Text('Try Again'),
              ),
            ],
          ),
        ),
      );
    }

    final question = _currentQuestion;
    if (question == null) return const SizedBox();

    final progress = (question['progress'] as num?)?.toDouble() ?? 0;

    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          if (progress > 0) ...[
            LinearProgressIndicator(
              value: progress / 100,
              backgroundColor: Colors.grey.withOpacity(0.3),
            ),
            const SizedBox(height: 8),
            Text(
              '${progress.toInt()}% complete',
              style: Theme.of(context).textTheme.bodySmall,
            ),
            const SizedBox(height: 16),
          ],
          
          Card(
            child: Padding(
              padding: const EdgeInsets.all(16),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  if (question['currentSection'] != null) ...[
                    Text(
                      question['currentSection'],
                      style: Theme.of(context).textTheme.labelMedium?.copyWith(
                        color: isDark ? AppColors.primaryLight : AppColors.primary,
                      ),
                    ),
                    const SizedBox(height: 8),
                  ],
                  Row(
                    children: [
                      Expanded(
                        child: Text(
                          question['question'] ?? 'Tell me about this section',
                          style: Theme.of(context).textTheme.titleMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ),
                      IconButton(
                        icon: const Icon(Icons.volume_up, size: 20),
                        onPressed: () => _voiceService.speak(question['question'] ?? ''),
                      ),
                    ],
                  ),
                  if (question['explanation'] != null) ...[
                    const SizedBox(height: 8),
                    Text(
                      question['explanation'],
                      style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                  ],
                ],
              ),
            ),
          ),
          const SizedBox(height: 16),
          
          TextField(
            controller: _answerController,
            decoration: InputDecoration(
              hintText: 'Type your answer here...',
              border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
              suffixIcon: MicButton(
                onResult: (text) {
                  _answerController.text = text;
                },
              ),
            ),
            maxLines: 3,
          ),
          const SizedBox(height: 16),
          
          if (question['tips'] != null && (question['tips'] as List).isNotEmpty) ...[
            ...((question['tips'] as List).map((tip) => Padding(
              padding: const EdgeInsets.only(bottom: 8),
              child: Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Icon(Icons.lightbulb_outline, size: 18, color: Colors.amber[600]),
                  const SizedBox(width: 8),
                  Expanded(
                    child: Text(
                      tip.toString(),
                      style: Theme.of(context).textTheme.bodySmall,
                    ),
                  ),
                ],
              ),
            ))),
            const SizedBox(height: 16),
          ],
          
          if (question['documentsNeeded'] != null && (question['documentsNeeded'] as List).isNotEmpty) ...[
            Card(
              color: Colors.blue.withOpacity(0.1),
              child: Padding(
                padding: const EdgeInsets.all(12),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.folder_outlined, color: Colors.blue[400]),
                        const SizedBox(width: 8),
                        Text(
                          'Documents You Might Need',
                          style: Theme.of(context).textTheme.titleSmall?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    ...((question['documentsNeeded'] as List).map((doc) => Padding(
                      padding: const EdgeInsets.only(bottom: 4),
                      child: Text('• ${doc.toString()}'),
                    ))),
                  ],
                ),
              ),
            ),
            const SizedBox(height: 16),
          ],
          
          SizedBox(
            width: double.infinity,
            child: ElevatedButton.icon(
              onPressed: _submitAnswer,
              icon: const Icon(Icons.arrow_forward),
              label: const Text('Next Question'),
              style: ElevatedButton.styleFrom(
                padding: const EdgeInsets.symmetric(vertical: 16),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
            ),
          ),
          const SizedBox(height: 12),
          SizedBox(
            width: double.infinity,
            child: TextButton(
              onPressed: _resetForm,
              child: const Text('Start Over'),
            ),
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }
}
