import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:url_launcher/url_launcher.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/widgets/shared_widgets.dart';

class FoodPage extends StatefulWidget {
  const FoodPage({super.key});

  @override
  State<FoodPage> createState() => _FoodPageState();
}

class _FoodPageState extends State<FoodPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _apiClient = ApiClient();
  
  bool _isLoading = false;
  String _selectedState = 'NSW';
  List<_EmergencyService> _services = [];
  String? _aiAdvice;

  static const List<String> _states = [
    'NSW', 'VIC', 'QLD', 'SA', 'WA', 'TAS', 'NT', 'ACT'
  ];

  static final List<_EmergencyService> _offlineServices = [
    _EmergencyService(
      name: '13YARN',
      description: 'Aboriginal & Torres Strait Islander crisis support',
      phone: '13 92 76',
      isAboriginal: true,
      category: 'crisis',
    ),
    _EmergencyService(
      name: 'Foodbank Australia',
      description: 'Emergency food relief - find your nearest location',
      phone: '1800 032 633',
      website: 'https://www.foodbank.org.au/find-food/',
      isAboriginal: false,
      category: 'food',
    ),
    _EmergencyService(
      name: 'OzHarvest',
      description: 'Free rescued food for those in need',
      phone: '1800 108 006',
      website: 'https://www.ozharvest.org/',
      isAboriginal: false,
      category: 'food',
    ),
    _EmergencyService(
      name: 'Salvation Army',
      description: 'Emergency relief, food vouchers, and support',
      phone: '13 72 58',
      website: 'https://www.salvationarmy.org.au/',
      isAboriginal: false,
      category: 'food',
    ),
    _EmergencyService(
      name: 'St Vincent de Paul',
      description: 'Food, clothing, furniture, and bill assistance',
      phone: '13 18 12',
      website: 'https://www.vinnies.org.au/',
      isAboriginal: false,
      category: 'essentials',
    ),
    _EmergencyService(
      name: 'Centrelink Crisis Payment',
      description: 'One-off payment for extreme circumstances',
      phone: '132 850',
      isAboriginal: false,
      category: 'financial',
    ),
    _EmergencyService(
      name: 'Aboriginal Hostels Limited',
      description: 'Emergency accommodation for First Nations people',
      phone: '1800 355 544',
      website: 'https://www.ahl.gov.au/',
      isAboriginal: true,
      category: 'housing',
    ),
    _EmergencyService(
      name: 'No Interest Loan Scheme (NILS)',
      description: 'Interest-free loans for essential items',
      phone: '13 64 57',
      website: 'https://nils.com.au/',
      isAboriginal: false,
      category: 'financial',
    ),
  ];

  @override
  void initState() {
    super.initState();
    _initVoice();
    _services = List.from(_offlineServices);
    _sortServicesByPriority();
  }

  Future<void> _initVoice() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted && _prefs.autoPlayVoice) {
      await Future.delayed(const Duration(milliseconds: 500));
      _voiceService.speak(
        "This page helps you find emergency food and essentials. "
        "Aboriginal services are listed first. "
        "Tap any service to call them directly."
      );
    }
  }

  void _sortServicesByPriority() {
    _services.sort((a, b) {
      if (a.isAboriginal && !b.isAboriginal) return -1;
      if (!a.isAboriginal && b.isAboriginal) return 1;
      if (a.category == 'food' && b.category != 'food') return -1;
      if (a.category != 'food' && b.category == 'food') return 1;
      return 0;
    });
  }

  @override
  void dispose() {
    _voiceService.stop();
    super.dispose();
  }

  Future<void> _getPersonalizedAdvice() async {
    setState(() => _isLoading = true);
    
    try {
      final data = await _apiClient.post('/api/food/advice', {
        'state': _selectedState,
      });
      
      setState(() {
        _aiAdvice = data['advice'] ?? 'Check with local services for help.';
        _isLoading = false;
      });
      
      if (_prefs.autoPlayVoice && _aiAdvice != null) {
        _voiceService.speak(_aiAdvice!);
      }
    } catch (e) {
      setState(() {
        _aiAdvice = "I couldn't get online advice right now, but the services below can still help. Start with 13YARN if you need to talk to someone who understands.";
        _isLoading = false;
      });
    }
  }

  Future<void> _callService(String phone) async {
    final cleaned = phone.replaceAll(' ', '');
    final uri = Uri.parse('tel:$cleaned');
    
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Calling now");
    }
    
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri);
    }
  }

  Future<void> _openWebsite(String url) async {
    final uri = Uri.parse(url);
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri, mode: LaunchMode.externalApplication);
    }
  }

  void _goBack() {
    _voiceService.stop();
    context.go('/');
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              _buildHeader(isDark),
              Expanded(
                child: ScrollableWithIndicator(
                  padding: const EdgeInsets.all(16),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      _buildUrgentBanner(isDark),
                      const SizedBox(height: 20),
                      _buildStateSelector(isDark),
                      const SizedBox(height: 16),
                      _buildGetAdviceButton(isDark),
                      if (_aiAdvice != null) ...[
                        const SizedBox(height: 16),
                        _buildAdviceCard(isDark),
                      ],
                      const SizedBox(height: 24),
                      _buildServicesSection(isDark),
                    ],
                  ),
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildHeader(bool isDark) {
    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 8),
      child: Row(
        children: [
          IconButton(
            icon: const Icon(Icons.arrow_back),
            onPressed: _goBack,
            tooltip: 'Back to home',
          ),
          const SizedBox(width: 8),
          Expanded(
            child: Text(
              "Food & Essentials",
              style: Theme.of(context).textTheme.titleLarge,
            ),
          ),
          IconButton(
            icon: const Icon(Icons.volume_up),
            onPressed: () {
              _voiceService.speak(
                "This page lists emergency food and essential services. "
                "Aboriginal services are shown first. "
                "Tap the phone icon next to any service to call them."
              );
            },
            tooltip: 'Read aloud',
          ),
        ],
      ),
    );
  }

  Widget _buildUrgentBanner(bool isDark) {
    return Container(
      padding: const EdgeInsets.all(16),
      decoration: BoxDecoration(
        gradient: LinearGradient(
          colors: [
            AppColors.ochre.withOpacity(0.3),
            AppColors.terracotta.withOpacity(0.2),
          ],
        ),
        borderRadius: BorderRadius.circular(12),
        border: Border.all(color: AppColors.ochre.withOpacity(0.5)),
      ),
      child: Row(
        children: [
          Icon(Icons.info_outline, color: AppColors.ochre, size: 28),
          const SizedBox(width: 12),
          Expanded(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(
                  "Need help right now?",
                  style: Theme.of(context).textTheme.titleMedium?.copyWith(
                    fontWeight: FontWeight.bold,
                  ),
                ),
                const SizedBox(height: 4),
                Text(
                  "These services are free and confidential. You don't need ID or a referral.",
                  style: Theme.of(context).textTheme.bodyMedium,
                ),
              ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildStateSelector(bool isDark) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          "Your state/territory",
          style: Theme.of(context).textTheme.labelLarge,
        ),
        const SizedBox(height: 8),
        Wrap(
          spacing: 8,
          runSpacing: 8,
          children: _states.map((state) {
            final isSelected = state == _selectedState;
            return ChoiceChip(
              label: Text(state),
              selected: isSelected,
              onSelected: (selected) {
                if (selected) {
                  setState(() => _selectedState = state);
                }
              },
            );
          }).toList(),
        ),
      ],
    );
  }

  Widget _buildGetAdviceButton(bool isDark) {
    return SizedBox(
      width: double.infinity,
      child: ElevatedButton.icon(
        onPressed: _isLoading ? null : _getPersonalizedAdvice,
        icon: _isLoading 
            ? SizedBox(
                width: 20,
                height: 20,
                child: CircularProgressIndicator(
                  strokeWidth: 2,
                  color: Colors.white,
                ),
              )
            : const Icon(Icons.tips_and_updates),
        label: Text(_isLoading ? "Getting advice..." : "Get personalised tips"),
      ),
    );
  }

  Widget _buildAdviceCard(bool isDark) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                Icon(Icons.lightbulb_outline, color: AppColors.primary),
                const SizedBox(width: 8),
                Text(
                  "Tips for you",
                  style: Theme.of(context).textTheme.titleMedium?.copyWith(
                    fontWeight: FontWeight.w600,
                  ),
                ),
                const Spacer(),
                IconButton(
                  icon: const Icon(Icons.volume_up, size: 20),
                  onPressed: () {
                    if (_aiAdvice != null) {
                      _voiceService.speak(_aiAdvice!);
                    }
                  },
                  tooltip: 'Read aloud',
                ),
              ],
            ),
            const SizedBox(height: 8),
            Text(
              _aiAdvice ?? '',
              style: Theme.of(context).textTheme.bodyMedium,
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildServicesSection(bool isDark) {
    final foodServices = _services.where((s) => s.category == 'food' || s.category == 'crisis').toList();
    final otherServices = _services.where((s) => s.category != 'food' && s.category != 'crisis').toList();
    
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          "Emergency Services",
          style: Theme.of(context).textTheme.titleMedium?.copyWith(
            fontWeight: FontWeight.w600,
          ),
        ),
        const SizedBox(height: 4),
        Text(
          "Aboriginal services listed first",
          style: Theme.of(context).textTheme.bodySmall?.copyWith(
            color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
          ),
        ),
        const SizedBox(height: 12),
        ...foodServices.map((service) => _buildServiceCard(service, isDark)),
        const SizedBox(height: 20),
        Text(
          "Other Support",
          style: Theme.of(context).textTheme.titleMedium?.copyWith(
            fontWeight: FontWeight.w600,
          ),
        ),
        const SizedBox(height: 12),
        ...otherServices.map((service) => _buildServiceCard(service, isDark)),
      ],
    );
  }

  Widget _buildServiceCard(_EmergencyService service, bool isDark) {
    return Card(
      margin: const EdgeInsets.only(bottom: 12),
      child: InkWell(
        onTap: () => _callService(service.phone),
        borderRadius: BorderRadius.circular(12),
        child: Padding(
          padding: const EdgeInsets.all(16),
          child: Row(
            children: [
              Container(
                width: 48,
                height: 48,
                decoration: BoxDecoration(
                  color: service.isAboriginal 
                      ? AppColors.ochre.withOpacity(0.2)
                      : AppColors.primary.withOpacity(0.1),
                  borderRadius: BorderRadius.circular(12),
                ),
                child: Icon(
                  _getCategoryIcon(service.category),
                  color: service.isAboriginal ? AppColors.ochre : AppColors.primary,
                ),
              ),
              const SizedBox(width: 16),
              Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Expanded(
                          child: Text(
                            service.name,
                            style: Theme.of(context).textTheme.titleMedium?.copyWith(
                              fontWeight: FontWeight.w600,
                            ),
                          ),
                        ),
                        if (service.isAboriginal)
                          Container(
                            padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 2),
                            decoration: BoxDecoration(
                              color: AppColors.ochre.withOpacity(0.2),
                              borderRadius: BorderRadius.circular(8),
                            ),
                            child: Text(
                              "Mob-run",
                              style: Theme.of(context).textTheme.labelSmall?.copyWith(
                                color: AppColors.ochre,
                                fontWeight: FontWeight.w600,
                              ),
                            ),
                          ),
                      ],
                    ),
                    const SizedBox(height: 4),
                    Text(
                      service.description,
                      style: Theme.of(context).textTheme.bodySmall?.copyWith(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                    const SizedBox(height: 8),
                    Row(
                      children: [
                        Icon(Icons.phone, size: 16, color: AppColors.primary),
                        const SizedBox(width: 4),
                        Text(
                          service.phone,
                          style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                            color: AppColors.primary,
                            fontWeight: FontWeight.w600,
                          ),
                        ),
                        if (service.website != null) ...[
                          const SizedBox(width: 16),
                          GestureDetector(
                            onTap: () => _openWebsite(service.website!),
                            child: Row(
                              children: [
                                Icon(Icons.open_in_new, size: 16, color: AppColors.primary),
                                const SizedBox(width: 4),
                                Text(
                                  "Website",
                                  style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                                    color: AppColors.primary,
                                  ),
                                ),
                              ],
                            ),
                          ),
                        ],
                      ],
                    ),
                  ],
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  IconData _getCategoryIcon(String category) {
    switch (category) {
      case 'food':
        return Icons.restaurant;
      case 'crisis':
        return Icons.favorite;
      case 'housing':
        return Icons.home;
      case 'essentials':
        return Icons.shopping_bag;
      case 'financial':
        return Icons.account_balance_wallet;
      default:
        return Icons.help_outline;
    }
  }
}

class _EmergencyService {
  final String name;
  final String description;
  final String phone;
  final String? website;
  final bool isAboriginal;
  final String category;

  _EmergencyService({
    required this.name,
    required this.description,
    required this.phone,
    this.website,
    required this.isAboriginal,
    required this.category,
  });
}
