import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:url_launcher/url_launcher.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/voice_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class FamilySafetyPage extends StatefulWidget {
  const FamilySafetyPage({super.key});

  @override
  State<FamilySafetyPage> createState() => _FamilySafetyPageState();
}

class _FamilySafetyPageState extends State<FamilySafetyPage> {
  bool _consentGiven = false;
  bool _voiceEnabled = false;

  final String _introText = 
      "This is a safe space for information about family safety. "
      "Everything stays on your device. No one can see what you look at here. "
      "There's a quick exit button if you need to leave fast.";

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () => context.go('/'),
          tooltip: 'Back to home',
        ),
        title: const Text('Family Safety'),
        actions: [
          IconButton(
            icon: Icon(
              _voiceEnabled ? Icons.volume_up : Icons.volume_off,
              color: _voiceEnabled ? AppColors.primary : Colors.grey,
            ),
            onPressed: () => setState(() => _voiceEnabled = !_voiceEnabled),
            tooltip: _voiceEnabled ? 'Disable voice' : 'Enable voice',
          ),
          Container(
            margin: const EdgeInsets.only(right: 8),
            child: ElevatedButton(
              onPressed: _quickExit,
              style: ElevatedButton.styleFrom(
                backgroundColor: Colors.grey[800],
                foregroundColor: Colors.white,
              ),
              child: const Text('Quick Exit'),
            ),
          ),
        ],
      ),
      body: ScrollableWithIndicator(
        padding: const EdgeInsets.all(16),
        child: !_consentGiven ? _buildConsentScreen() : _buildContent(),
      ),
    );
  }

  void _quickExit() async {
    final uri = Uri.parse('https://www.google.com');
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri, mode: LaunchMode.externalApplication);
    }
  }

  Widget _buildConsentScreen() {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.stretch,
      children: [
        const Icon(Icons.shield, size: 64, color: AppColors.primary),
        const SizedBox(height: 24),
        const Text(
          'Staying Safe',
          style: TextStyle(fontSize: 24, fontWeight: FontWeight.bold),
          textAlign: TextAlign.center,
        ),
        const SizedBox(height: 16),
        Container(
          padding: const EdgeInsets.all(16),
          decoration: BoxDecoration(
            color: Colors.red.withOpacity(0.1),
            borderRadius: BorderRadius.circular(12),
            border: Border.all(color: Colors.red.withOpacity(0.3)),
          ),
          child: Column(
            children: [
              const Icon(Icons.warning_amber, color: Colors.red),
              const SizedBox(height: 8),
              const Text(
                'Shared Device Warning',
                style: TextStyle(fontWeight: FontWeight.bold, color: Colors.red),
              ),
              const SizedBox(height: 8),
              const Text(
                'If you share this device with someone who might hurt you, '
                'clear your history after visiting this page or use the Quick Exit button.',
                textAlign: TextAlign.center,
              ),
            ],
          ),
        ),
        const SizedBox(height: 16),
        Container(
          padding: const EdgeInsets.all(16),
          decoration: BoxDecoration(
            color: AppColors.primary.withOpacity(0.1),
            borderRadius: BorderRadius.circular(12),
          ),
          child: Column(
            children: [
              const Icon(Icons.volume_up, color: AppColors.primary),
              const SizedBox(height: 8),
              const Text(
                'Voice is OFF by default for your safety.',
                style: TextStyle(fontWeight: FontWeight.w500),
                textAlign: TextAlign.center,
              ),
              const SizedBox(height: 4),
              Text(
                'You can turn it on using the speaker icon in the top bar.',
                style: TextStyle(color: Colors.grey[600], fontSize: 12),
                textAlign: TextAlign.center,
              ),
            ],
          ),
        ),
        const SizedBox(height: 24),
        ElevatedButton(
          onPressed: () => setState(() => _consentGiven = true),
          style: ElevatedButton.styleFrom(
            padding: const EdgeInsets.all(16),
          ),
          child: const Text('I understand, let me in'),
        ),
      ],
    );
  }

  Widget _buildContent() {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.stretch,
      children: [
        _buildCrisisBar(),
        const SizedBox(height: 16),
        _buildSection(
          icon: Icons.favorite,
          title: "You're Not Alone",
          voiceText: "Many people go through this. It's not your fault. There is help available.",
          children: [
            _buildInfoCard(
              "It's not your fault",
              "No one deserves to be hurt. What's happening is not okay.",
            ),
            _buildInfoCard(
              'Help is available',
              'There are people who understand and can help. Mob-run services know our ways.',
            ),
          ],
        ),
        const SizedBox(height: 16),
        _buildSection(
          icon: Icons.security,
          title: 'Safety Planning',
          voiceText: "Think about safe places you could go. Keep important documents somewhere safe. "
              "Have a bag ready with essentials if you need to leave quickly.",
          children: [
            _buildSafetyTip(Icons.location_on, 'Safe places', 'Think about where you could go in an emergency'),
            _buildSafetyTip(Icons.folder, 'Important papers', 'Keep copies somewhere safe'),
            _buildSafetyTip(Icons.phone, 'Emergency contacts', 'Have trusted numbers ready'),
            _buildSafetyTip(Icons.luggage, 'Go bag', 'Essentials ready if you need to leave fast'),
          ],
        ),
        const SizedBox(height: 16),
        _buildSection(
          icon: Icons.phone,
          title: 'Get Help',
          voiceText: "These services are free and confidential. They can help you make a plan.",
          children: [
            _buildServiceCard(
              '1800RESPECT',
              '1800 737 732',
              '24/7 domestic violence support',
              isEmergency: true,
            ),
            _buildServiceCard(
              '13YARN',
              '13 92 76',
              '24/7 crisis support by Aboriginal people',
              isMobRun: true,
            ),
            _buildServiceCard(
              "Women's Legal Service",
              '1800 639 784',
              'Free legal help',
            ),
          ],
        ),
      ],
    );
  }

  Widget _buildCrisisBar() {
    return Container(
      padding: const EdgeInsets.all(16),
      decoration: BoxDecoration(
        color: Colors.red.withOpacity(0.1),
        borderRadius: BorderRadius.circular(12),
        border: Border.all(color: Colors.red.withOpacity(0.3)),
      ),
      child: Column(
        children: [
          Row(
            children: [
              const Icon(Icons.favorite, color: Colors.red),
              const SizedBox(width: 8),
              const Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      'Need help right now?',
                      style: TextStyle(fontWeight: FontWeight.bold, color: Colors.red),
                    ),
                    Text(
                      'These lines are confidential and free',
                      style: TextStyle(fontSize: 12),
                    ),
                  ],
                ),
              ),
            ],
          ),
          const SizedBox(height: 12),
          Row(
            children: [
              Expanded(
                child: ElevatedButton.icon(
                  onPressed: () => _makeCall('1800737732'),
                  style: ElevatedButton.styleFrom(
                    backgroundColor: Colors.red,
                    foregroundColor: Colors.white,
                  ),
                  icon: const Icon(Icons.phone, size: 16),
                  label: const Text('1800RESPECT'),
                ),
              ),
              const SizedBox(width: 8),
              Expanded(
                child: OutlinedButton.icon(
                  onPressed: () => _makeCall('139276'),
                  style: OutlinedButton.styleFrom(
                    side: const BorderSide(color: Colors.red),
                    foregroundColor: Colors.red,
                  ),
                  icon: const Icon(Icons.phone, size: 16),
                  label: const Text('13YARN'),
                ),
              ),
            ],
          ),
        ],
      ),
    );
  }

  Widget _buildSection({
    required IconData icon,
    required String title,
    required String voiceText,
    required List<Widget> children,
  }) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                Icon(icon, color: AppColors.primary),
                const SizedBox(width: 8),
                Expanded(
                  child: Text(
                    title,
                    style: const TextStyle(
                      fontSize: 18,
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                ),
                if (_voiceEnabled) ListenButton(text: voiceText, size: 36),
              ],
            ),
            const SizedBox(height: 12),
            ...children,
          ],
        ),
      ),
    );
  }

  Widget _buildInfoCard(String title, String description) {
    return Container(
      margin: const EdgeInsets.only(bottom: 8),
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        color: Colors.grey.withOpacity(0.1),
        borderRadius: BorderRadius.circular(8),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(title, style: const TextStyle(fontWeight: FontWeight.w600)),
          const SizedBox(height: 4),
          Text(description, style: TextStyle(color: Colors.grey[600])),
        ],
      ),
    );
  }

  Widget _buildSafetyTip(IconData icon, String title, String description) {
    return Container(
      margin: const EdgeInsets.only(bottom: 8),
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        border: Border.all(color: Colors.grey.withOpacity(0.3)),
        borderRadius: BorderRadius.circular(8),
      ),
      child: Row(
        children: [
          Icon(icon, color: AppColors.primary, size: 24),
          const SizedBox(width: 12),
          Expanded(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(title, style: const TextStyle(fontWeight: FontWeight.w600)),
                Text(description, style: TextStyle(fontSize: 12, color: Colors.grey[600])),
              ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildServiceCard(
    String name,
    String phone,
    String description, {
    bool isMobRun = false,
    bool isEmergency = false,
  }) {
    return Container(
      margin: const EdgeInsets.only(bottom: 8),
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        color: isEmergency
            ? Colors.red.withOpacity(0.05)
            : isMobRun
                ? AppColors.primary.withOpacity(0.05)
                : null,
        border: Border.all(
          color: isEmergency
              ? Colors.red.withOpacity(0.3)
              : isMobRun
                  ? AppColors.primary.withOpacity(0.2)
                  : Colors.grey.withOpacity(0.3),
        ),
        borderRadius: BorderRadius.circular(8),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            children: [
              Expanded(
                child: Text(name, style: const TextStyle(fontWeight: FontWeight.w600)),
              ),
              if (isMobRun)
                Container(
                  padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 2),
                  decoration: BoxDecoration(
                    color: AppColors.primary.withOpacity(0.1),
                    borderRadius: BorderRadius.circular(12),
                  ),
                  child: const Text(
                    'Mob-run',
                    style: TextStyle(fontSize: 12, color: AppColors.primary),
                  ),
                ),
              if (isEmergency)
                Container(
                  padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 2),
                  decoration: BoxDecoration(
                    color: Colors.red.withOpacity(0.1),
                    borderRadius: BorderRadius.circular(12),
                  ),
                  child: const Text(
                    '24/7',
                    style: TextStyle(fontSize: 12, color: Colors.red),
                  ),
                ),
            ],
          ),
          const SizedBox(height: 4),
          Text(description, style: TextStyle(fontSize: 12, color: Colors.grey[600])),
          const SizedBox(height: 8),
          OutlinedButton.icon(
            onPressed: () => _makeCall(phone),
            icon: const Icon(Icons.phone, size: 16),
            label: Text(phone),
          ),
        ],
      ),
    );
  }

  Future<void> _makeCall(String number) async {
    final cleanNumber = number.replaceAll(' ', '');
    final uri = Uri.parse('tel:$cleanNumber');
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri);
    }
  }
}
