import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/services/voice_scripts.dart';
import '../../../core/widgets/shared_widgets.dart';

class DocumentsPage extends StatefulWidget {
  const DocumentsPage({super.key});

  @override
  State<DocumentsPage> createState() => _DocumentsPageState();
}

class _DocumentsPageState extends State<DocumentsPage> {
  final TextEditingController _documentController = TextEditingController();
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _apiClient = ApiClient();
  final VoiceScripts _scripts = VoiceScripts();
  
  bool _isLoading = false;
  bool _isListening = false;
  bool _hasPlayedIntro = false;
  Map<String, dynamic>? _analysis;

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    debugPrint('DocumentsPage: Initializing voice service...');
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playIntro();
    }
  }

  @override
  void dispose() {
    _documentController.dispose();
    _voiceService.stop();
    super.dispose();
  }

  void _playIntro() {
    debugPrint('DocumentsPage: _playIntro called, autoPlay=${_prefs.autoPlayVoice}');
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      _voiceService.speak(_scripts.documentHelperIntro);
    }
  }

  Future<void> _analyzeDocument() async {
    if (_documentController.text.trim().isEmpty) {
      if (_prefs.autoPlayVoice) {
        _voiceService.speak("You haven't added any text yet. Paste your letter or tap the mic to read it to me.");
      }
      return;
    }

    _voiceService.stop();
    setState(() => _isLoading = true);

    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Let me read through this for you...");
    }

    try {
      final result = await _apiClient.post('/api/documents/analyze', {
        'content': _documentController.text,
      });

      setState(() {
        _analysis = result;
        _isLoading = false;
      });

      if (_analysis != null && _analysis!['summary'] != null) {
        await _voiceService.speak(
          "Okay, here's what this is about. " + _analysis!['summary']
        );
        
        if (_analysis!['actionItems'] != null && (_analysis!['actionItems'] as List).isNotEmpty) {
          await Future.delayed(const Duration(milliseconds: 500));
          _voiceService.speak(
            "There are ${(_analysis!['actionItems'] as List).length} things you might need to do. Tap the speaker icon next to each one to hear it."
          );
        }
      }
    } catch (e) {
      setState(() => _isLoading = false);
      if (_prefs.autoPlayVoice) {
        _voiceService.speak("Sorry, something went wrong. Want to try again?");
      }
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(content: Text('Something went wrong. Try again?')),
        );
      }
    }
  }

  Future<void> _toggleListening() async {
    _voiceService.stop();
    
    if (_isListening) {
      await _voiceService.stopListening();
      setState(() => _isListening = false);
    } else {
      setState(() => _isListening = true);
      
      await _voiceService.speakThenListen(
        prompt: "I'm listening. Read out your letter or tell me what it says.",
        onResult: (text) {
          setState(() {
            _isListening = false;
            _documentController.text = _documentController.text.isEmpty 
                ? text 
                : '${_documentController.text} $text';
          });
          if (_prefs.autoPlayVoice) {
            _voiceService.speak("Got it. Tap the mic to add more, or tap Explain This when you're ready.");
          }
        },
        onError: (error) {
          setState(() => _isListening = false);
          if (_prefs.autoPlayVoice) {
            _voiceService.speak("I couldn't hear that clearly. Tap the mic to try again.");
          }
        },
      );
    }
  }

  void _speakActionItem(String item) {
    _voiceService.stop();
    _voiceService.speak(item);
  }

  void _clearAndStartOver() {
    _voiceService.stop();
    setState(() {
      _documentController.clear();
      _analysis = null;
    });
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Starting fresh. Paste a new letter or tap the mic to read it out.");
    }
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () {
            _voiceService.stop();
            context.go('/');
          },
        ),
        title: const Text('Explain This'),
        actions: [
          IconButton(
            icon: const Icon(Icons.volume_up),
            tooltip: 'Read page aloud',
            onPressed: () {
              _voiceService.speak(
                _analysis != null
                    ? "I've explained your letter. Scroll down to see the summary and action items. Tap any speaker icon to hear it read aloud."
                    : "Paste your letter in the box, or tap the mic to read it to me. Then tap Explain This."
              );
            },
          ),
          const SettingsButton(),
        ],
      ),
      body: CulturalBackground(
        child: ScrollableWithIndicator(
          padding: const EdgeInsets.all(16),
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Text(
                'Got a letter or message?',
                style: Theme.of(context).textTheme.headlineSmall,
              ),
              const SizedBox(height: 8),
              Text(
                'Paste it here, tap the mic to read it out, or take a photo of the letter with your phone and upload it. I\'ll break it down for you.',
                style: Theme.of(context).textTheme.bodyMedium,
              ),
              const SizedBox(height: 12),
              ExpansionTile(
                tilePadding: EdgeInsets.zero,
                title: Row(
                  children: [
                    Icon(Icons.camera_alt, 
                      size: 20, 
                      color: isDark ? AppColors.primaryLight : AppColors.primary,
                    ),
                    const SizedBox(width: 8),
                    Flexible(
                      child: Text(
                        'Need help getting text from a letter?',
                        style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                          fontWeight: FontWeight.w500,
                        ),
                      ),
                    ),
                  ],
                ),
                onExpansionChanged: (expanded) {
                  if (expanded && _prefs.autoPlayVoice) {
                    _voiceService.speak(
                      "No worries, here's how to get text from a letter into the app. "
                      "First, take a photo of your letter. Then open the photo and your phone "
                      "should recognise the words. Tap and hold on the text, then copy it. "
                      "Come back here and paste it in the box."
                    );
                  }
                },
                children: [
                  Card(
                    child: Padding(
                      padding: const EdgeInsets.all(16),
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Text(
                            'Easy steps to photograph your letter:',
                            style: Theme.of(context).textTheme.titleSmall?.copyWith(
                              fontWeight: FontWeight.bold,
                            ),
                          ),
                          const SizedBox(height: 16),
                          _buildNumberedStep(1, 'Take a photo', 
                            'Open your camera and take a clear photo of your letter. '
                            'Good light helps - near a window is great!', 
                            Icons.camera_alt),
                          _buildNumberedStep(2, 'Open the photo', 
                            'Go to your Photos app and tap on the picture you just took.', 
                            Icons.photo_library),
                          _buildNumberedStep(3, 'Select the words', 
                            'Your phone can read the words in photos! '
                            'Tap and hold your finger on any word in the photo. '
                            'You\'ll see the text get highlighted.', 
                            Icons.touch_app),
                          _buildNumberedStep(4, 'Copy the text', 
                            'Tap "Select All" to grab all the words, then tap "Copy".', 
                            Icons.copy),
                          _buildNumberedStep(5, 'Paste it here', 
                            'Come back to YarnAI, tap and hold in the box below, '
                            'then tap "Paste". All done!', 
                            Icons.paste),
                          const SizedBox(height: 12),
                          Container(
                            padding: const EdgeInsets.all(12),
                            decoration: BoxDecoration(
                              color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.1),
                              borderRadius: BorderRadius.circular(8),
                            ),
                            child: Row(
                              children: [
                                Icon(Icons.lightbulb_outline, 
                                  size: 20, 
                                  color: isDark ? AppColors.ochre : AppColors.primary,
                                ),
                                const SizedBox(width: 10),
                                Expanded(
                                  child: Text(
                                    'Tip: If your camera has a "Scan" or "Document" mode, use that - it makes the text even clearer!',
                                    style: Theme.of(context).textTheme.bodySmall?.copyWith(
                                      fontStyle: FontStyle.italic,
                                    ),
                                  ),
                                ),
                              ],
                            ),
                          ),
                        ],
                      ),
                    ),
                  ),
                ],
              ),
              const SizedBox(height: 16),
              Stack(
                children: [
                  TextField(
                    controller: _documentController,
                    maxLines: 8,
                    decoration: const InputDecoration(
                      hintText: 'Paste your letter or document here...',
                      alignLabelWithHint: true,
                    ),
                  ),
                  Positioned(
                    right: 8,
                    top: 8,
                    child: FloatingActionButton.small(
                      onPressed: _toggleListening,
                      backgroundColor: _isListening 
                          ? Colors.red 
                          : (isDark ? AppColors.primaryLight : AppColors.primary),
                      child: Icon(
                        _isListening ? Icons.stop : Icons.mic,
                        color: Colors.white,
                      ),
                    ),
                  ),
                ],
              ),
              if (_isListening)
                Padding(
                  padding: const EdgeInsets.only(top: 8),
                  child: Text(
                    'Listening... speak now',
                    style: TextStyle(
                      color: Colors.red,
                      fontStyle: FontStyle.italic,
                    ),
                  ),
                ),
              const SizedBox(height: 16),
              SizedBox(
                width: double.infinity,
                child: ElevatedButton(
                  onPressed: _isLoading ? null : _analyzeDocument,
                  child: _isLoading
                      ? Row(
                          mainAxisAlignment: MainAxisAlignment.center,
                          children: [
                            const SizedBox(
                              height: 20,
                              width: 20,
                              child: CircularProgressIndicator(
                                strokeWidth: 2,
                                color: Colors.white,
                              ),
                            ),
                            const SizedBox(width: 12),
                            const Text('Reading...'),
                          ],
                        )
                      : const Text('Explain This'),
                ),
              ),
              if (_analysis != null) ...[
                const SizedBox(height: 24),
                _buildAnalysisCard(),
                const SizedBox(height: 16),
                SizedBox(
                  width: double.infinity,
                  child: OutlinedButton.icon(
                    onPressed: _clearAndStartOver,
                    icon: const Icon(Icons.refresh),
                    label: const Text('Explain a different letter'),
                  ),
                ),
              ],
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildTipRow(IconData icon, String text) {
    return Padding(
      padding: const EdgeInsets.only(bottom: 8),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Icon(icon, size: 18, color: Colors.grey),
          const SizedBox(width: 8),
          Expanded(
            child: Text(text, style: Theme.of(context).textTheme.bodyMedium),
          ),
        ],
      ),
    );
  }

  Widget _buildNumberedStep(int number, String title, String description, IconData icon) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final primaryColor = isDark ? AppColors.primaryLight : AppColors.primary;
    
    return Padding(
      padding: const EdgeInsets.only(bottom: 16),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Container(
            width: 32,
            height: 32,
            decoration: BoxDecoration(
              gradient: LinearGradient(
                colors: [primaryColor, AppColors.ochre],
              ),
              shape: BoxShape.circle,
            ),
            child: Center(
              child: Text(
                '$number',
                style: const TextStyle(
                  color: Colors.white,
                  fontWeight: FontWeight.bold,
                  fontSize: 14,
                ),
              ),
            ),
          ),
          const SizedBox(width: 12),
          Expanded(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Row(
                  children: [
                    Icon(icon, size: 18, color: primaryColor),
                    const SizedBox(width: 6),
                    Text(
                      title,
                      style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                        fontWeight: FontWeight.w600,
                      ),
                    ),
                  ],
                ),
                const SizedBox(height: 4),
                Text(
                  description,
                  style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                    color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                    height: 1.4,
                  ),
                ),
              ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildAnalysisCard() {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                Icon(
                  Icons.lightbulb_outline, 
                  color: isDark ? AppColors.primaryLight : AppColors.primary,
                ),
                const SizedBox(width: 8),
                Expanded(
                  child: Text(
                    'Here\'s what it means',
                    style: Theme.of(context).textTheme.titleMedium?.copyWith(
                      fontWeight: FontWeight.w600,
                    ),
                  ),
                ),
                IconButton(
                  icon: Icon(
                    Icons.volume_up,
                    color: isDark ? AppColors.primaryLight : AppColors.primary,
                  ),
                  tooltip: 'Read summary aloud',
                  onPressed: () {
                    if (_analysis?['summary'] != null) {
                      _voiceService.speak(_analysis!['summary']);
                    }
                  },
                ),
              ],
            ),
            const SizedBox(height: 12),
            Text(
              _analysis?['summary'] ?? '',
              style: Theme.of(context).textTheme.bodyLarge,
            ),
            if (_analysis?['actionItems'] != null && (_analysis!['actionItems'] as List).isNotEmpty) ...[
              const SizedBox(height: 16),
              Row(
                children: [
                  Expanded(
                    child: Text(
                      'What you might need to do:',
                      style: Theme.of(context).textTheme.titleSmall?.copyWith(
                        fontWeight: FontWeight.w600,
                      ),
                    ),
                  ),
                  TextButton.icon(
                    onPressed: () {
                      final items = (_analysis!['actionItems'] as List).join('. ');
                      _voiceService.speak("Here's what you might need to do. $items");
                    },
                    icon: Icon(Icons.play_arrow, size: 18),
                    label: Text('Read all', style: TextStyle(fontSize: 12)),
                  ),
                ],
              ),
              const SizedBox(height: 8),
              ...(_analysis!['actionItems'] as List).asMap().entries.map((entry) {
                final index = entry.key;
                final item = entry.value.toString();
                return Padding(
                  padding: const EdgeInsets.only(bottom: 8),
                  child: InkWell(
                    onTap: () => _speakActionItem(item),
                    borderRadius: BorderRadius.circular(8),
                    child: Padding(
                      padding: const EdgeInsets.all(8),
                      child: Row(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Container(
                            width: 24,
                            height: 24,
                            decoration: BoxDecoration(
                              color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.1),
                              shape: BoxShape.circle,
                            ),
                            child: Center(
                              child: Text(
                                '${index + 1}',
                                style: TextStyle(
                                  fontSize: 12,
                                  fontWeight: FontWeight.bold,
                                  color: isDark ? AppColors.primaryLight : AppColors.primary,
                                ),
                              ),
                            ),
                          ),
                          const SizedBox(width: 12),
                          Expanded(child: Text(item)),
                          Icon(
                            Icons.volume_up,
                            size: 18,
                            color: isDark ? AppColors.primaryLight : AppColors.primary,
                          ),
                        ],
                      ),
                    ),
                  ),
                );
              }),
            ],
          ],
        ),
      ),
    );
  }
}
