import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:url_launcher/url_launcher.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/services/crisis_detection_service.dart';
import '../../../core/services/voice_scripts.dart';
import '../../../core/widgets/shared_widgets.dart' show CulturalBackground, SettingsButton, FunkyMicButton;

class ChatPage extends StatefulWidget {
  final String? initialContext;
  final String? journeyId;
  
  const ChatPage({
    super.key,
    this.initialContext,
    this.journeyId,
  });

  @override
  State<ChatPage> createState() => _ChatPageState();
}

class _ChatPageState extends State<ChatPage> {
  final TextEditingController _messageController = TextEditingController();
  final ScrollController _scrollController = ScrollController();
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _apiClient = ApiClient();
  final CrisisDetectionService _crisisService = CrisisDetectionService();
  final VoiceScripts _scripts = VoiceScripts();
  
  final List<ChatMessage> _messages = [];
  bool _isLoading = false;
  bool _isListening = false;
  bool _hasPlayedIntro = false;

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
    
    if (widget.initialContext != null && widget.initialContext!.isNotEmpty) {
      _processInitialContext();
    }
  }
  
  Future<void> _processInitialContext() async {
    await Future.delayed(const Duration(milliseconds: 800));
    
    setState(() {
      _messages.add(ChatMessage(
        text: "You shared: ${widget.initialContext}",
        isUser: true,
      ));
      _isLoading = true;
    });
    
    try {
      final data = await _apiClient.post('/api/chat-simple', {
        'message': widget.initialContext,
        'journeyId': widget.journeyId,
      });

      final reply = data['response'] ?? "I hear you. Let's work through this together.";

      setState(() {
        _messages.add(ChatMessage(text: reply, isUser: false));
        _isLoading = false;
      });

      _scrollToBottom();
      
      if (_prefs.autoPlayVoice) {
        await _voiceService.speak(reply);
      }
    } catch (e) {
      setState(() {
        _messages.add(ChatMessage(
          text: "I hear you. Let's work through this together. What would you like to focus on first?",
          isUser: false,
        ));
        _isLoading = false;
      });
    }
  }

  Future<void> _initializeAndPlay() async {
    debugPrint('ChatPage: Initializing voice service...');
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playIntro();
    }
  }

  @override
  void dispose() {
    _messageController.dispose();
    _scrollController.dispose();
    _voiceService.stop();
    super.dispose();
  }

  void _playIntro() {
    debugPrint('ChatPage: _playIntro called, autoPlay=${_prefs.autoPlayVoice}');
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      _voiceService.speak(_scripts.askAnythingIntro);
    }
  }

  Future<void> _sendMessage(String text) async {
    if (text.trim().isEmpty) return;

    final crisisCheck = _crisisService.checkMessage(text);
    
    setState(() {
      _messages.add(ChatMessage(text: text, isUser: true));
      _isLoading = true;
    });
    
    _messageController.clear();
    _scrollToBottom();

    if (crisisCheck.isCritical) {
      setState(() {
        _messages.add(ChatMessage(
          text: crisisCheck.message ?? '',
          isUser: false,
          isCrisisResponse: true,
          crisisData: crisisCheck,
        ));
        _isLoading = false;
      });
      _scrollToBottom();
      
      if (_prefs.autoPlayVoice && crisisCheck.message != null) {
        await _voiceService.speak(crisisCheck.message!);
      }
      return;
    }

    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Let me think about that...");
    }

    try {
      final data = await _apiClient.post('/api/chat-simple', {
        'message': text,
      });

      String reply = data['response'] ?? 'Sorry, I couldn\'t understand that.';
      
      if (crisisCheck.shouldIntervene && !crisisCheck.isCritical && crisisCheck.message != null) {
        reply = "$reply\n\n${crisisCheck.message}";
      }

      setState(() {
        _messages.add(ChatMessage(
          text: reply, 
          isUser: false,
          isCrisisResponse: crisisCheck.shouldIntervene && !crisisCheck.isCritical,
          crisisData: crisisCheck.shouldIntervene ? crisisCheck : null,
        ));
        _isLoading = false;
      });

      _scrollToBottom();
      
      await _voiceService.speak(reply);
    } catch (e) {
      setState(() {
        _messages.add(ChatMessage(
          text: 'Sorry, something went wrong. Try again?',
          isUser: false,
        ));
        _isLoading = false;
      });
      
      if (_prefs.autoPlayVoice) {
        _voiceService.speak("Sorry, something went wrong. Want to try again?");
      }
    }
  }

  Future<void> _callCrisisLine(String phone) async {
    final cleaned = phone.replaceAll(' ', '');
    final uri = Uri.parse('tel:$cleaned');
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri);
    }
  }

  void _scrollToBottom() {
    Future.delayed(const Duration(milliseconds: 100), () {
      if (_scrollController.hasClients) {
        _scrollController.animateTo(
          _scrollController.position.maxScrollExtent,
          duration: const Duration(milliseconds: 300),
          curve: Curves.easeOut,
        );
      }
    });
  }

  Future<void> _toggleListening() async {
    _voiceService.stop();
    
    if (_isListening) {
      await _voiceService.stopListening();
      setState(() => _isListening = false);
    } else {
      setState(() => _isListening = true);
      
      await _voiceService.speakThenListen(
        prompt: "I'm listening. What would you like to know?",
        onResult: (text) {
          setState(() => _isListening = false);
          if (text.isNotEmpty) {
            _sendMessage(text);
          }
        },
        onError: (error) {
          setState(() => _isListening = false);
          if (_prefs.autoPlayVoice) {
            _voiceService.speak("I couldn't hear that clearly. Tap the mic to try again.");
          }
          ScaffoldMessenger.of(context).showSnackBar(
            SnackBar(content: Text(error)),
          );
        },
      );
    }
  }

  void _speakMessage(ChatMessage message) {
    _voiceService.stop();
    _voiceService.speak(message.text);
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () {
            _voiceService.stop();
            if (_prefs.autoPlayVoice && _messages.isNotEmpty) {
              _voiceService.speak("Heading back home");
            }
            context.go('/');
          },
        ),
        title: const Text('Ask Anything'),
        actions: [
          IconButton(
            icon: const Icon(Icons.volume_up),
            tooltip: 'Read page aloud',
            onPressed: () {
              _voiceService.speak(
                _messages.isEmpty
                    ? "This is where you can ask me anything. Tap the mic to speak, or type your question."
                    : "Tap any message to hear it read aloud. Tap the mic to ask another question."
              );
            },
          ),
          const SettingsButton(),
        ],
      ),
      body: CulturalBackground(
        child: Column(
          children: [
            Expanded(
              child: _messages.isEmpty
                  ? _buildEmptyState()
                  : ListView.builder(
                      controller: _scrollController,
                      padding: const EdgeInsets.all(16),
                      itemCount: _messages.length + (_isLoading ? 1 : 0),
                      itemBuilder: (context, index) {
                        if (index == _messages.length && _isLoading) {
                          return _buildLoadingBubble();
                        }
                        return _buildMessageBubble(_messages[index]);
                      },
                    ),
            ),
            _buildInputArea(),
          ],
        ),
      ),
    );
  }

  Widget _buildEmptyState() {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Center(
      child: Padding(
        padding: const EdgeInsets.all(32),
        child: Column(
          mainAxisAlignment: MainAxisAlignment.center,
          children: [
            FunkyMicButton(
              onTap: _toggleListening,
              isListening: _isListening,
              size: 100,
            ),
            const SizedBox(height: 24),
            Text(
              'No wrong questions here',
              style: Theme.of(context).textTheme.headlineSmall,
              textAlign: TextAlign.center,
            ),
            const SizedBox(height: 8),
            Text(
              _isListening 
                  ? 'Listening...' 
                  : 'Tap the mic to speak, or type below',
              style: Theme.of(context).textTheme.bodyMedium,
              textAlign: TextAlign.center,
            ),
            const SizedBox(height: 32),
            Text(
              'Try asking:',
              style: Theme.of(context).textTheme.bodySmall?.copyWith(
                color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
              ),
            ),
            const SizedBox(height: 8),
            Wrap(
              spacing: 8,
              runSpacing: 8,
              alignment: WrapAlignment.center,
              children: [
                _buildSuggestionChip('What does this word mean?'),
                _buildSuggestionChip('Help me write a text'),
                _buildSuggestionChip('Explain my rights'),
              ],
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildSuggestionChip(String text) {
    return ActionChip(
      label: Text(text, style: const TextStyle(fontSize: 12)),
      onPressed: () {
        if (_prefs.autoPlayVoice) {
          _voiceService.speak("Good choice. Let me help you with that.");
        }
        _sendMessage(text);
      },
    );
  }

  Widget _buildMessageBubble(ChatMessage message) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    if (message.isCrisisResponse) {
      return _buildCrisisResponseBubble(message);
    }
    
    return GestureDetector(
      onTap: () => _speakMessage(message),
      child: Align(
        alignment: message.isUser ? Alignment.centerRight : Alignment.centerLeft,
        child: Container(
          margin: const EdgeInsets.only(bottom: 12),
          padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
          constraints: BoxConstraints(
            maxWidth: MediaQuery.of(context).size.width * 0.75,
          ),
          decoration: BoxDecoration(
            color: message.isUser 
                ? (isDark ? AppColors.primaryLight : AppColors.primary) 
                : Theme.of(context).cardColor,
            borderRadius: BorderRadius.circular(16),
          ),
          child: Row(
            mainAxisSize: MainAxisSize.min,
            crossAxisAlignment: CrossAxisAlignment.end,
            children: [
              Flexible(
                child: Text(
                  message.text,
                  style: TextStyle(
                    color: message.isUser ? (isDark ? AppColors.textPrimary : Colors.white) : null,
                    fontSize: 16,
                  ),
                ),
              ),
              if (!message.isUser) ...[
                const SizedBox(width: 8),
                Icon(
                  Icons.volume_up,
                  size: 16,
                  color: isDark ? AppColors.primaryLight : AppColors.primary,
                ),
              ],
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildCrisisResponseBubble(ChatMessage message) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Align(
      alignment: Alignment.centerLeft,
      child: Container(
        margin: const EdgeInsets.only(bottom: 12),
        constraints: BoxConstraints(
          maxWidth: MediaQuery.of(context).size.width * 0.9,
        ),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            GestureDetector(
              onTap: () => _speakMessage(message),
              child: Container(
                padding: const EdgeInsets.all(16),
                decoration: BoxDecoration(
                  color: AppColors.terracotta.withOpacity(0.15),
                  borderRadius: BorderRadius.circular(16),
                  border: Border.all(
                    color: AppColors.terracotta.withOpacity(0.3),
                    width: 1,
                  ),
                ),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.favorite, color: AppColors.terracotta, size: 20),
                        const SizedBox(width: 8),
                        Text(
                          "You're not alone",
                          style: TextStyle(
                            fontWeight: FontWeight.w600,
                            color: AppColors.terracotta,
                          ),
                        ),
                        const Spacer(),
                        Icon(
                          Icons.volume_up,
                          size: 16,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                        ),
                      ],
                    ),
                    const SizedBox(height: 12),
                    Text(
                      message.text,
                      style: const TextStyle(fontSize: 16),
                    ),
                  ],
                ),
              ),
            ),
            if (message.crisisData != null && message.crisisData!.crisisContacts.isNotEmpty) ...[
              const SizedBox(height: 12),
              ...message.crisisData!.crisisContacts.map((contact) => 
                Container(
                  margin: const EdgeInsets.only(bottom: 8),
                  child: Material(
                    color: contact.isPrimary 
                        ? AppColors.primary.withOpacity(0.1)
                        : Theme.of(context).cardColor,
                    borderRadius: BorderRadius.circular(12),
                    child: InkWell(
                      onTap: () => _callCrisisLine(contact.phone),
                      borderRadius: BorderRadius.circular(12),
                      child: Padding(
                        padding: const EdgeInsets.all(12),
                        child: Row(
                          children: [
                            Container(
                              width: 40,
                              height: 40,
                              decoration: BoxDecoration(
                                color: contact.isMobRun 
                                    ? AppColors.ochre.withOpacity(0.2)
                                    : AppColors.primary.withOpacity(0.1),
                                borderRadius: BorderRadius.circular(20),
                              ),
                              child: Icon(
                                Icons.phone,
                                color: contact.isMobRun ? AppColors.ochre : AppColors.primary,
                                size: 20,
                              ),
                            ),
                            const SizedBox(width: 12),
                            Expanded(
                              child: Column(
                                crossAxisAlignment: CrossAxisAlignment.start,
                                children: [
                                  Row(
                                    children: [
                                      Text(
                                        contact.name,
                                        style: const TextStyle(
                                          fontWeight: FontWeight.w600,
                                          fontSize: 15,
                                        ),
                                      ),
                                      if (contact.isMobRun) ...[
                                        const SizedBox(width: 8),
                                        Container(
                                          padding: const EdgeInsets.symmetric(
                                            horizontal: 6,
                                            vertical: 2,
                                          ),
                                          decoration: BoxDecoration(
                                            color: AppColors.ochre,
                                            borderRadius: BorderRadius.circular(4),
                                          ),
                                          child: const Text(
                                            'Mob-run',
                                            style: TextStyle(
                                              color: Colors.white,
                                              fontSize: 10,
                                              fontWeight: FontWeight.w500,
                                            ),
                                          ),
                                        ),
                                      ],
                                    ],
                                  ),
                                  Text(
                                    contact.description,
                                    style: TextStyle(
                                      fontSize: 13,
                                      color: isDark 
                                          ? AppColors.textSecondaryDark 
                                          : AppColors.textSecondary,
                                    ),
                                  ),
                                ],
                              ),
                            ),
                            Text(
                              contact.phone,
                              style: TextStyle(
                                fontWeight: FontWeight.w600,
                                fontSize: 16,
                                color: isDark ? AppColors.primaryLight : AppColors.primary,
                              ),
                            ),
                          ],
                        ),
                      ),
                    ),
                  ),
                ),
              ),
            ],
          ],
        ),
      ),
    );
  }

  Widget _buildLoadingBubble() {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Align(
      alignment: Alignment.centerLeft,
      child: Container(
        margin: const EdgeInsets.only(bottom: 12),
        padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
        decoration: BoxDecoration(
          color: Theme.of(context).cardColor,
          borderRadius: BorderRadius.circular(16),
        ),
        child: Row(
          mainAxisSize: MainAxisSize.min,
          children: [
            SizedBox(
              width: 16,
              height: 16,
              child: CircularProgressIndicator(
                strokeWidth: 2,
                color: isDark ? AppColors.primaryLight : AppColors.primary,
              ),
            ),
            const SizedBox(width: 8),
            const Text('Thinking...'),
          ],
        ),
      ),
    );
  }

  Widget _buildInputArea() {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Container(
      padding: const EdgeInsets.all(16),
      decoration: BoxDecoration(
        color: Theme.of(context).cardColor,
        border: Border(
          top: BorderSide(
            color: (isDark ? AppColors.textTertiaryDark : AppColors.textTertiary).withOpacity(0.2),
          ),
        ),
      ),
      child: SafeArea(
        top: false,
        child: Row(
          children: [
            Expanded(
              child: TextField(
                controller: _messageController,
                decoration: InputDecoration(
                  hintText: 'Type your question...',
                  border: OutlineInputBorder(
                    borderRadius: BorderRadius.circular(24),
                  ),
                  contentPadding: const EdgeInsets.symmetric(
                    horizontal: 20,
                    vertical: 12,
                  ),
                ),
                textInputAction: TextInputAction.send,
                onSubmitted: _sendMessage,
              ),
            ),
            const SizedBox(width: 8),
            FunkyMicButton(
              onTap: _toggleListening,
              isListening: _isListening,
              size: 56,
            ),
          ],
        ),
      ),
    );
  }
}

class ChatMessage {
  final String text;
  final bool isUser;
  final bool isCrisisResponse;
  final CrisisResponse? crisisData;

  ChatMessage({
    required this.text, 
    required this.isUser,
    this.isCrisisResponse = false,
    this.crisisData,
  });
}
