import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:url_launcher/url_launcher.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/widgets/shared_widgets.dart';

class CentrelinkPage extends StatefulWidget {
  const CentrelinkPage({super.key});

  @override
  State<CentrelinkPage> createState() => _CentrelinkPageState();
}

class _CentrelinkPageState extends State<CentrelinkPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _apiClient = ApiClient();
  final TextEditingController _issueController = TextEditingController();
  
  bool _isLoading = false;
  String? _selectedIssue;
  Map<String, dynamic>? _helpResponse;

  static const List<_CentrelinkIssue> _commonIssues = [
    _CentrelinkIssue(
      id: 'payment_stopped',
      title: 'My payment stopped',
      description: 'Payment cut off or suspended',
      icon: Icons.money_off,
    ),
    _CentrelinkIssue(
      id: 'breach',
      title: 'Got a breach letter',
      description: 'Compliance issue or penalty',
      icon: Icons.warning_amber,
    ),
    _CentrelinkIssue(
      id: 'debt',
      title: 'They say I owe money',
      description: 'Debt notice or overpayment',
      icon: Icons.receipt_long,
    ),
    _CentrelinkIssue(
      id: 'appointment_missed',
      title: 'Missed an appointment',
      description: 'Forgot or couldn\'t make it',
      icon: Icons.event_busy,
    ),
    _CentrelinkIssue(
      id: 'wait_times',
      title: 'Can\'t get through',
      description: 'Long wait times on phone',
      icon: Icons.phone_disabled,
    ),
    _CentrelinkIssue(
      id: 'crisis_payment',
      title: 'Need emergency money',
      description: 'Crisis or hardship payment',
      icon: Icons.emergency,
    ),
    _CentrelinkIssue(
      id: 'appeal',
      title: 'Want to appeal a decision',
      description: 'Disagree with their decision',
      icon: Icons.gavel,
    ),
    _CentrelinkIssue(
      id: 'other',
      title: 'Something else',
      description: 'Describe your issue',
      icon: Icons.help_outline,
    ),
  ];

  static final List<_QuickTip> _offlineTips = [
    _QuickTip(
      title: 'Always get it in writing',
      content: 'Ask them to put decisions in writing. If they refuse, note the date, time, and name of who you spoke to.',
    ),
    _QuickTip(
      title: 'You can ask for a review',
      content: 'If you disagree with a decision, you have 13 weeks to ask for an internal review. This is free.',
    ),
    _QuickTip(
      title: 'Special circumstances count',
      content: 'Sorry business, cultural obligations, caring for family - these are valid reasons. Tell them.',
    ),
    _QuickTip(
      title: 'Get an advocate',
      content: 'Free services can help you deal with Centrelink. You don\'t have to do this alone.',
    ),
  ];

  static final List<_SupportService> _supportServices = [
    _SupportService(
      name: 'Aboriginal Legal Services',
      description: 'Free legal help for First Nations people',
      phone: '1800 732 273',
      isAboriginal: true,
    ),
    _SupportService(
      name: 'Welfare Rights Centre',
      description: 'Free help with Centrelink problems',
      phone: '1800 226 028',
      isAboriginal: false,
    ),
    _SupportService(
      name: 'National Debt Helpline',
      description: 'Free financial counselling',
      phone: '1800 007 007',
      isAboriginal: false,
    ),
    _SupportService(
      name: 'Centrelink Indigenous Line',
      description: 'Dedicated line for First Nations customers',
      phone: '1800 136 380',
      isAboriginal: true,
    ),
  ];

  @override
  void initState() {
    super.initState();
    _initVoice();
  }

  Future<void> _initVoice() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted && _prefs.autoPlayVoice) {
      await Future.delayed(const Duration(milliseconds: 500));
      _voiceService.speak(
        "This page helps you navigate Centrelink issues. "
        "Pick what's going on, and I'll help you work out what to do. "
        "You've got rights, and I'll help you use them."
      );
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _issueController.dispose();
    super.dispose();
  }

  Future<void> _getHelp() async {
    if (_selectedIssue == null) return;
    
    setState(() => _isLoading = true);
    
    final issueDetails = _selectedIssue == 'other' 
        ? _issueController.text 
        : _selectedIssue;
    
    try {
      final data = await _apiClient.post('/api/centrelink/help', {
        'issueType': _selectedIssue,
        'details': issueDetails,
      });
      
      setState(() {
        _helpResponse = data;
        _isLoading = false;
      });
      
      if (_prefs.autoPlayVoice && data['summary'] != null) {
        _voiceService.speak(data['summary']);
      }
    } catch (e) {
      setState(() {
        _helpResponse = {
          'summary': "I couldn't get online help right now, but here's what you can do:",
          'steps': [
            "Call the Centrelink Indigenous Line on 1800 136 380 - they're set up to help mob",
            "Ask for everything in writing - this protects you",
            "If you've been breached or cut off, you have the right to ask why",
            "Contact the Welfare Rights Centre on 1800 226 028 for free help"
          ],
          'script': "I'm calling about my payment. Can you please explain what's happened and put it in writing for me?",
          'rights': "You have the right to an internal review of any decision within 13 weeks.",
          'escalation': "If Centrelink won't help, call Aboriginal Legal Services on 1800 732 273."
        };
        _isLoading = false;
      });
    }
  }

  Future<void> _callService(String phone) async {
    final cleaned = phone.replaceAll(' ', '');
    final uri = Uri.parse('tel:$cleaned');
    
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Calling now");
    }
    
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri);
    }
  }

  void _goBack() {
    _voiceService.stop();
    context.go('/');
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              _buildHeader(isDark),
              Expanded(
                child: ScrollableWithIndicator(
                  padding: const EdgeInsets.all(16),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      _buildEmpowermentBanner(isDark),
                      const SizedBox(height: 20),
                      _buildIssueSelector(isDark),
                      if (_selectedIssue == 'other') ...[
                        const SizedBox(height: 16),
                        _buildCustomIssueInput(isDark),
                      ],
                      if (_selectedIssue != null) ...[
                        const SizedBox(height: 16),
                        _buildGetHelpButton(isDark),
                      ],
                      if (_helpResponse != null) ...[
                        const SizedBox(height: 20),
                        _buildHelpResponse(isDark),
                      ],
                      const SizedBox(height: 24),
                      _buildQuickTips(isDark),
                      const SizedBox(height: 24),
                      _buildSupportServices(isDark),
                    ],
                  ),
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildHeader(bool isDark) {
    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 8),
      child: Row(
        children: [
          IconButton(
            icon: const Icon(Icons.arrow_back),
            onPressed: _goBack,
            tooltip: 'Back to home',
          ),
          const SizedBox(width: 8),
          Expanded(
            child: Text(
              "Centrelink Navigator",
              style: Theme.of(context).textTheme.titleLarge,
            ),
          ),
          IconButton(
            icon: const Icon(Icons.volume_up),
            onPressed: () {
              _voiceService.speak(
                "Pick your issue from the list. "
                "I'll give you steps to follow and what to say. "
                "You've got rights and I'll help you use them."
              );
            },
            tooltip: 'Read aloud',
          ),
        ],
      ),
    );
  }

  Widget _buildEmpowermentBanner(bool isDark) {
    return Container(
      padding: const EdgeInsets.all(16),
      decoration: BoxDecoration(
        gradient: LinearGradient(
          colors: [
            AppColors.primary.withOpacity(0.2),
            AppColors.ochre.withOpacity(0.15),
          ],
        ),
        borderRadius: BorderRadius.circular(12),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            children: [
              Icon(Icons.shield_outlined, color: AppColors.primary, size: 28),
              const SizedBox(width: 12),
              Expanded(
                child: Text(
                  "You've got rights",
                  style: Theme.of(context).textTheme.titleMedium?.copyWith(
                    fontWeight: FontWeight.bold,
                  ),
                ),
              ),
            ],
          ),
          const SizedBox(height: 8),
          Text(
            "Centrelink has rules they have to follow too. "
            "If something's wrong, you can challenge it. "
            "Let me help you work out what to do.",
            style: Theme.of(context).textTheme.bodyMedium,
          ),
        ],
      ),
    );
  }

  Widget _buildIssueSelector(bool isDark) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          "What's going on?",
          style: Theme.of(context).textTheme.titleMedium?.copyWith(
            fontWeight: FontWeight.w600,
          ),
        ),
        const SizedBox(height: 12),
        Wrap(
          spacing: 8,
          runSpacing: 8,
          children: _commonIssues.map((issue) {
            final isSelected = issue.id == _selectedIssue;
            return InkWell(
              onTap: () {
                setState(() {
                  _selectedIssue = issue.id;
                  _helpResponse = null;
                });
              },
              borderRadius: BorderRadius.circular(12),
              child: Container(
                width: (MediaQuery.of(context).size.width - 48) / 2,
                padding: const EdgeInsets.all(12),
                decoration: BoxDecoration(
                  color: isSelected 
                      ? AppColors.primary.withOpacity(0.2)
                      : (isDark ? AppColors.cardDark : AppColors.cardLight),
                  borderRadius: BorderRadius.circular(12),
                  border: Border.all(
                    color: isSelected 
                        ? AppColors.primary 
                        : Colors.transparent,
                    width: 2,
                  ),
                ),
                child: Row(
                  children: [
                    Icon(
                      issue.icon,
                      size: 24,
                      color: isSelected ? AppColors.primary : (isDark ? AppColors.textSecondaryDark : AppColors.textSecondary),
                    ),
                    const SizedBox(width: 8),
                    Expanded(
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Text(
                            issue.title,
                            style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                              fontWeight: FontWeight.w600,
                            ),
                          ),
                          Text(
                            issue.description,
                            style: Theme.of(context).textTheme.bodySmall?.copyWith(
                              color: isDark ? AppColors.textTertiaryDark : AppColors.textTertiary,
                            ),
                            maxLines: 1,
                            overflow: TextOverflow.ellipsis,
                          ),
                        ],
                      ),
                    ),
                  ],
                ),
              ),
            );
          }).toList(),
        ),
      ],
    );
  }

  Widget _buildCustomIssueInput(bool isDark) {
    return TextField(
      controller: _issueController,
      maxLines: 3,
      decoration: InputDecoration(
        hintText: "Tell me what's happening with Centrelink...",
        suffixIcon: MicButton(onResult: (t) => _issueController.text = t),
      ),
    );
  }

  Widget _buildGetHelpButton(bool isDark) {
    return SizedBox(
      width: double.infinity,
      child: ElevatedButton.icon(
        onPressed: _isLoading ? null : _getHelp,
        icon: _isLoading 
            ? SizedBox(
                width: 20,
                height: 20,
                child: CircularProgressIndicator(
                  strokeWidth: 2,
                  color: Colors.white,
                ),
              )
            : const Icon(Icons.lightbulb_outline),
        label: Text(_isLoading ? "Working on it..." : "Tell me what to do"),
      ),
    );
  }

  Widget _buildHelpResponse(bool isDark) {
    if (_helpResponse == null) return const SizedBox.shrink();
    
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Row(
              children: [
                Icon(Icons.check_circle_outline, color: AppColors.primary),
                const SizedBox(width: 8),
                Expanded(
                  child: Text(
                    "Here's what to do",
                    style: Theme.of(context).textTheme.titleMedium?.copyWith(
                      fontWeight: FontWeight.w600,
                    ),
                  ),
                ),
                IconButton(
                  icon: const Icon(Icons.volume_up, size: 20),
                  onPressed: () {
                    final summary = _helpResponse!['summary'] ?? '';
                    final steps = (_helpResponse!['steps'] as List?)?.join('. ') ?? '';
                    _voiceService.speak('$summary $steps');
                  },
                  tooltip: 'Read aloud',
                ),
              ],
            ),
            const SizedBox(height: 12),
            Text(
              _helpResponse!['summary'] ?? '',
              style: Theme.of(context).textTheme.bodyMedium,
            ),
            const SizedBox(height: 16),
            if (_helpResponse!['steps'] != null)
              ...(_helpResponse!['steps'] as List).asMap().entries.map((entry) {
                return Padding(
                  padding: const EdgeInsets.only(bottom: 8),
                  child: Row(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Container(
                        width: 24,
                        height: 24,
                        decoration: BoxDecoration(
                          color: AppColors.primary.withOpacity(0.2),
                          shape: BoxShape.circle,
                        ),
                        child: Center(
                          child: Text(
                            '${entry.key + 1}',
                            style: TextStyle(
                              color: AppColors.primary,
                              fontWeight: FontWeight.bold,
                              fontSize: 12,
                            ),
                          ),
                        ),
                      ),
                      const SizedBox(width: 12),
                      Expanded(
                        child: Text(
                          entry.value.toString(),
                          style: Theme.of(context).textTheme.bodyMedium,
                        ),
                      ),
                    ],
                  ),
                );
              }),
            if (_helpResponse!['script'] != null) ...[
              const SizedBox(height: 16),
              Container(
                padding: const EdgeInsets.all(12),
                decoration: BoxDecoration(
                  color: AppColors.ochre.withOpacity(0.1),
                  borderRadius: BorderRadius.circular(8),
                  border: Border.all(color: AppColors.ochre.withOpacity(0.3)),
                ),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.format_quote, color: AppColors.ochre, size: 20),
                        const SizedBox(width: 8),
                        Text(
                          "What to say",
                          style: Theme.of(context).textTheme.labelLarge?.copyWith(
                            color: AppColors.ochre,
                            fontWeight: FontWeight.w600,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(
                      _helpResponse!['script'] ?? '',
                      style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                        fontStyle: FontStyle.italic,
                      ),
                    ),
                  ],
                ),
              ),
            ],
            if (_helpResponse!['rights'] != null) ...[
              const SizedBox(height: 12),
              Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Icon(Icons.shield_outlined, color: AppColors.primary, size: 18),
                  const SizedBox(width: 8),
                  Expanded(
                    child: Text(
                      _helpResponse!['rights'] ?? '',
                      style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                        fontWeight: FontWeight.w500,
                      ),
                    ),
                  ),
                ],
              ),
            ],
            if (_helpResponse!['escalation'] != null) ...[
              const SizedBox(height: 8),
              Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Icon(Icons.arrow_upward, color: AppColors.terracotta, size: 18),
                  const SizedBox(width: 8),
                  Expanded(
                    child: Text(
                      _helpResponse!['escalation'] ?? '',
                      style: Theme.of(context).textTheme.bodySmall,
                    ),
                  ),
                ],
              ),
            ],
          ],
        ),
      ),
    );
  }

  Widget _buildQuickTips(bool isDark) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          "Quick Tips",
          style: Theme.of(context).textTheme.titleMedium?.copyWith(
            fontWeight: FontWeight.w600,
          ),
        ),
        const SizedBox(height: 12),
        ..._offlineTips.map((tip) => Card(
          margin: const EdgeInsets.only(bottom: 8),
          child: ExpansionTile(
            leading: Icon(Icons.lightbulb_outline, color: AppColors.ochre),
            title: Text(
              tip.title,
              style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                fontWeight: FontWeight.w600,
              ),
            ),
            children: [
              Padding(
                padding: const EdgeInsets.fromLTRB(16, 0, 16, 16),
                child: Text(
                  tip.content,
                  style: Theme.of(context).textTheme.bodyMedium,
                ),
              ),
            ],
          ),
        )),
      ],
    );
  }

  Widget _buildSupportServices(bool isDark) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          "Free Help Available",
          style: Theme.of(context).textTheme.titleMedium?.copyWith(
            fontWeight: FontWeight.w600,
          ),
        ),
        const SizedBox(height: 4),
        Text(
          "These services can advocate for you",
          style: Theme.of(context).textTheme.bodySmall?.copyWith(
            color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
          ),
        ),
        const SizedBox(height: 12),
        ..._supportServices.map((service) => Card(
          margin: const EdgeInsets.only(bottom: 8),
          child: ListTile(
            leading: Container(
              width: 40,
              height: 40,
              decoration: BoxDecoration(
                color: service.isAboriginal 
                    ? AppColors.ochre.withOpacity(0.2)
                    : AppColors.primary.withOpacity(0.1),
                borderRadius: BorderRadius.circular(8),
              ),
              child: Icon(
                Icons.support_agent,
                color: service.isAboriginal ? AppColors.ochre : AppColors.primary,
              ),
            ),
            title: Row(
              children: [
                Expanded(child: Text(service.name)),
                if (service.isAboriginal)
                  Container(
                    padding: const EdgeInsets.symmetric(horizontal: 6, vertical: 2),
                    decoration: BoxDecoration(
                      color: AppColors.ochre.withOpacity(0.2),
                      borderRadius: BorderRadius.circular(6),
                    ),
                    child: Text(
                      "Mob",
                      style: Theme.of(context).textTheme.labelSmall?.copyWith(
                        color: AppColors.ochre,
                        fontWeight: FontWeight.w600,
                      ),
                    ),
                  ),
              ],
            ),
            subtitle: Text(service.description),
            trailing: IconButton(
              icon: Icon(Icons.phone, color: AppColors.primary),
              onPressed: () => _callService(service.phone),
            ),
            onTap: () => _callService(service.phone),
          ),
        )),
      ],
    );
  }
}

class _CentrelinkIssue {
  final String id;
  final String title;
  final String description;
  final IconData icon;

  const _CentrelinkIssue({
    required this.id,
    required this.title,
    required this.description,
    required this.icon,
  });
}

class _QuickTip {
  final String title;
  final String content;

  const _QuickTip({
    required this.title,
    required this.content,
  });
}

class _SupportService {
  final String name;
  final String description;
  final String phone;
  final bool isAboriginal;

  const _SupportService({
    required this.name,
    required this.description,
    required this.phone,
    required this.isAboriginal,
  });
}
