import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/services/voice_scripts.dart';
import '../../../core/widgets/shared_widgets.dart';

class CallCoachPage extends StatefulWidget {
  const CallCoachPage({super.key});

  @override
  State<CallCoachPage> createState() => _CallCoachPageState();
}

class _CallCoachPageState extends State<CallCoachPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _apiClient = ApiClient();
  final VoiceScripts _scripts = VoiceScripts();
  String? _selectedScenario;
  bool _isListening = false;
  bool _isLoading = false;
  String _userGoal = '';
  bool _hasPlayedIntro = false;
  Map<String, dynamic>? _coaching;

  final List<Map<String, dynamic>> _scenarios = [
    {'id': 'centrelink', 'name': 'Centrelink', 'icon': Icons.account_balance},
    {'id': 'housing', 'name': 'Housing', 'icon': Icons.home},
    {'id': 'legal', 'name': 'Legal Aid', 'icon': Icons.gavel},
    {'id': 'school', 'name': 'Schools', 'icon': Icons.school},
    {'id': 'doctor', 'name': 'Doctors', 'icon': Icons.local_hospital},
    {'id': 'employer', 'name': 'Employer', 'icon': Icons.work},
  ];

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    debugPrint('CallCoachPage: Initializing voice service...');
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    
    if (mounted) {
      await Future.delayed(const Duration(milliseconds: 500));
      _playIntro();
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    super.dispose();
  }

  void _playIntro() {
    debugPrint('CallCoachPage: _playIntro called, autoPlay=${_prefs.autoPlayVoice}');
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      _voiceService.speak(_scripts.callCoachIntro);
    }
  }

  Future<void> _startListening() async {
    _voiceService.stop();
    setState(() => _isListening = true);
    
    await _voiceService.speakThenListen(
      prompt: "I'm listening. Tell me what you need to sort out on this call.",
      onResult: (text) {
        setState(() {
          _isListening = false;
          _userGoal = text;
        });
        if (text.isNotEmpty && _prefs.autoPlayVoice) {
          _voiceService.speak("Got it. Let me get some tips ready for you.");
          _getCoachingTips();
        }
      },
      onError: (error) {
        setState(() => _isListening = false);
        if (_prefs.autoPlayVoice) {
          _voiceService.speak("I couldn't hear that clearly. Tap the mic to try again.");
        }
      },
    );
  }

  Future<void> _getCoachingTips() async {
    if (_selectedScenario == null || _userGoal.isEmpty) return;

    setState(() => _isLoading = true);

    try {
      final result = await _apiClient.post('/api/call-coach/prepare', {
        'scenario': _selectedScenario,
        'goal': _userGoal,
      });

      setState(() {
        _coaching = result;
        _isLoading = false;
      });

      if (_coaching != null && _prefs.autoPlayVoice) {
        _speakCoachingIntro();
      }
    } catch (e) {
      setState(() => _isLoading = false);
      _provideDefaultCoaching();
    }
  }

  void _provideDefaultCoaching() {
    setState(() {
      _coaching = {
        'intro': "You're ready for this call.",
        'tips': [
          "Ask them to slow down if they talk too fast",
          "It's okay to say you need time to think",
          "Write down their name and any reference numbers",
          "Ask them to explain anything you don't understand",
          "You can ask for a callback if you need to",
        ],
        'rights': [
          "You have the right to be treated with respect",
          "You can ask for an interpreter if you need one",
          "You can ask for things in writing",
        ],
      };
    });
    if (_prefs.autoPlayVoice) {
      _speakCoachingIntro();
    }
  }

  void _speakCoachingIntro() {
    final intro = _coaching?['intro'] ?? "Here are some tips for your call.";
    _voiceService.speak(
      "$intro Scroll down to see your tips. Tap any tip to hear it read aloud."
    );
  }

  void _speakTip(String tip) {
    _voiceService.stop();
    _voiceService.speak(tip);
  }

  void _speakAllTips() {
    final tips = (_coaching?['tips'] as List?)?.join('. ') ?? '';
    final rights = (_coaching?['rights'] as List?)?.join('. ') ?? '';
    _voiceService.speak(
      "Here are your tips for this call. $tips "
      "And remember your rights. $rights"
    );
  }

  void _startOver() {
    _voiceService.stop();
    setState(() {
      _selectedScenario = null;
      _userGoal = '';
      _coaching = null;
    });
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Starting fresh. Who do you need to call?");
    }
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () {
            _voiceService.stop();
            context.go('/');
          },
        ),
        title: const Text('Call Coach'),
        actions: [
          IconButton(
            icon: const Icon(Icons.volume_up),
            tooltip: 'Read page aloud',
            onPressed: () {
              if (_coaching != null) {
                _voiceService.speak(
                  "You've got coaching tips ready. Scroll down to see them, or tap any tip to hear it."
                );
              } else if (_selectedScenario != null) {
                _voiceService.speak(
                  "You've selected ${_scenarios.firstWhere((s) => s['id'] == _selectedScenario)['name']}. "
                  "Now tap the mic and tell me what you need to sort out."
                );
              } else {
                _voiceService.speak(
                  "Tap one of the buttons to choose who you're calling. "
                  "Centrelink, Housing, Legal Aid, Schools, Doctors, or Employer."
                );
              }
            },
          ),
          const SettingsButton(),
        ],
      ),
      body: CulturalBackground(
        child: ScrollableWithIndicator(
          padding: const EdgeInsets.all(16),
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Text(
                'Got a call to make?',
                style: Theme.of(context).textTheme.headlineSmall,
              ),
              const SizedBox(height: 8),
              Text(
                'Let\'s get you ready. Who do you need to call?',
                style: Theme.of(context).textTheme.bodyMedium,
              ),
              const SizedBox(height: 24),
              Wrap(
                spacing: 12,
                runSpacing: 12,
                children: _scenarios.map((scenario) {
                  final isSelected = _selectedScenario == scenario['id'];
                  return ChoiceChip(
                    label: Row(
                      mainAxisSize: MainAxisSize.min,
                      children: [
                        Icon(
                          scenario['icon'] as IconData,
                          size: 18,
                          color: isSelected ? Colors.white : AppColors.primary,
                        ),
                        const SizedBox(width: 8),
                        Text(scenario['name'] as String),
                      ],
                    ),
                    selected: isSelected,
                    selectedColor: AppColors.primary,
                    onSelected: (selected) {
                      _voiceService.stop();
                      setState(() {
                        _selectedScenario = selected ? scenario['id'] as String : null;
                        _userGoal = '';
                        _coaching = null;
                      });
                      if (selected && _prefs.autoPlayVoice) {
                        _voiceService.speak(
                          "${scenario['name']}. No worries, I'll help you through it. "
                          "Tap the big mic button and tell me what you need to sort out on this call."
                        );
                      }
                    },
                  );
                }).toList(),
              ),
              if (_selectedScenario != null && _coaching == null) ...[
                const SizedBox(height: 32),
                Text(
                  'What do you need to sort out?',
                  style: Theme.of(context).textTheme.titleMedium,
                ),
                const SizedBox(height: 8),
                Text(
                  'Tap the mic and tell me in your own words.',
                  style: Theme.of(context).textTheme.bodySmall,
                ),
                const SizedBox(height: 16),
                if (_userGoal.isEmpty && !_isLoading)
                  Center(
                    child: Column(
                      children: [
                        FloatingActionButton.large(
                          onPressed: _isListening ? null : _startListening,
                          backgroundColor: _isListening ? Colors.red : AppColors.primary,
                          child: Icon(
                            _isListening ? Icons.hearing : Icons.mic,
                            size: 32,
                            color: Colors.white,
                          ),
                        ),
                        const SizedBox(height: 12),
                        Text(
                          _isListening ? 'Listening...' : 'Tap to speak',
                          style: Theme.of(context).textTheme.bodyMedium,
                        ),
                      ],
                    ),
                  )
                else if (_isLoading)
                  Center(
                    child: Column(
                      children: [
                        CircularProgressIndicator(
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                        ),
                        const SizedBox(height: 12),
                        Text(
                          'Getting your tips ready...',
                          style: Theme.of(context).textTheme.bodyMedium,
                        ),
                      ],
                    ),
                  )
                else
                  Card(
                    child: Padding(
                      padding: const EdgeInsets.all(16),
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Row(
                            children: [
                              const Icon(Icons.check_circle, color: Colors.green),
                              const SizedBox(width: 8),
                              Text(
                                'Got it!',
                                style: Theme.of(context).textTheme.titleMedium,
                              ),
                            ],
                          ),
                          const SizedBox(height: 8),
                          Text(_userGoal),
                        ],
                      ),
                    ),
                  ),
              ],
              if (_coaching != null) ...[
                const SizedBox(height: 24),
                _buildCoachingCard(),
                const SizedBox(height: 24),
                Row(
                  children: [
                    Expanded(
                      child: ElevatedButton.icon(
                        onPressed: () {
                          _voiceService.speak(
                            "Alright, you're set. Take a breath. "
                            "Ask them to slow down if you need to - they talk too fast anyway, that's their problem. "
                            "Write down any reference numbers. And it's okay to say you need time to think."
                          );
                        },
                        icon: const Icon(Icons.phone),
                        label: const Text("I'm ready"),
                      ),
                    ),
                  ],
                ),
                const SizedBox(height: 12),
                SizedBox(
                  width: double.infinity,
                  child: OutlinedButton.icon(
                    onPressed: _startOver,
                    icon: const Icon(Icons.refresh),
                    label: const Text('Prepare for a different call'),
                  ),
                ),
              ],
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildCoachingCard() {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final tips = (_coaching?['tips'] as List?) ?? [];
    final rights = (_coaching?['rights'] as List?) ?? [];
    
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Card(
          child: Padding(
            padding: const EdgeInsets.all(16),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Row(
                  children: [
                    Icon(
                      Icons.lightbulb_outline,
                      color: isDark ? AppColors.primaryLight : AppColors.primary,
                    ),
                    const SizedBox(width: 8),
                    Expanded(
                      child: Text(
                        'Your coaching tips',
                        style: Theme.of(context).textTheme.titleMedium?.copyWith(
                          fontWeight: FontWeight.w600,
                        ),
                      ),
                    ),
                    IconButton(
                      icon: Icon(Icons.play_arrow, color: isDark ? AppColors.primaryLight : AppColors.primary),
                      tooltip: 'Read all tips',
                      onPressed: _speakAllTips,
                    ),
                  ],
                ),
                const SizedBox(height: 12),
                ...tips.asMap().entries.map((entry) {
                  final index = entry.key;
                  final tip = entry.value.toString();
                  return InkWell(
                    onTap: () => _speakTip(tip),
                    borderRadius: BorderRadius.circular(8),
                    child: Padding(
                      padding: const EdgeInsets.symmetric(vertical: 8),
                      child: Row(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Container(
                            width: 24,
                            height: 24,
                            decoration: BoxDecoration(
                              color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.1),
                              shape: BoxShape.circle,
                            ),
                            child: Center(
                              child: Text(
                                '${index + 1}',
                                style: TextStyle(
                                  fontSize: 12,
                                  fontWeight: FontWeight.bold,
                                  color: isDark ? AppColors.primaryLight : AppColors.primary,
                                ),
                              ),
                            ),
                          ),
                          const SizedBox(width: 12),
                          Expanded(child: Text(tip)),
                          Icon(
                            Icons.volume_up,
                            size: 18,
                            color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.5),
                          ),
                        ],
                      ),
                    ),
                  );
                }),
              ],
            ),
          ),
        ),
        if (rights.isNotEmpty) ...[
          const SizedBox(height: 16),
          Card(
            color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.1),
            child: Padding(
              padding: const EdgeInsets.all(16),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Row(
                    children: [
                      Icon(
                        Icons.shield,
                        color: isDark ? AppColors.primaryLight : AppColors.primary,
                      ),
                      const SizedBox(width: 8),
                      Text(
                        'Remember your rights',
                        style: Theme.of(context).textTheme.titleMedium?.copyWith(
                          fontWeight: FontWeight.w600,
                        ),
                      ),
                    ],
                  ),
                  const SizedBox(height: 12),
                  ...rights.map((right) => InkWell(
                    onTap: () => _speakTip(right.toString()),
                    child: Padding(
                      padding: const EdgeInsets.symmetric(vertical: 4),
                      child: Row(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Icon(Icons.check, size: 18, color: Colors.green),
                          const SizedBox(width: 8),
                          Expanded(child: Text(right.toString())),
                        ],
                      ),
                    ),
                  )),
                ],
              ),
            ),
          ),
        ],
      ],
    );
  }
}
