import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/services/api_client.dart';
import '../../../core/widgets/shared_widgets.dart';

class AppointmentsPage extends StatefulWidget {
  const AppointmentsPage({super.key});

  @override
  State<AppointmentsPage> createState() => _AppointmentsPageState();
}

class _AppointmentsPageState extends State<AppointmentsPage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final ApiClient _api = ApiClient();
  
  final _purposeController = TextEditingController();
  final _concernsController = TextEditingController();
  
  bool _hasPlayedIntro = false;
  bool _isLoading = false;
  String? _selectedAgency;
  Map<String, dynamic>? _preparation;
  String? _error;

  final List<Map<String, String>> _agencies = [
    {'id': 'centrelink', 'name': 'Centrelink', 'icon': 'government'},
    {'id': 'housing', 'name': 'Housing Services', 'icon': 'home'},
    {'id': 'hospital', 'name': 'Hospital/Doctor', 'icon': 'health'},
    {'id': 'school', 'name': 'School Meeting', 'icon': 'school'},
    {'id': 'employment', 'name': 'Job Services', 'icon': 'work'},
    {'id': 'legal', 'name': 'Legal Services', 'icon': 'gavel'},
    {'id': 'bank', 'name': 'Bank', 'icon': 'bank'},
    {'id': 'court', 'name': 'Court/Tribunal', 'icon': 'court'},
    {'id': 'other', 'name': 'Other', 'icon': 'other'},
  ];

  @override
  void initState() {
    super.initState();
    _initializeAndPlay();
  }

  Future<void> _initializeAndPlay() async {
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    if (mounted && !_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      await Future.delayed(const Duration(milliseconds: 300));
      _voiceService.speak(
        "Hey mob, you've got an appointment coming up? Let's get you ready. "
        "Pick who you're meeting with - Centrelink, housing, doctor, school, or whoever. "
        "I'll tell you your rights, what to bring, and what questions to ask. "
        "Knowledge is power, and I want you walking in there feeling strong."
      );
    }
  }

  @override
  void dispose() {
    _voiceService.stop();
    _purposeController.dispose();
    _concernsController.dispose();
    super.dispose();
  }

  IconData _getIconForType(String iconType) {
    switch (iconType) {
      case 'government': return Icons.account_balance;
      case 'home': return Icons.home_outlined;
      case 'health': return Icons.local_hospital_outlined;
      case 'school': return Icons.school_outlined;
      case 'work': return Icons.work_outline;
      case 'gavel': return Icons.gavel;
      case 'bank': return Icons.account_balance_wallet;
      case 'court': return Icons.balance;
      default: return Icons.event_outlined;
    }
  }

  Future<void> _prepareForAppointment() async {
    if (_selectedAgency == null) {
      setState(() => _error = 'Please select who the appointment is with');
      return;
    }
    
    setState(() {
      _isLoading = true;
      _error = null;
    });
    
    try {
      final result = await _api.post('/api/appointments/prepare', {
        'appointmentType': _selectedAgency,
        'agency': _agencies.firstWhere((a) => a['id'] == _selectedAgency)['name'],
        'purpose': _purposeController.text,
        'concerns': _concernsController.text,
      });
      
      setState(() {
        _preparation = result;
        _isLoading = false;
      });
      
      if (_prefs.autoPlayVoice && result['whatToExpect'] != null) {
        _voiceService.speak(result['whatToExpect']);
      }
    } catch (e) {
      setState(() {
        _error = 'Having trouble preparing. Try again?';
        _isLoading = false;
      });
    }
  }

  void _reset() {
    setState(() {
      _selectedAgency = null;
      _preparation = null;
      _purposeController.clear();
      _concernsController.clear();
    });
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;

    return Scaffold(
      body: CulturalBackground(
        child: SafeArea(
          child: Column(
            children: [
              Padding(
                padding: const EdgeInsets.all(16.0),
                child: Row(
                  children: [
                    IconButton(
                      icon: const Icon(Icons.arrow_back),
                      onPressed: () {
                        if (_preparation != null) {
                          _reset();
                        } else {
                          context.go('/');
                        }
                      },
                    ),
                    const Spacer(),
                    IconButton(
                      icon: const Icon(Icons.volume_up),
                      onPressed: () {
                        if (_preparation != null) {
                          _voiceService.speak(_preparation!['whatToExpect'] ?? 'Here is what to expect');
                        } else {
                          _voiceService.speak("Pick who your appointment is with and I'll help you get ready");
                        }
                      },
                    ),
                    IconButton(
                      icon: const Icon(Icons.settings_outlined),
                      onPressed: () => context.go('/settings'),
                    ),
                  ],
                ),
              ),
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 20.0),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(
                          Icons.event_available,
                          color: isDark ? AppColors.primaryLight : AppColors.primary,
                          size: 28,
                        ),
                        const SizedBox(width: 12),
                        Text(
                          'Appointment Helper',
                          style: Theme.of(context).textTheme.headlineMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(
                      _preparation == null 
                        ? 'Get ready for your appointment'
                        : 'Here\'s how to prepare',
                      style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                        color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                      ),
                    ),
                  ],
                ),
              ),
              const SizedBox(height: 20),
              Expanded(
                child: _preparation == null
                  ? _buildInputSection(isDark)
                  : _buildResultSection(isDark),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildInputSection(bool isDark) {
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(
            'Who is the appointment with?',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 12),
          Wrap(
            spacing: 8,
            runSpacing: 8,
            children: _agencies.map((agency) {
              final isSelected = _selectedAgency == agency['id'];
              return FilterChip(
                selected: isSelected,
                label: Row(
                  mainAxisSize: MainAxisSize.min,
                  children: [
                    Icon(
                      _getIconForType(agency['icon']!),
                      size: 18,
                      color: isSelected 
                        ? Colors.white 
                        : (isDark ? AppColors.primaryLight : AppColors.primary),
                    ),
                    const SizedBox(width: 8),
                    Text(agency['name']!),
                  ],
                ),
                onSelected: (selected) {
                  setState(() => _selectedAgency = selected ? agency['id'] : null);
                },
              );
            }).toList(),
          ),
          const SizedBox(height: 24),
          
          Text(
            'What is the appointment about?',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          TextField(
            controller: _purposeController,
            decoration: InputDecoration(
              hintText: 'e.g., Review my payment, discuss housing application',
              border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
              suffixIcon: MicButton(onResult: (t) => _purposeController.text = t),
            ),
            maxLines: 2,
          ),
          const SizedBox(height: 16),
          
          Text(
            'Any concerns or worries? (optional)',
            style: Theme.of(context).textTheme.titleMedium?.copyWith(
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 8),
          TextField(
            controller: _concernsController,
            decoration: InputDecoration(
              hintText: 'e.g., Worried they\'ll cut my payment',
              border: OutlineInputBorder(borderRadius: BorderRadius.circular(12)),
              suffixIcon: MicButton(onResult: (t) => _concernsController.text = t),
            ),
            maxLines: 2,
          ),
          const SizedBox(height: 24),
          
          if (_error != null)
            Padding(
              padding: const EdgeInsets.only(bottom: 16),
              child: Text(_error!, style: TextStyle(color: Colors.red[400])),
            ),
          
          SizedBox(
            width: double.infinity,
            child: ElevatedButton.icon(
              onPressed: _isLoading ? null : _prepareForAppointment,
              icon: _isLoading 
                ? const SizedBox(
                    width: 20, height: 20,
                    child: CircularProgressIndicator(strokeWidth: 2),
                  )
                : const Icon(Icons.checklist),
              label: Text(_isLoading ? 'Preparing...' : 'Help Me Prepare'),
              style: ElevatedButton.styleFrom(
                padding: const EdgeInsets.symmetric(vertical: 16),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
            ),
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }

  Widget _buildResultSection(bool isDark) {
    final prep = _preparation!;
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.symmetric(horizontal: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          _buildSection(
            'What to Bring',
            Icons.inventory_2_outlined,
            prep['whatToBring'] as List? ?? [],
            isDark,
          ),
          
          if (prep['whatToExpect'] != null) ...[
            Card(
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.info_outline, color: isDark ? AppColors.primaryLight : AppColors.primary),
                        const SizedBox(width: 8),
                        Text(
                          'What to Expect',
                          style: Theme.of(context).textTheme.titleMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                        const Spacer(),
                        IconButton(
                          icon: const Icon(Icons.volume_up, size: 20),
                          onPressed: () => _voiceService.speak(prep['whatToExpect']),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(prep['whatToExpect']),
                  ],
                ),
              ),
            ),
            const SizedBox(height: 16),
          ],
          
          _buildSection(
            'Your Rights',
            Icons.shield_outlined,
            prep['yourRights'] as List? ?? [],
            isDark,
            color: Colors.green,
          ),
          
          _buildSection(
            'Things You Can Say',
            Icons.chat_bubble_outline,
            prep['phrasesToUse'] as List? ?? [],
            isDark,
            color: Colors.blue,
          ),
          
          _buildSection(
            'Questions to Ask',
            Icons.help_outline,
            prep['questionsToAsk'] as List? ?? [],
            isDark,
          ),
          
          _buildSection(
            'Staying Calm',
            Icons.self_improvement,
            prep['stayingCalm'] as List? ?? [],
            isDark,
            color: Colors.purple,
          ),
          
          if (prep['backup'] != null) ...[
            Card(
              color: Colors.orange.withOpacity(0.1),
              child: Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Row(
                      children: [
                        Icon(Icons.warning_amber, color: Colors.orange[400]),
                        const SizedBox(width: 8),
                        Text(
                          'If Things Don\'t Go Well',
                          style: Theme.of(context).textTheme.titleMedium?.copyWith(
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                    const SizedBox(height: 8),
                    Text(prep['backup']),
                  ],
                ),
              ),
            ),
            const SizedBox(height: 16),
          ],
          
          SizedBox(
            width: double.infinity,
            child: OutlinedButton.icon(
              onPressed: _reset,
              icon: const Icon(Icons.refresh),
              label: const Text('Prepare for Another Appointment'),
              style: OutlinedButton.styleFrom(
                padding: const EdgeInsets.symmetric(vertical: 16),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
            ),
          ),
          const SizedBox(height: 24),
        ],
      ),
    );
  }

  Widget _buildSection(String title, IconData icon, List items, bool isDark, {Color? color}) {
    if (items.isEmpty) return const SizedBox();
    
    final iconColor = color ?? (isDark ? AppColors.primaryLight : AppColors.primary);
    
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Row(
          children: [
            Icon(icon, color: iconColor, size: 20),
            const SizedBox(width: 8),
            Text(
              title,
              style: Theme.of(context).textTheme.titleMedium?.copyWith(
                fontWeight: FontWeight.bold,
              ),
            ),
          ],
        ),
        const SizedBox(height: 8),
        ...items.map((item) => Padding(
          padding: const EdgeInsets.only(bottom: 8),
          child: Row(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Icon(Icons.check, color: iconColor, size: 18),
              const SizedBox(width: 8),
              Expanded(child: Text(item.toString())),
            ],
          ),
        )),
        const SizedBox(height: 16),
      ],
    );
  }
}
