import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:hive_flutter/hive_flutter.dart';
import '../../../core/theme/app_theme.dart';
import '../../../core/services/elevenlabs_service.dart';
import '../../../core/services/preferences_service.dart';
import '../../../core/widgets/shared_widgets.dart';

class AdvocatePage extends StatefulWidget {
  const AdvocatePage({super.key});

  @override
  State<AdvocatePage> createState() => _AdvocatePageState();
}

class _AdvocatePageState extends State<AdvocatePage> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  final PreferencesService _prefs = PreferencesService();
  final TextEditingController _summaryController = TextEditingController();
  final TextEditingController _factsController = TextEditingController();
  
  bool _hasConsented = false;
  bool _isListening = false;
  String _activeField = '';
  List<Map<String, dynamic>> _cases = [];
  bool _hasPlayedIntro = false;

  @override
  void initState() {
    super.initState();
    _initializeAndLoad();
  }

  Future<void> _initializeAndLoad() async {
    debugPrint('AdvocatePage: Initializing voice service...');
    await _voiceService.initialize();
    _voiceService.setPersona(_prefs.voicePersona);
    await _loadCases();
  }

  @override
  void dispose() {
    _summaryController.dispose();
    _factsController.dispose();
    _voiceService.stop();
    super.dispose();
  }

  void _playIntro() {
    debugPrint('AdvocatePage: _playIntro called, autoPlay=${_prefs.autoPlayVoice}');
    if (!_hasPlayedIntro && _prefs.autoPlayVoice) {
      _hasPlayedIntro = true;
      if (_hasConsented) {
        if (_cases.isEmpty) {
          _voiceService.speak(
            "Your advocate space is ready. Add your first case below. "
            "Tap the mic to speak, or type in the box. "
            "Everything stays on your phone. No one else can see it."
          );
        } else {
          _voiceService.speak(
            "Your advocate space. You have ${_cases.length} ${_cases.length == 1 ? 'case' : 'cases'} saved. "
            "Scroll down to see them, or add a new one at the top."
          );
        }
      } else {
        _voiceService.speak(
          "This is your private advocate space. "
          "You can keep track of important things here, like cases with Centrelink, housing, or legal matters. "
          "Nothing leaves your phone. Tap the button when you're ready to start."
        );
      }
    }
  }

  Future<void> _loadCases() async {
    final box = await Hive.openBox('advocate_cases');
    final cases = box.get('cases', defaultValue: <Map<String, dynamic>>[]);
    final consented = box.get('consented', defaultValue: false);
    setState(() {
      _cases = List<Map<String, dynamic>>.from(cases);
      _hasConsented = consented;
    });
    Future.delayed(const Duration(milliseconds: 300), _playIntro);
  }

  Future<void> _saveCase() async {
    if (_summaryController.text.isEmpty) {
      if (_prefs.autoPlayVoice) {
        _voiceService.speak("Tell me what this case is about first. Tap the mic to speak.");
      }
      return;
    }
    
    final newCase = {
      'id': DateTime.now().millisecondsSinceEpoch.toString(),
      'summary': _summaryController.text,
      'facts': _factsController.text,
      'createdAt': DateTime.now().toIso8601String(),
    };
    
    _cases.add(newCase);
    
    final box = await Hive.openBox('advocate_cases');
    await box.put('cases', _cases);
    
    _summaryController.clear();
    _factsController.clear();
    
    setState(() {});
    
    await _voiceService.speak("Saved! Your case is stored safely on your phone. You can find it below.");
  }

  Future<void> _deleteCase(String id) async {
    _cases.removeWhere((c) => c['id'] == id);
    
    final box = await Hive.openBox('advocate_cases');
    await box.put('cases', _cases);
    
    setState(() {});
    
    if (_prefs.autoPlayVoice) {
      _voiceService.speak("Case deleted.");
    }
  }

  Future<void> _clearAllData() async {
    final box = await Hive.openBox('advocate_cases');
    await box.clear();
    setState(() {
      _cases = [];
      _hasConsented = false;
    });
    await _voiceService.speak("All your data has been cleared. Your advocate space is empty now.");
  }

  Future<void> _startListening(String field, TextEditingController controller) async {
    _voiceService.stop();
    setState(() {
      _isListening = true;
      _activeField = field;
    });
    
    String prompt;
    if (field == 'summary') {
      prompt = "What's this case about? For example, housing complaint, or Centrelink payment issue.";
    } else {
      prompt = "Tell me the key facts to remember about this case.";
    }
    
    await _voiceService.speakThenListen(
      prompt: prompt,
      onResult: (text) {
        setState(() {
          _isListening = false;
          _activeField = '';
          controller.text = controller.text.isEmpty ? text : '${controller.text} $text';
        });
        if (_prefs.autoPlayVoice && text.isNotEmpty) {
          if (field == 'summary') {
            _voiceService.speak("Got it. Now add any key facts below, or tap Save Case if you're done.");
          } else {
            _voiceService.speak("Got it. Tap the mic to add more, or tap Save Case when you're ready.");
          }
        }
      },
      onError: (error) {
        setState(() {
          _isListening = false;
          _activeField = '';
        });
        if (_prefs.autoPlayVoice) {
          _voiceService.speak("I couldn't hear that clearly. Tap the mic to try again.");
        }
      },
    );
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () {
            _voiceService.stop();
            context.go('/');
          },
        ),
        title: const Text('My Advocate'),
        actions: [
          IconButton(
            icon: const Icon(Icons.volume_up),
            tooltip: 'Read page aloud',
            onPressed: () {
              if (_hasConsented) {
                _voiceService.speak(
                  _cases.isEmpty
                      ? "Your advocate space is empty. Add your first case below."
                      : "You have ${_cases.length} ${_cases.length == 1 ? 'case' : 'cases'} saved. Tap any case to hear it read aloud."
                );
              } else {
                _voiceService.speak(
                  "This is your private space to keep track of cases. Tap the button to get started."
                );
              }
            },
          ),
          if (_hasConsented)
            IconButton(
              icon: const Icon(Icons.delete_outline),
              onPressed: () => _showClearDataDialog(),
            ),
          const SettingsButton(),
        ],
      ),
      body: CulturalBackground(
        child: _hasConsented ? _buildMainContent() : _buildConsentScreen(),
      ),
    );
  }

  Widget _buildConsentScreen() {
    return ScrollableWithIndicator(
      padding: const EdgeInsets.all(24),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Icon(
            Icons.folder_shared_outlined,
            size: 64,
            color: AppColors.primary,
          ),
          const SizedBox(height: 24),
          Text(
            'Keep track of your cases',
            style: Theme.of(context).textTheme.headlineSmall,
          ),
          const SizedBox(height: 12),
          Text(
            'This helps you remember important details when dealing with services like Centrelink, housing, or legal matters.',
            style: Theme.of(context).textTheme.bodyLarge,
          ),
          const SizedBox(height: 24),
          Card(
            child: Padding(
              padding: const EdgeInsets.all(16),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Row(
                    children: [
                      Icon(Icons.lock_outline, color: AppColors.primary),
                      const SizedBox(width: 8),
                      Expanded(
                        child: Text(
                          'Your data stays on your phone',
                          style: Theme.of(context).textTheme.titleMedium,
                        ),
                      ),
                      IconButton(
                        icon: const Icon(Icons.volume_up),
                        onPressed: () {
                          _voiceService.speak(
                            "Your data stays on your phone. Nothing is sent to the internet. You can delete everything anytime."
                          );
                        },
                      ),
                    ],
                  ),
                  const SizedBox(height: 8),
                  Text(
                    'Nothing is sent to the internet. You can delete everything anytime.',
                    style: Theme.of(context).textTheme.bodyMedium,
                  ),
                ],
              ),
            ),
          ),
          const SizedBox(height: 32),
          SizedBox(
            width: double.infinity,
            child: ElevatedButton(
              onPressed: () async {
                final box = await Hive.openBox('advocate_cases');
                await box.put('consented', true);
                setState(() => _hasConsented = true);
                _voiceService.speak("Great, let's get started. Add your first case below. Just tell me what it's about.");
              },
              child: const Text('I understand, let\'s start'),
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildMainContent() {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return ScrollableWithIndicator(
      padding: const EdgeInsets.all(16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(
            'Add a new case',
            style: Theme.of(context).textTheme.titleMedium,
          ),
          const SizedBox(height: 4),
          Text(
            'Keep track of something important',
            style: Theme.of(context).textTheme.bodySmall,
          ),
          const SizedBox(height: 12),
          _buildVoiceTextField(
            controller: _summaryController,
            hint: 'What\'s this about? (e.g., Housing complaint)',
            field: 'summary',
            maxLines: 2,
          ),
          const SizedBox(height: 12),
          _buildVoiceTextField(
            controller: _factsController,
            hint: 'Key facts to remember...',
            field: 'facts',
            maxLines: 4,
          ),
          const SizedBox(height: 16),
          SizedBox(
            width: double.infinity,
            child: ElevatedButton.icon(
              onPressed: _saveCase,
              icon: const Icon(Icons.save),
              label: const Text('Save Case'),
            ),
          ),
          if (_cases.isNotEmpty) ...[
            const SizedBox(height: 32),
            Row(
              children: [
                Expanded(
                  child: Text(
                    'Your Cases (${_cases.length})',
                    style: Theme.of(context).textTheme.titleMedium,
                  ),
                ),
                TextButton.icon(
                  onPressed: () {
                    final allCases = _cases.map((c) => '${c['summary']}. ${c['facts'] ?? ''}').join('. Next case: ');
                    _voiceService.speak("Your saved cases: $allCases");
                  },
                  icon: const Icon(Icons.play_arrow, size: 18),
                  label: const Text('Read all', style: TextStyle(fontSize: 12)),
                ),
              ],
            ),
            const SizedBox(height: 12),
            ..._cases.reversed.map((c) => _buildCaseCard(c)),
          ] else ...[
            const SizedBox(height: 32),
            Card(
              child: Padding(
                padding: const EdgeInsets.all(24),
                child: Column(
                  children: [
                    Icon(
                      Icons.folder_open,
                      size: 48,
                      color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.5),
                    ),
                    const SizedBox(height: 12),
                    Text(
                      'No cases yet',
                      style: Theme.of(context).textTheme.titleMedium,
                    ),
                    const SizedBox(height: 4),
                    Text(
                      'Add your first case above',
                      style: Theme.of(context).textTheme.bodySmall,
                    ),
                  ],
                ),
              ),
            ),
          ],
        ],
      ),
    );
  }

  Widget _buildVoiceTextField({
    required TextEditingController controller,
    required String hint,
    required String field,
    int maxLines = 4,
  }) {
    final isActive = _isListening && _activeField == field;
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Stack(
      children: [
        TextField(
          controller: controller,
          maxLines: maxLines,
          decoration: InputDecoration(
            hintText: hint,
          ),
        ),
        Positioned(
          right: 8,
          top: 8,
          child: FloatingActionButton.small(
            onPressed: isActive ? null : () => _startListening(field, controller),
            backgroundColor: isActive ? Colors.red : (isDark ? AppColors.primaryLight : AppColors.primary),
            child: Icon(
              isActive ? Icons.hearing : Icons.mic,
              color: Colors.white,
            ),
          ),
        ),
      ],
    );
  }

  Widget _buildCaseCard(Map<String, dynamic> caseData) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Dismissible(
      key: Key(caseData['id']),
      direction: DismissDirection.endToStart,
      background: Container(
        alignment: Alignment.centerRight,
        padding: const EdgeInsets.only(right: 20),
        color: Colors.red,
        child: const Icon(Icons.delete, color: Colors.white),
      ),
      onDismissed: (_) => _deleteCase(caseData['id']),
      child: Card(
        margin: const EdgeInsets.only(bottom: 12),
        child: InkWell(
          onTap: () {
            final summary = caseData['summary'] ?? '';
            final facts = caseData['facts'] ?? '';
            _voiceService.speak('$summary. $facts');
          },
          borderRadius: BorderRadius.circular(12),
          child: Padding(
            padding: const EdgeInsets.all(16),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Row(
                  children: [
                    Expanded(
                      child: Text(
                        caseData['summary'] ?? '',
                        style: Theme.of(context).textTheme.titleSmall?.copyWith(
                          fontWeight: FontWeight.w600,
                        ),
                      ),
                    ),
                    Icon(
                      Icons.volume_up,
                      size: 18,
                      color: (isDark ? AppColors.primaryLight : AppColors.primary).withOpacity(0.5),
                    ),
                  ],
                ),
                if (caseData['facts']?.isNotEmpty == true) ...[
                  const SizedBox(height: 8),
                  Text(
                    caseData['facts'],
                    style: Theme.of(context).textTheme.bodyMedium,
                  ),
                ],
                const SizedBox(height: 8),
                Text(
                  'Swipe left to delete',
                  style: Theme.of(context).textTheme.bodySmall?.copyWith(
                    fontStyle: FontStyle.italic,
                    color: Colors.grey,
                  ),
                ),
              ],
            ),
          ),
        ),
      ),
    );
  }

  void _showClearDataDialog() {
    _voiceService.speak("This will delete all your saved cases. Are you sure?");
    
    showDialog(
      context: context,
      builder: (context) => AlertDialog(
        title: const Text('Clear All Data?'),
        content: const Text(
          'This will delete all your saved cases. This cannot be undone.',
        ),
        actions: [
          TextButton(
            onPressed: () {
              Navigator.pop(context);
              if (_prefs.autoPlayVoice) {
                _voiceService.speak("Cancelled. Your cases are safe.");
              }
            },
            child: const Text('Cancel'),
          ),
          TextButton(
            onPressed: () {
              Navigator.pop(context);
              _clearAllData();
            },
            style: TextButton.styleFrom(foregroundColor: Colors.red),
            child: const Text('Delete Everything'),
          ),
        ],
      ),
    );
  }
}
