import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:speech_to_text/speech_to_text.dart' as stt;
import '../services/elevenlabs_service.dart';
import '../services/preferences_service.dart';
import '../theme/app_theme.dart';

/// Universal settings button that can be added to any page's AppBar actions
class SettingsButton extends StatelessWidget {
  final VoidCallback? onPressed;
  
  const SettingsButton({super.key, this.onPressed});
  
  @override
  Widget build(BuildContext context) {
    final voiceService = ElevenLabsService();
    final prefs = PreferencesService();
    
    return IconButton(
      icon: const Icon(Icons.settings_outlined, size: 22),
      tooltip: 'Settings',
      onPressed: onPressed ?? () {
        voiceService.stop();
        if (prefs.autoPlayVoice) {
          voiceService.speak("Opening settings");
        }
        context.go('/settings');
      },
    );
  }
}

/// Listen button for text-to-speech - reads text aloud when tapped
class ListenButton extends StatefulWidget {
  final String? text;
  final VoidCallback? onPressed;
  final double size;
  final Color? color;
  
  const ListenButton({
    super.key,
    this.text,
    this.onPressed,
    this.size = 24,
    this.color,
  });
  
  @override
  State<ListenButton> createState() => _ListenButtonState();
}

class _ListenButtonState extends State<ListenButton> {
  final ElevenLabsService _voiceService = ElevenLabsService();
  bool _isSpeaking = false;
  
  Future<void> _toggleSpeech() async {
    if (widget.onPressed != null) {
      widget.onPressed!();
      return;
    }
    
    if (_isSpeaking) {
      await _voiceService.stop();
      if (mounted) setState(() => _isSpeaking = false);
    } else if (widget.text != null) {
      setState(() => _isSpeaking = true);
      await _voiceService.speak(widget.text!);
      if (mounted) setState(() => _isSpeaking = false);
    }
  }
  
  @override
  void dispose() {
    _voiceService.stop();
    super.dispose();
  }
  
  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final defaultColor = isDark ? AppColors.primaryLight : AppColors.primary;
    final iconColor = widget.color ?? defaultColor;
    
    return IconButton(
      icon: Icon(
        _isSpeaking ? Icons.stop_circle : Icons.volume_up,
        size: widget.size,
        color: iconColor,
      ),
      onPressed: _toggleSpeech,
      tooltip: _isSpeaking ? 'Stop reading' : 'Read aloud',
    );
  }
}

/// Scroll indicator that shows when there's more content below
/// Positions itself at the bottom center without covering text
class ScrollIndicator extends StatefulWidget {
  final ScrollController scrollController;
  final double bottomPadding;
  
  const ScrollIndicator({
    super.key,
    required this.scrollController,
    this.bottomPadding = 16,
  });
  
  @override
  State<ScrollIndicator> createState() => _ScrollIndicatorState();
}

class _ScrollIndicatorState extends State<ScrollIndicator> with SingleTickerProviderStateMixin {
  bool _showIndicator = false;
  late AnimationController _animationController;
  late Animation<double> _bounceAnimation;
  
  @override
  void initState() {
    super.initState();
    _animationController = AnimationController(
      duration: const Duration(milliseconds: 1200),
      vsync: this,
    )..repeat(reverse: true);
    
    _bounceAnimation = Tween<double>(begin: 0, end: 6).animate(
      CurvedAnimation(parent: _animationController, curve: Curves.easeInOut),
    );
    
    widget.scrollController.addListener(_checkScrollPosition);
    WidgetsBinding.instance.addPostFrameCallback((_) => _checkScrollPosition());
  }
  
  @override
  void dispose() {
    widget.scrollController.removeListener(_checkScrollPosition);
    _animationController.dispose();
    super.dispose();
  }
  
  void _checkScrollPosition() {
    if (!widget.scrollController.hasClients) return;
    
    final maxScroll = widget.scrollController.position.maxScrollExtent;
    final currentScroll = widget.scrollController.position.pixels;
    
    // Show indicator if there's more than 50 pixels to scroll
    final shouldShow = maxScroll > 50 && (maxScroll - currentScroll) > 50;
    
    if (shouldShow != _showIndicator) {
      setState(() => _showIndicator = shouldShow);
    }
  }
  
  void _scrollDown() {
    widget.scrollController.animateTo(
      widget.scrollController.position.maxScrollExtent,
      duration: const Duration(milliseconds: 500),
      curve: Curves.easeOut,
    );
  }
  
  @override
  Widget build(BuildContext context) {
    if (!_showIndicator) return const SizedBox.shrink();
    
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final primaryColor = isDark ? AppColors.primaryLight : AppColors.primary;
    final ochreColor = AppColors.ochre;
    
    return Positioned(
      bottom: widget.bottomPadding,
      left: 0,
      right: 0,
      child: Center(
        child: AnimatedBuilder(
          animation: _bounceAnimation,
          builder: (context, child) {
            return Transform.translate(
              offset: Offset(0, _bounceAnimation.value),
              child: child,
            );
          },
          child: GestureDetector(
            onTap: _scrollDown,
            child: Container(
              padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 10),
              decoration: BoxDecoration(
                gradient: LinearGradient(
                  begin: Alignment.topLeft,
                  end: Alignment.bottomRight,
                  colors: [
                    primaryColor.withOpacity(0.95),
                    ochreColor.withOpacity(0.9),
                  ],
                ),
                borderRadius: BorderRadius.circular(24),
                border: Border.all(
                  color: Colors.white.withOpacity(0.3),
                  width: 1.5,
                ),
                boxShadow: [
                  BoxShadow(
                    color: primaryColor.withOpacity(0.4),
                    blurRadius: 12,
                    offset: const Offset(0, 4),
                  ),
                ],
              ),
              child: Row(
                mainAxisSize: MainAxisSize.min,
                children: [
                  // Indigenous dot pattern (3 dots)
                  Row(
                    mainAxisSize: MainAxisSize.min,
                    children: List.generate(3, (i) => Container(
                      width: 6,
                      height: 6,
                      margin: EdgeInsets.only(right: i < 2 ? 3 : 0),
                      decoration: BoxDecoration(
                        color: Colors.white.withOpacity(0.9),
                        shape: BoxShape.circle,
                      ),
                    )),
                  ),
                  const SizedBox(width: 8),
                  const Icon(
                    Icons.keyboard_double_arrow_down,
                    size: 20,
                    color: Colors.white,
                  ),
                  const SizedBox(width: 6),
                  const Text(
                    'Keep scrollin\'',
                    style: TextStyle(
                      fontSize: 14,
                      fontWeight: FontWeight.w600,
                      color: Colors.white,
                      letterSpacing: 0.3,
                    ),
                  ),
                  const SizedBox(width: 8),
                  // Indigenous dot pattern (3 dots)
                  Row(
                    mainAxisSize: MainAxisSize.min,
                    children: List.generate(3, (i) => Container(
                      width: 6,
                      height: 6,
                      margin: EdgeInsets.only(right: i < 2 ? 3 : 0),
                      decoration: BoxDecoration(
                        color: Colors.white.withOpacity(0.9),
                        shape: BoxShape.circle,
                      ),
                    )),
                  ),
                ],
              ),
            ),
          ),
        ),
      ),
    );
  }
}

/// Helper widget to wrap scrollable content with a scroll indicator
class ScrollableWithIndicator extends StatefulWidget {
  final Widget child;
  final ScrollController? controller;
  final EdgeInsets? padding;
  final double indicatorBottomPadding;
  
  const ScrollableWithIndicator({
    super.key,
    required this.child,
    this.controller,
    this.padding,
    this.indicatorBottomPadding = 16,
  });
  
  @override
  State<ScrollableWithIndicator> createState() => _ScrollableWithIndicatorState();
}

class _ScrollableWithIndicatorState extends State<ScrollableWithIndicator> {
  late ScrollController _scrollController;
  
  @override
  void initState() {
    super.initState();
    _scrollController = widget.controller ?? ScrollController();
  }
  
  @override
  void dispose() {
    if (widget.controller == null) {
      _scrollController.dispose();
    }
    super.dispose();
  }
  
  @override
  Widget build(BuildContext context) {
    return Stack(
      children: [
        SingleChildScrollView(
          controller: _scrollController,
          padding: widget.padding,
          child: widget.child,
        ),
        ScrollIndicator(
          scrollController: _scrollController,
          bottomPadding: widget.indicatorBottomPadding,
        ),
      ],
    );
  }
}

/// Indigenous-styled floating home button - a friendly guide back to the hub
/// Designed as a culturally meaningful boomerang motif representing return/home
class YarnHomeButton extends StatefulWidget {
  final bool showLabel;
  
  const YarnHomeButton({
    super.key,
    this.showLabel = true,
  });

  @override
  State<YarnHomeButton> createState() => _YarnHomeButtonState();
}

class _YarnHomeButtonState extends State<YarnHomeButton> with SingleTickerProviderStateMixin {
  late AnimationController _controller;
  late Animation<double> _pulseAnimation;
  
  @override
  void initState() {
    super.initState();
    _controller = AnimationController(
      duration: const Duration(milliseconds: 2000),
      vsync: this,
    )..repeat(reverse: true);
    
    _pulseAnimation = Tween<double>(begin: 1.0, end: 1.08).animate(
      CurvedAnimation(parent: _controller, curve: Curves.easeInOut),
    );
  }
  
  @override
  void dispose() {
    _controller.dispose();
    super.dispose();
  }
  
  @override
  Widget build(BuildContext context) {
    final voiceService = ElevenLabsService();
    final prefs = PreferencesService();
    
    return GestureDetector(
      onTap: () {
        voiceService.stop();
        if (prefs.autoPlayVoice) {
          voiceService.speak("Heading home");
        }
        context.go('/');
      },
      child: ScaleTransition(
        scale: _pulseAnimation,
        child: Container(
          padding: EdgeInsets.symmetric(
            horizontal: widget.showLabel ? 16 : 12,
            vertical: 10,
          ),
          decoration: BoxDecoration(
            gradient: LinearGradient(
              begin: Alignment.topLeft,
              end: Alignment.bottomRight,
              colors: [
                AppColors.ochre,
                AppColors.terracotta,
                AppColors.earth,
              ],
            ),
            borderRadius: BorderRadius.circular(24),
            boxShadow: [
              BoxShadow(
                color: AppColors.ochre.withOpacity(0.4),
                blurRadius: 12,
                offset: const Offset(0, 4),
              ),
              BoxShadow(
                color: AppColors.terracotta.withOpacity(0.3),
                blurRadius: 20,
                spreadRadius: 2,
              ),
            ],
          ),
          child: Row(
            mainAxisSize: MainAxisSize.min,
            children: [
              SizedBox(
                width: 28,
                height: 28,
                child: CustomPaint(
                  painter: _BoomerangPainter(
                    color: Colors.white,
                    arrowColor: Colors.white.withOpacity(0.9),
                  ),
                ),
              ),
              if (widget.showLabel) ...[
                const SizedBox(width: 8),
                const Text(
                  'Home',
                  style: TextStyle(
                    color: Colors.white,
                    fontWeight: FontWeight.w700,
                    fontSize: 14,
                    letterSpacing: 0.5,
                  ),
                ),
              ],
            ],
          ),
        ),
      ),
    );
  }
}

/// Custom painter for a boomerang with return arrow
class _BoomerangPainter extends CustomPainter {
  final Color color;
  final Color arrowColor;
  
  _BoomerangPainter({
    required this.color,
    required this.arrowColor,
  });
  
  @override
  void paint(Canvas canvas, Size size) {
    final paint = Paint()
      ..color = color
      ..style = PaintingStyle.stroke
      ..strokeWidth = 2.5
      ..strokeCap = StrokeCap.round
      ..strokeJoin = StrokeJoin.round;
    
    final center = Offset(size.width / 2, size.height / 2);
    final radius = size.width * 0.38;
    
    final boomerangPath = Path();
    boomerangPath.moveTo(center.dx - radius * 0.3, center.dy - radius * 0.8);
    boomerangPath.quadraticBezierTo(
      center.dx - radius * 1.1, center.dy - radius * 0.2,
      center.dx - radius * 0.5, center.dy + radius * 0.6,
    );
    boomerangPath.quadraticBezierTo(
      center.dx - radius * 0.2, center.dy + radius * 0.9,
      center.dx + radius * 0.2, center.dy + radius * 0.7,
    );
    boomerangPath.quadraticBezierTo(
      center.dx + radius * 0.5, center.dy + radius * 0.5,
      center.dx + radius * 0.3, center.dy + radius * 0.1,
    );
    boomerangPath.quadraticBezierTo(
      center.dx + radius * 0.1, center.dy - radius * 0.3,
      center.dx + radius * 0.5, center.dy - radius * 0.6,
    );
    boomerangPath.quadraticBezierTo(
      center.dx + radius * 0.9, center.dy - radius * 0.9,
      center.dx + radius * 0.7, center.dy - radius * 0.3,
    );
    
    canvas.drawPath(boomerangPath, paint);
    
    final arrowPaint = Paint()
      ..color = arrowColor
      ..style = PaintingStyle.stroke
      ..strokeWidth = 2.0
      ..strokeCap = StrokeCap.round;
    
    final arrowStart = Offset(center.dx + radius * 0.1, center.dy - radius * 0.1);
    final arrowEnd = Offset(center.dx - radius * 0.4, center.dy - radius * 0.5);
    
    canvas.drawLine(arrowStart, arrowEnd, arrowPaint);
    
    final arrowHeadPath = Path();
    arrowHeadPath.moveTo(arrowEnd.dx, arrowEnd.dy);
    arrowHeadPath.lineTo(arrowEnd.dx + 5, arrowEnd.dy + 2);
    arrowHeadPath.moveTo(arrowEnd.dx, arrowEnd.dy);
    arrowHeadPath.lineTo(arrowEnd.dx + 2, arrowEnd.dy + 5);
    
    canvas.drawPath(arrowHeadPath, arrowPaint);
  }
  
  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

/// Floating action button style home button for bottom of pages
class FloatingHomeButton extends StatelessWidget {
  const FloatingHomeButton({super.key});
  
  @override
  Widget build(BuildContext context) {
    return Positioned(
      bottom: 20,
      left: 0,
      right: 0,
      child: Center(
        child: YarnHomeButton(),
      ),
    );
  }
}

/// Standard page scaffold with Indigenous-styled home button
class YarnPageScaffold extends StatelessWidget {
  final String title;
  final Widget body;
  final List<Widget>? actions;
  final bool showHomeButton;
  final Widget? floatingActionButton;
  final VoidCallback? onVoicePressed;
  final String? voiceDescription;
  
  const YarnPageScaffold({
    super.key,
    required this.title,
    required this.body,
    this.actions,
    this.showHomeButton = true,
    this.floatingActionButton,
    this.onVoicePressed,
    this.voiceDescription,
  });
  
  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final voiceService = ElevenLabsService();
    final prefs = PreferencesService();
    
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () {
            voiceService.stop();
            if (Navigator.canPop(context)) {
              Navigator.pop(context);
            } else {
              context.go('/');
            }
          },
        ),
        title: Text(title),
        actions: [
          if (onVoicePressed != null || voiceDescription != null)
            IconButton(
              icon: const Icon(Icons.volume_up),
              onPressed: onVoicePressed ?? () {
                if (voiceDescription != null && prefs.autoPlayVoice) {
                  voiceService.speak(voiceDescription!);
                }
              },
              tooltip: 'Read page aloud',
            ),
          ...?actions,
          const YarnHomeButton(showLabel: false),
          const SizedBox(width: 8),
        ],
      ),
      body: CulturalBackground(
        child: Stack(
          children: [
            body,
            if (showHomeButton && floatingActionButton == null)
              const FloatingHomeButton(),
          ],
        ),
      ),
      floatingActionButton: floatingActionButton,
    );
  }
}

/// Global speech recognition service - pre-initialized for smooth experience
class SpeechService {
  static final SpeechService _instance = SpeechService._internal();
  factory SpeechService() => _instance;
  SpeechService._internal();
  
  final stt.SpeechToText _speech = stt.SpeechToText();
  bool _isInitialized = false;
  bool _isAvailable = false;
  bool _isListening = false;
  
  bool get isAvailable => _isAvailable;
  bool get isListening => _isListening;
  
  /// Pre-initialize speech recognition - call this early in app lifecycle
  Future<bool> preInitialize() async {
    if (_isInitialized) return _isAvailable;
    
    try {
      _isAvailable = await _speech.initialize(
        onError: (error) {
          debugPrint('Speech error: ${error.errorMsg}');
          _isListening = false;
        },
        onStatus: (status) {
          debugPrint('Speech status: $status');
          if (status == 'done' || status == 'notListening') {
            _isListening = false;
          }
        },
      );
      _isInitialized = true;
      debugPrint('SpeechService initialized: available=$_isAvailable');
    } catch (e) {
      debugPrint('SpeechService init error: $e');
      _isAvailable = false;
      _isInitialized = true;
    }
    return _isAvailable;
  }
  
  Future<bool> startListening({
    required Function(String) onResult,
    Function()? onListeningStarted,
  }) async {
    if (_isListening) {
      await stopListening();
      return false;
    }
    
    if (!_isInitialized) {
      await preInitialize();
    }
    
    if (!_isAvailable) {
      return false;
    }
    
    _isListening = true;
    onListeningStarted?.call();
    
    try {
      await _speech.listen(
        onResult: (result) {
          if (result.finalResult && result.recognizedWords.isNotEmpty) {
            onResult(result.recognizedWords);
            _isListening = false;
          }
        },
        localeId: 'en_AU',
        listenMode: stt.ListenMode.dictation,
        cancelOnError: false,
        partialResults: true,
        listenFor: const Duration(seconds: 30),
        pauseFor: const Duration(seconds: 3),
      );
      return true;
    } catch (e) {
      debugPrint('Listen error: $e');
      _isListening = false;
      return false;
    }
  }
  
  Future<void> stopListening() async {
    if (_isListening) {
      await _speech.stop();
      _isListening = false;
    }
  }
}

/// Microphone button for speech-to-text input with friendly permission flow
class MicButton extends StatefulWidget {
  final Function(String) onResult;
  final Color? activeColor;
  
  const MicButton({
    super.key,
    required this.onResult,
    this.activeColor,
  });
  
  @override
  State<MicButton> createState() => _MicButtonState();
}

class _MicButtonState extends State<MicButton> {
  final SpeechService _speechService = SpeechService();
  bool _isListening = false;
  bool _hasShownPermissionGuide = false;
  
  @override
  void initState() {
    super.initState();
    _initialize();
  }
  
  Future<void> _initialize() async {
    final prefs = PreferencesService();
    _hasShownPermissionGuide = prefs.hasShownMicPermissionGuide;
    
    // Pre-initialize speech in background
    await _speechService.preInitialize();
    if (mounted) setState(() {});
  }
  
  Future<void> _showPermissionGuide() async {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final primaryColor = isDark ? AppColors.primaryLight : AppColors.primary;
    
    await showDialog(
      context: context,
      barrierDismissible: false,
      builder: (context) => AlertDialog(
        backgroundColor: isDark ? AppColors.cardDark : AppColors.cardLight,
        shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(16)),
        title: Row(
          children: [
            Icon(Icons.mic, color: primaryColor, size: 28),
            const SizedBox(width: 12),
            Expanded(
              child: Text(
                'Using Your Voice',
                style: TextStyle(
                  color: isDark ? AppColors.textPrimaryDark : AppColors.textPrimary,
                  fontWeight: FontWeight.bold,
                ),
              ),
            ),
          ],
        ),
        content: Column(
          mainAxisSize: MainAxisSize.min,
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              "Instead of typing, you can speak and I'll write it down for you.",
              style: TextStyle(
                fontSize: 16,
                color: isDark ? AppColors.textPrimaryDark : AppColors.textPrimary,
                height: 1.4,
              ),
            ),
            const SizedBox(height: 16),
            Container(
              padding: const EdgeInsets.all(12),
              decoration: BoxDecoration(
                color: primaryColor.withOpacity(0.1),
                borderRadius: BorderRadius.circular(12),
              ),
              child: Row(
                children: [
                  Icon(Icons.info_outline, color: primaryColor, size: 20),
                  const SizedBox(width: 10),
                  Expanded(
                    child: Text(
                      "Your phone will ask if it's okay to listen. Just tap Allow.",
                      style: TextStyle(
                        fontSize: 14,
                        color: isDark ? AppColors.textPrimaryDark : AppColors.textPrimary,
                      ),
                    ),
                  ),
                ],
              ),
            ),
            const SizedBox(height: 12),
            Text(
              "Your voice stays private - it never leaves your phone.",
              style: TextStyle(
                fontSize: 14,
                color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                fontStyle: FontStyle.italic,
              ),
            ),
          ],
        ),
        actions: [
          SizedBox(
            width: double.infinity,
            child: ElevatedButton(
              style: ElevatedButton.styleFrom(
                backgroundColor: primaryColor,
                foregroundColor: Colors.white,
                padding: const EdgeInsets.symmetric(vertical: 14),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
              onPressed: () {
                Navigator.of(context).pop();
              },
              child: const Text(
                "Got it, let's try!",
                style: TextStyle(fontSize: 16, fontWeight: FontWeight.w600),
              ),
            ),
          ),
        ],
      ),
    );
    
    // Mark as shown
    final prefs = PreferencesService();
    await prefs.setHasShownMicPermissionGuide(true);
    _hasShownPermissionGuide = true;
  }
  
  void _showNotAvailableMessage() {
    if (!mounted) return;
    
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final primaryColor = isDark ? AppColors.primaryLight : AppColors.primary;
    
    showDialog(
      context: context,
      builder: (context) => AlertDialog(
        backgroundColor: isDark ? AppColors.cardDark : AppColors.cardLight,
        shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(16)),
        title: Row(
          children: [
            Icon(Icons.mic_off, color: Colors.orange, size: 28),
            const SizedBox(width: 12),
            Expanded(
              child: Text(
                'Voice Not Ready',
                style: TextStyle(
                  color: isDark ? AppColors.textPrimaryDark : AppColors.textPrimary,
                  fontWeight: FontWeight.bold,
                ),
              ),
            ),
          ],
        ),
        content: Column(
          mainAxisSize: MainAxisSize.min,
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              "No worries! You can type instead, or try these steps to enable voice:",
              style: TextStyle(
                fontSize: 16,
                color: isDark ? AppColors.textPrimaryDark : AppColors.textPrimary,
                height: 1.4,
              ),
            ),
            const SizedBox(height: 16),
            Text(
              "1. Open Settings on your phone\n"
              "2. Go to Privacy & Security\n"
              "3. Tap Speech Recognition\n"
              "4. Turn on YarnAI",
              style: TextStyle(
                fontSize: 14,
                color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
                height: 1.6,
              ),
            ),
          ],
        ),
        actions: [
          SizedBox(
            width: double.infinity,
            child: ElevatedButton(
              style: ElevatedButton.styleFrom(
                backgroundColor: primaryColor,
                foregroundColor: Colors.white,
                padding: const EdgeInsets.symmetric(vertical: 14),
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
              onPressed: () => Navigator.of(context).pop(),
              child: const Text(
                "Got it",
                style: TextStyle(fontSize: 16, fontWeight: FontWeight.w600),
              ),
            ),
          ),
        ],
      ),
    );
  }
  
  Future<void> _toggleListening() async {
    // Show friendly guide first time
    if (!_hasShownPermissionGuide) {
      await _showPermissionGuide();
    }
    
    // Check if speech is available
    if (!_speechService.isAvailable) {
      _showNotAvailableMessage();
      return;
    }
    
    // Toggle listening
    if (_isListening) {
      await _speechService.stopListening();
      if (mounted) setState(() => _isListening = false);
    } else {
      final started = await _speechService.startListening(
        onResult: (text) {
          widget.onResult(text);
          if (mounted) setState(() => _isListening = false);
        },
        onListeningStarted: () {
          if (mounted) setState(() => _isListening = true);
        },
      );
      
      if (!started && mounted) {
        _showNotAvailableMessage();
      }
    }
  }
  
  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    final activeColor = widget.activeColor ?? AppColors.primary;
    
    return IconButton(
      icon: Icon(
        _isListening ? Icons.mic : Icons.mic_none,
        color: _isListening 
            ? activeColor 
            : (isDark ? AppColors.textSecondaryDark : AppColors.textSecondary),
      ),
      onPressed: _toggleListening,
      tooltip: _isListening ? 'Stop listening' : 'Speak',
    );
  }
}

/// Funky culturally-styled microphone button with pulse animation
/// Used for prominent voice input across the app
class FunkyMicButton extends StatefulWidget {
  final VoidCallback onTap;
  final bool isListening;
  final double size;
  final String? label;
  
  const FunkyMicButton({
    super.key,
    required this.onTap,
    this.isListening = false,
    this.size = 80,
    this.label,
  });
  
  @override
  State<FunkyMicButton> createState() => _FunkyMicButtonState();
}

class _FunkyMicButtonState extends State<FunkyMicButton> 
    with SingleTickerProviderStateMixin {
  late AnimationController _pulseController;
  late Animation<double> _pulseAnimation;
  late Animation<double> _glowAnimation;
  
  @override
  void initState() {
    super.initState();
    _pulseController = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 1500),
    )..repeat(reverse: true);
    
    _pulseAnimation = Tween<double>(begin: 1.0, end: 1.08).animate(
      CurvedAnimation(parent: _pulseController, curve: Curves.easeInOut),
    );
    
    _glowAnimation = Tween<double>(begin: 0.3, end: 0.7).animate(
      CurvedAnimation(parent: _pulseController, curve: Curves.easeInOut),
    );
  }
  
  @override
  void dispose() {
    _pulseController.dispose();
    super.dispose();
  }
  
  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Column(
      mainAxisSize: MainAxisSize.min,
      children: [
        AnimatedBuilder(
          animation: _pulseController,
          builder: (context, child) {
            return Transform.scale(
              scale: widget.isListening ? 1.1 : _pulseAnimation.value,
              child: Container(
                width: widget.size,
                height: widget.size,
                decoration: BoxDecoration(
                  shape: BoxShape.circle,
                  gradient: LinearGradient(
                    begin: Alignment.topLeft,
                    end: Alignment.bottomRight,
                    colors: widget.isListening
                        ? [
                            AppColors.error,
                            AppColors.error.withOpacity(0.8),
                          ]
                        : [
                            AppColors.ochre,
                            AppColors.terracotta,
                            AppColors.primary,
                          ],
                  ),
                  boxShadow: [
                    BoxShadow(
                      color: (widget.isListening 
                          ? AppColors.error 
                          : AppColors.ochre).withOpacity(_glowAnimation.value),
                      blurRadius: widget.isListening ? 24 : 16,
                      spreadRadius: widget.isListening ? 4 : 2,
                    ),
                    BoxShadow(
                      color: Colors.black.withOpacity(0.2),
                      blurRadius: 8,
                      offset: const Offset(0, 4),
                    ),
                  ],
                ),
                child: Material(
                  color: Colors.transparent,
                  child: InkWell(
                    onTap: widget.onTap,
                    customBorder: const CircleBorder(),
                    splashColor: Colors.white.withOpacity(0.3),
                    child: Center(
                      child: Icon(
                        widget.isListening ? Icons.stop_rounded : Icons.mic_rounded,
                        size: widget.size * 0.45,
                        color: Colors.white,
                      ),
                    ),
                  ),
                ),
              ),
            );
          },
        ),
        if (widget.label != null) ...[
          const SizedBox(height: 12),
          Text(
            widget.label!,
            style: TextStyle(
              fontSize: 14,
              fontWeight: FontWeight.w600,
              color: isDark ? AppColors.textSecondaryDark : AppColors.textSecondary,
            ),
          ),
        ],
      ],
    );
  }
}
