import 'dart:math' as math;
import 'package:flutter/material.dart';
import '../theme/app_theme.dart';

class CeremonialFire extends StatefulWidget {
  final double size;
  final VoidCallback? onTap;
  final bool isListening;
  
  const CeremonialFire({
    super.key,
    this.size = 200,
    this.onTap,
    this.isListening = false,
  });

  @override
  State<CeremonialFire> createState() => _CeremonialFireState();
}

class _CeremonialFireState extends State<CeremonialFire>
    with TickerProviderStateMixin {
  late AnimationController _flameController;
  late AnimationController _glowController;
  late AnimationController _sparkController;
  late AnimationController _pulseController;
  
  final List<_Spark> _sparks = [];
  final math.Random _random = math.Random();

  @override
  void initState() {
    super.initState();
    
    _flameController = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 800),
    )..repeat();
    
    _glowController = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 2000),
    )..repeat(reverse: true);
    
    _sparkController = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 100),
    )..addListener(_updateSparks)..repeat();
    
    _pulseController = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 1500),
    )..repeat(reverse: true);
    
    _generateInitialSparks();
  }
  
  void _generateInitialSparks() {
    for (int i = 0; i < 8; i++) {
      _sparks.add(_Spark(
        x: _random.nextDouble() * 0.6 - 0.3,
        y: _random.nextDouble() * 0.5,
        size: 2 + _random.nextDouble() * 3,
        speed: 0.01 + _random.nextDouble() * 0.02,
        opacity: 0.3 + _random.nextDouble() * 0.7,
      ));
    }
  }
  
  void _updateSparks() {
    for (var spark in _sparks) {
      spark.y -= spark.speed;
      spark.x += (_random.nextDouble() - 0.5) * 0.02;
      spark.opacity -= 0.015;
      
      if (spark.opacity <= 0 || spark.y < -0.3) {
        spark.x = _random.nextDouble() * 0.4 - 0.2;
        spark.y = 0.3 + _random.nextDouble() * 0.2;
        spark.opacity = 0.5 + _random.nextDouble() * 0.5;
        spark.size = 2 + _random.nextDouble() * 3;
      }
    }
  }

  @override
  void dispose() {
    _flameController.dispose();
    _glowController.dispose();
    _sparkController.dispose();
    _pulseController.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return GestureDetector(
      onTap: widget.onTap,
      child: SizedBox(
        width: widget.size,
        height: widget.size,
        child: AnimatedBuilder(
          animation: Listenable.merge([
            _flameController,
            _glowController,
            _sparkController,
            _pulseController,
          ]),
          builder: (context, child) {
            return CustomPaint(
              painter: _CeremonialFirePainter(
                flamePhase: _flameController.value,
                glowPhase: _glowController.value,
                pulsePhase: _pulseController.value,
                sparks: _sparks,
                isDark: isDark,
                isListening: widget.isListening,
              ),
              child: Center(
                child: _buildCenterContent(isDark),
              ),
            );
          },
        ),
      ),
    );
  }
  
  Widget _buildCenterContent(bool isDark) {
    final pulseScale = 1.0 + (_pulseController.value * 0.08);
    
    return Transform.scale(
      scale: widget.isListening ? pulseScale : 1.0,
      child: Container(
        width: widget.size * 0.35,
        height: widget.size * 0.35,
        decoration: BoxDecoration(
          shape: BoxShape.circle,
          gradient: RadialGradient(
            colors: widget.isListening
                ? [
                    AppColors.ochre.withOpacity(0.9),
                    AppColors.primary.withOpacity(0.7),
                    AppColors.primaryDark.withOpacity(0.5),
                  ]
                : [
                    AppColors.ochre.withOpacity(0.6),
                    AppColors.primary.withOpacity(0.4),
                    Colors.transparent,
                  ],
          ),
          boxShadow: [
            BoxShadow(
              color: AppColors.ochre.withOpacity(widget.isListening ? 0.6 : 0.3),
              blurRadius: widget.isListening ? 30 : 20,
              spreadRadius: widget.isListening ? 10 : 5,
            ),
          ],
        ),
        child: Icon(
          widget.isListening ? Icons.mic : Icons.local_fire_department,
          size: widget.size * 0.15,
          color: isDark ? AppColors.accentLight : Colors.white,
        ),
      ),
    );
  }
}

class _Spark {
  double x;
  double y;
  double size;
  double speed;
  double opacity;
  
  _Spark({
    required this.x,
    required this.y,
    required this.size,
    required this.speed,
    required this.opacity,
  });
}

class _CeremonialFirePainter extends CustomPainter {
  final double flamePhase;
  final double glowPhase;
  final double pulsePhase;
  final List<_Spark> sparks;
  final bool isDark;
  final bool isListening;
  
  _CeremonialFirePainter({
    required this.flamePhase,
    required this.glowPhase,
    required this.pulsePhase,
    required this.sparks,
    required this.isDark,
    required this.isListening,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final center = Offset(size.width / 2, size.height / 2);
    final baseRadius = size.width * 0.35;
    
    _drawOuterGlow(canvas, center, baseRadius, size);
    _drawConcentricRings(canvas, center, baseRadius);
    _drawFlames(canvas, center, baseRadius, size);
    _drawSparks(canvas, center, size);
    _drawInnerGlow(canvas, center, baseRadius);
  }
  
  void _drawOuterGlow(Canvas canvas, Offset center, double baseRadius, Size size) {
    final glowRadius = baseRadius * (1.8 + glowPhase * 0.3);
    final glowOpacity = isListening ? 0.25 : 0.15;
    
    final glowPaint = Paint()
      ..shader = RadialGradient(
        colors: [
          AppColors.ochre.withOpacity(glowOpacity),
          AppColors.primary.withOpacity(glowOpacity * 0.5),
          Colors.transparent,
        ],
        stops: const [0.0, 0.5, 1.0],
      ).createShader(Rect.fromCircle(center: center, radius: glowRadius));
    
    canvas.drawCircle(center, glowRadius, glowPaint);
  }
  
  void _drawConcentricRings(Canvas canvas, Offset center, double baseRadius) {
    final ringPaint = Paint()
      ..style = PaintingStyle.stroke
      ..strokeWidth = 2.0;
    
    for (int i = 0; i < 4; i++) {
      final radius = baseRadius * (0.6 + i * 0.15);
      final opacity = 0.3 - i * 0.06;
      
      ringPaint.color = AppColors.ochre.withOpacity(opacity);
      canvas.drawCircle(center, radius, ringPaint);
    }
  }
  
  void _drawFlames(Canvas canvas, Offset center, double baseRadius, Size size) {
    final flameCount = 12;
    final flamePaint = Paint()..style = PaintingStyle.fill;
    
    for (int i = 0; i < flameCount; i++) {
      final angle = (i / flameCount) * 2 * math.pi;
      final phaseOffset = (flamePhase + i * 0.08) % 1.0;
      
      final flameHeight = baseRadius * (0.4 + math.sin(phaseOffset * 2 * math.pi) * 0.15);
      final flameWidth = baseRadius * 0.12;
      
      final flameBase = Offset(
        center.dx + math.cos(angle) * baseRadius * 0.5,
        center.dy + math.sin(angle) * baseRadius * 0.5,
      );
      
      final flameTip = Offset(
        center.dx + math.cos(angle) * (baseRadius * 0.5 + flameHeight),
        center.dy + math.sin(angle) * (baseRadius * 0.5 + flameHeight),
      );
      
      final path = Path();
      
      final perpAngle = angle + math.pi / 2;
      final leftBase = Offset(
        flameBase.dx + math.cos(perpAngle) * flameWidth,
        flameBase.dy + math.sin(perpAngle) * flameWidth,
      );
      final rightBase = Offset(
        flameBase.dx - math.cos(perpAngle) * flameWidth,
        flameBase.dy - math.sin(perpAngle) * flameWidth,
      );
      
      path.moveTo(leftBase.dx, leftBase.dy);
      path.quadraticBezierTo(flameTip.dx, flameTip.dy, rightBase.dx, rightBase.dy);
      path.close();
      
      final flameGradient = RadialGradient(
        center: Alignment.center,
        colors: [
          AppColors.accentLight.withOpacity(0.8),
          AppColors.ochre.withOpacity(0.6),
          AppColors.primary.withOpacity(0.4),
        ],
      ).createShader(Rect.fromCircle(center: flameBase, radius: flameHeight));
      
      flamePaint.shader = flameGradient;
      canvas.drawPath(path, flamePaint);
    }
    
    _drawInnerFlames(canvas, center, baseRadius);
  }
  
  void _drawInnerFlames(Canvas canvas, Offset center, double baseRadius) {
    final innerFlameCount = 8;
    final flamePaint = Paint()..style = PaintingStyle.fill;
    
    for (int i = 0; i < innerFlameCount; i++) {
      final angle = (i / innerFlameCount) * 2 * math.pi + math.pi / 8;
      final phaseOffset = (flamePhase * 1.5 + i * 0.1) % 1.0;
      
      final flameHeight = baseRadius * (0.25 + math.sin(phaseOffset * 2 * math.pi) * 0.1);
      final flameWidth = baseRadius * 0.08;
      
      final flameBase = Offset(
        center.dx + math.cos(angle) * baseRadius * 0.35,
        center.dy + math.sin(angle) * baseRadius * 0.35,
      );
      
      final flameTip = Offset(
        center.dx + math.cos(angle) * (baseRadius * 0.35 + flameHeight),
        center.dy + math.sin(angle) * (baseRadius * 0.35 + flameHeight),
      );
      
      final path = Path();
      
      final perpAngle = angle + math.pi / 2;
      final leftBase = Offset(
        flameBase.dx + math.cos(perpAngle) * flameWidth,
        flameBase.dy + math.sin(perpAngle) * flameWidth,
      );
      final rightBase = Offset(
        flameBase.dx - math.cos(perpAngle) * flameWidth,
        flameBase.dy - math.sin(perpAngle) * flameWidth,
      );
      
      path.moveTo(leftBase.dx, leftBase.dy);
      path.quadraticBezierTo(flameTip.dx, flameTip.dy, rightBase.dx, rightBase.dy);
      path.close();
      
      flamePaint.shader = null;
      flamePaint.color = AppColors.accentLight.withOpacity(0.5 + phaseOffset * 0.3);
      canvas.drawPath(path, flamePaint);
    }
  }
  
  void _drawSparks(Canvas canvas, Offset center, Size size) {
    final sparkPaint = Paint()..style = PaintingStyle.fill;
    
    for (var spark in sparks) {
      if (spark.opacity <= 0) continue;
      
      final sparkPos = Offset(
        center.dx + spark.x * size.width,
        center.dy - spark.y * size.height,
      );
      
      sparkPaint.color = AppColors.accentLight.withOpacity(spark.opacity);
      canvas.drawCircle(sparkPos, spark.size, sparkPaint);
    }
  }
  
  void _drawInnerGlow(Canvas canvas, Offset center, double baseRadius) {
    final innerGlowRadius = baseRadius * 0.4;
    final glowIntensity = isListening ? 0.6 : 0.4;
    
    final innerGlowPaint = Paint()
      ..shader = RadialGradient(
        colors: [
          AppColors.accentLight.withOpacity(glowIntensity),
          AppColors.ochre.withOpacity(glowIntensity * 0.5),
          Colors.transparent,
        ],
      ).createShader(Rect.fromCircle(center: center, radius: innerGlowRadius));
    
    canvas.drawCircle(center, innerGlowRadius, innerGlowPaint);
  }

  @override
  bool shouldRepaint(covariant _CeremonialFirePainter oldDelegate) {
    return oldDelegate.flamePhase != flamePhase ||
           oldDelegate.glowPhase != glowPhase ||
           oldDelegate.isListening != isListening;
  }
}

class YarnHomeButton extends StatelessWidget {
  final VoidCallback onTap;
  final double size;
  
  const YarnHomeButton({
    super.key,
    required this.onTap,
    this.size = 60,
  });

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return GestureDetector(
      onTap: onTap,
      child: Container(
        width: size,
        height: size,
        decoration: BoxDecoration(
          shape: BoxShape.circle,
          gradient: LinearGradient(
            begin: Alignment.topLeft,
            end: Alignment.bottomRight,
            colors: [
              AppColors.primary,
              AppColors.ochre,
            ],
          ),
          boxShadow: [
            BoxShadow(
              color: AppColors.primary.withOpacity(0.4),
              blurRadius: 12,
              offset: const Offset(0, 4),
            ),
          ],
        ),
        child: Icon(
          Icons.local_fire_department,
          size: size * 0.5,
          color: isDark ? AppColors.accentLight : Colors.white,
        ),
      ),
    );
  }
}
