import 'dart:math' as math;
import 'package:flutter/material.dart';

class AppColors {
  static const Color primary = Color(0xFFB85C38);
  static const Color primaryLight = Color(0xFFD4896C);
  static const Color primaryDark = Color(0xFF8B4429);
  
  static const Color accent = Color(0xFFE8C07D);
  static const Color accentLight = Color(0xFFF5DEB3);
  
  static const Color backgroundLight = Color(0xFFFAF6F0);
  static const Color backgroundDark = Color(0xFF000000);
  
  static const Color cardLight = Color(0xFFF5EEE6);
  static const Color cardDark = Color(0xFF1A1410);
  
  static const Color textPrimary = Color(0xFF2D2016);
  static const Color textSecondary = Color(0xFF6B5344);
  static const Color textTertiary = Color(0xFF9D8B7A);
  
  static const Color textPrimaryDark = Color(0xFFFAF6F0);
  static const Color textSecondaryDark = Color(0xFFD4C4B5);
  static const Color textTertiaryDark = Color(0xFFA69485);
  
  static const Color success = Color(0xFF4CAF50);
  static const Color error = Color(0xFFE53935);
  static const Color warning = Color(0xFFFF9800);
  
  static const Color earth = Color(0xFF8B6914);
  static const Color ochre = Color(0xFFCC7722);
  static const Color terracotta = Color(0xFFCB6843);
  static const Color eucalyptus = Color(0xFF5D8A66);
}

class AppTheme {
  static ThemeData get lightTheme {
    return ThemeData(
      useMaterial3: true,
      brightness: Brightness.light,
      primaryColor: AppColors.primary,
      scaffoldBackgroundColor: AppColors.backgroundLight,
      colorScheme: const ColorScheme.light(
        primary: AppColors.primary,
        secondary: AppColors.accent,
        surface: AppColors.cardLight,
        error: AppColors.error,
        onPrimary: Colors.white,
        onSecondary: AppColors.textPrimary,
        onSurface: AppColors.textPrimary,
        onError: Colors.white,
      ),
      appBarTheme: const AppBarTheme(
        backgroundColor: AppColors.backgroundLight,
        foregroundColor: AppColors.textPrimary,
        elevation: 0,
        centerTitle: false,
        titleTextStyle: TextStyle(
          fontSize: 20,
          fontWeight: FontWeight.w600,
          color: AppColors.textPrimary,
        ),
      ),
      cardTheme: CardThemeData(
        color: AppColors.cardLight,
        elevation: 0,
        shape: RoundedRectangleBorder(
          borderRadius: BorderRadius.circular(12),
          side: BorderSide(color: AppColors.textTertiary.withOpacity(0.2)),
        ),
      ),
      elevatedButtonTheme: ElevatedButtonThemeData(
        style: ElevatedButton.styleFrom(
          backgroundColor: AppColors.primary,
          foregroundColor: Colors.white,
          elevation: 0,
          padding: const EdgeInsets.symmetric(horizontal: 24, vertical: 16),
          shape: RoundedRectangleBorder(
            borderRadius: BorderRadius.circular(12),
          ),
          textStyle: const TextStyle(
            fontSize: 16,
            fontWeight: FontWeight.w600,
          ),
        ),
      ),
      outlinedButtonTheme: OutlinedButtonThemeData(
        style: OutlinedButton.styleFrom(
          foregroundColor: AppColors.primary,
          side: const BorderSide(color: AppColors.primary),
          padding: const EdgeInsets.symmetric(horizontal: 24, vertical: 16),
          shape: RoundedRectangleBorder(
            borderRadius: BorderRadius.circular(12),
          ),
        ),
      ),
      inputDecorationTheme: InputDecorationTheme(
        filled: true,
        fillColor: Colors.white,
        border: OutlineInputBorder(
          borderRadius: BorderRadius.circular(12),
          borderSide: BorderSide(color: AppColors.textTertiary.withOpacity(0.3)),
        ),
        enabledBorder: OutlineInputBorder(
          borderRadius: BorderRadius.circular(12),
          borderSide: BorderSide(color: AppColors.textTertiary.withOpacity(0.3)),
        ),
        focusedBorder: OutlineInputBorder(
          borderRadius: BorderRadius.circular(12),
          borderSide: const BorderSide(color: AppColors.primary, width: 2),
        ),
        contentPadding: const EdgeInsets.symmetric(horizontal: 16, vertical: 16),
      ),
      textTheme: const TextTheme(
        headlineLarge: TextStyle(
          fontSize: 28,
          fontWeight: FontWeight.bold,
          color: AppColors.textPrimary,
        ),
        headlineMedium: TextStyle(
          fontSize: 24,
          fontWeight: FontWeight.w600,
          color: AppColors.textPrimary,
        ),
        headlineSmall: TextStyle(
          fontSize: 20,
          fontWeight: FontWeight.w600,
          color: AppColors.textPrimary,
        ),
        bodyLarge: TextStyle(
          fontSize: 16,
          color: AppColors.textPrimary,
        ),
        bodyMedium: TextStyle(
          fontSize: 14,
          color: AppColors.textSecondary,
        ),
        bodySmall: TextStyle(
          fontSize: 12,
          color: AppColors.textTertiary,
        ),
      ),
    );
  }

  static ThemeData get darkTheme {
    return ThemeData(
      useMaterial3: true,
      brightness: Brightness.dark,
      primaryColor: AppColors.primary,
      scaffoldBackgroundColor: AppColors.backgroundDark,
      colorScheme: const ColorScheme.dark(
        primary: AppColors.primaryLight,
        secondary: AppColors.accent,
        surface: AppColors.cardDark,
        error: AppColors.error,
        onPrimary: AppColors.textPrimary,
        onSecondary: AppColors.textPrimary,
        onSurface: AppColors.textPrimaryDark,
        onError: Colors.white,
      ),
      appBarTheme: const AppBarTheme(
        backgroundColor: Colors.black,
        foregroundColor: AppColors.textPrimaryDark,
        elevation: 0,
        centerTitle: false,
        titleTextStyle: TextStyle(
          fontSize: 20,
          fontWeight: FontWeight.w600,
          color: AppColors.textPrimaryDark,
        ),
      ),
      cardTheme: CardThemeData(
        color: AppColors.cardDark,
        elevation: 0,
        shape: RoundedRectangleBorder(
          borderRadius: BorderRadius.circular(12),
          side: BorderSide(color: AppColors.textTertiaryDark.withOpacity(0.2)),
        ),
      ),
      elevatedButtonTheme: ElevatedButtonThemeData(
        style: ElevatedButton.styleFrom(
          backgroundColor: AppColors.primaryLight,
          foregroundColor: AppColors.textPrimary,
          elevation: 0,
          padding: const EdgeInsets.symmetric(horizontal: 24, vertical: 16),
          shape: RoundedRectangleBorder(
            borderRadius: BorderRadius.circular(12),
          ),
        ),
      ),
      inputDecorationTheme: InputDecorationTheme(
        filled: true,
        fillColor: AppColors.cardDark,
        border: OutlineInputBorder(
          borderRadius: BorderRadius.circular(12),
          borderSide: BorderSide(color: AppColors.textTertiaryDark.withOpacity(0.3)),
        ),
        enabledBorder: OutlineInputBorder(
          borderRadius: BorderRadius.circular(12),
          borderSide: BorderSide(color: AppColors.textTertiaryDark.withOpacity(0.3)),
        ),
        focusedBorder: OutlineInputBorder(
          borderRadius: BorderRadius.circular(12),
          borderSide: const BorderSide(color: AppColors.primaryLight, width: 2),
        ),
      ),
      textTheme: const TextTheme(
        headlineLarge: TextStyle(
          fontSize: 28,
          fontWeight: FontWeight.bold,
          color: AppColors.textPrimaryDark,
        ),
        headlineMedium: TextStyle(
          fontSize: 24,
          fontWeight: FontWeight.w600,
          color: AppColors.textPrimaryDark,
        ),
        headlineSmall: TextStyle(
          fontSize: 20,
          fontWeight: FontWeight.w600,
          color: AppColors.textPrimaryDark,
        ),
        bodyLarge: TextStyle(
          fontSize: 16,
          color: AppColors.textPrimaryDark,
        ),
        bodyMedium: TextStyle(
          fontSize: 14,
          color: AppColors.textSecondaryDark,
        ),
        bodySmall: TextStyle(
          fontSize: 12,
          color: AppColors.textTertiaryDark,
        ),
      ),
    );
  }
}

class CulturalBackground extends StatelessWidget {
  final Widget child;
  final bool showPattern;
  
  const CulturalBackground({
    super.key,
    required this.child,
    this.showPattern = true,
  });

  @override
  Widget build(BuildContext context) {
    final isDark = Theme.of(context).brightness == Brightness.dark;
    
    return Stack(
      children: [
        Positioned.fill(
          child: Container(
            decoration: BoxDecoration(
              gradient: LinearGradient(
                begin: Alignment.topCenter,
                end: Alignment.bottomCenter,
                colors: isDark 
                  ? [
                      const Color(0xFF1A1410),
                      const Color(0xFF0D0906),
                      Colors.black,
                    ]
                  : [
                      AppColors.backgroundLight,
                      const Color(0xFFF5EBE0),
                      const Color(0xFFEDE4D9),
                    ],
              ),
            ),
            child: showPattern ? CustomPaint(
              painter: _CulturalPatternPainter(isDark: isDark),
              size: Size.infinite,
            ) : null,
          ),
        ),
        child,
      ],
    );
  }
}

class _CulturalPatternPainter extends CustomPainter {
  final bool isDark;
  
  _CulturalPatternPainter({required this.isDark});

  @override
  void paint(Canvas canvas, Size size) {
    final paint = Paint()..style = PaintingStyle.fill;
    
    // Draw concentric circle motifs (Aboriginal art inspired)
    _drawConcentricCircles(canvas, size, paint);
    
    // Draw flowing arc pattern at bottom
    _drawFlowingArcs(canvas, size, paint);
    
    // Draw dot pattern overlay
    _drawDotPattern(canvas, size, paint);
  }
  
  void _drawConcentricCircles(Canvas canvas, Size size, Paint paint) {
    // Top-right concentric circles
    final center1 = Offset(size.width * 0.9, size.height * 0.08);
    _drawConcentricSet(canvas, center1, size.width * 0.18, paint);
    
    // Bottom-left concentric circles
    final center2 = Offset(size.width * 0.08, size.height * 0.92);
    _drawConcentricSet(canvas, center2, size.width * 0.15, paint);
    
    // Middle-right subtle circles
    final center3 = Offset(size.width * 0.95, size.height * 0.55);
    _drawConcentricSet(canvas, center3, size.width * 0.12, paint, subtle: true);
  }
  
  void _drawConcentricSet(Canvas canvas, Offset center, double maxRadius, Paint paint, {bool subtle = false}) {
    final baseOpacity = subtle ? 0.25 : 0.40;
    final colors = isDark 
      ? [AppColors.ochre, AppColors.primary, AppColors.earth]
      : [AppColors.primary, AppColors.ochre, AppColors.accent];
    
    for (int i = 0; i < 4; i++) {
      final radius = maxRadius * (1 - i * 0.22);
      if (radius <= 0) continue;
      
      paint.color = colors[i % colors.length].withOpacity(baseOpacity * (1 - i * 0.15));
      paint.style = PaintingStyle.stroke;
      paint.strokeWidth = isDark ? 3.5 : 3.0;
      canvas.drawCircle(center, radius, paint);
    }
    
    // Center dot
    paint.style = PaintingStyle.fill;
    paint.color = (isDark ? AppColors.ochre : AppColors.primary).withOpacity(baseOpacity * 1.5);
    canvas.drawCircle(center, maxRadius * 0.12, paint);
  }
  
  void _drawFlowingArcs(Canvas canvas, Size size, Paint paint) {
    paint.style = PaintingStyle.stroke;
    paint.strokeWidth = 2.5;
    
    final arcColor = isDark 
        ? AppColors.primary.withOpacity(0.35)
        : AppColors.earth.withOpacity(0.25);
    paint.color = arcColor;
    
    // Flowing curves at the bottom inspired by river/journey patterns
    final path = Path();
    path.moveTo(0, size.height * 0.75);
    path.quadraticBezierTo(
      size.width * 0.25, size.height * 0.72,
      size.width * 0.5, size.height * 0.78,
    );
    path.quadraticBezierTo(
      size.width * 0.75, size.height * 0.84,
      size.width, size.height * 0.80,
    );
    canvas.drawPath(path, paint);
    
    // Second arc
    final path2 = Path();
    path2.moveTo(0, size.height * 0.82);
    path2.quadraticBezierTo(
      size.width * 0.3, size.height * 0.78,
      size.width * 0.6, size.height * 0.85,
    );
    path2.quadraticBezierTo(
      size.width * 0.85, size.height * 0.90,
      size.width, size.height * 0.88,
    );
    canvas.drawPath(path2, paint);
  }
  
  void _drawDotPattern(Canvas canvas, Size size, Paint paint) {
    paint.style = PaintingStyle.fill;
    
    final random = math.Random(42);
    final dotSpacing = 30.0;
    
    // Primary dot color (more visible now)
    final dotColor1 = isDark 
        ? AppColors.primary.withOpacity(0.40)
        : AppColors.primary.withOpacity(0.25);
    
    // Secondary dot color for variation
    final dotColor2 = isDark 
        ? AppColors.ochre.withOpacity(0.35)
        : AppColors.earth.withOpacity(0.20);
    
    for (double x = 0; x < size.width; x += dotSpacing) {
      for (double y = 0; y < size.height; y += dotSpacing) {
        final offsetX = (random.nextDouble() - 0.5) * 12;
        final offsetY = (random.nextDouble() - 0.5) * 12;
        final dotSize = 2.5 + random.nextDouble() * 2.0;
        
        // Alternate between dot colors for visual interest
        final useSecondary = random.nextDouble() > 0.7;
        paint.color = useSecondary ? dotColor2 : dotColor1;
        
        canvas.drawCircle(
          Offset(x + offsetX, y + offsetY),
          dotSize,
          paint,
        );
      }
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}
