import 'package:flutter/foundation.dart';
import 'package:flutter_tts/flutter_tts.dart';
import 'package:speech_to_text/speech_to_text.dart';
import 'package:permission_handler/permission_handler.dart';
import 'elevenlabs_service.dart' show VoicePersona;

class VoiceService {
  static final VoiceService _instance = VoiceService._internal();
  factory VoiceService() => _instance;
  
  final FlutterTts _tts = FlutterTts();
  final SpeechToText _stt = SpeechToText();
  
  bool _isListening = false;
  bool _isSpeaking = false;
  VoicePersona _persona = VoicePersona.aunty;
  
  bool get isListening => _isListening;
  bool get isSpeaking => _isSpeaking;
  VoicePersona get persona => _persona;
  
  VoiceService._internal();
  
  Future<void> initialize() async {
    await _tts.setLanguage('en-AU');
    await _tts.setSpeechRate(0.45);
    await _tts.setVolume(1.0);
    await _tts.setPitch(1.0);
    
    _tts.setCompletionHandler(() {
      _isSpeaking = false;
    });
    
    _tts.setErrorHandler((message) {
      debugPrint('TTS Error: $message');
      _isSpeaking = false;
    });
  }
  
  void setPersona(VoicePersona persona) {
    _persona = persona;
    if (persona == VoicePersona.aunty) {
      _tts.setPitch(1.1);
      _tts.setSpeechRate(0.42);
    } else {
      _tts.setPitch(0.9);
      _tts.setSpeechRate(0.48);
    }
  }
  
  Future<bool> requestMicrophonePermission() async {
    final status = await Permission.microphone.request();
    return status.isGranted;
  }
  
  Future<void> speak(String text) async {
    if (_isSpeaking) {
      await stop();
    }
    
    _isSpeaking = true;
    await _tts.speak(text);
  }
  
  Future<void> stop() async {
    await _tts.stop();
    _isSpeaking = false;
  }
  
  Future<bool> startListening({
    required Function(String) onResult,
    Function(String)? onError,
  }) async {
    if (_isListening) return false;
    
    final hasPermission = await requestMicrophonePermission();
    if (!hasPermission) {
      onError?.call('Microphone permission denied');
      return false;
    }
    
    final available = await _stt.initialize(
      onError: (error) {
        debugPrint('STT Error: ${error.errorMsg}');
        onError?.call(error.errorMsg);
        _isListening = false;
      },
      onStatus: (status) {
        debugPrint('STT Status: $status');
        if (status == 'done' || status == 'notListening') {
          _isListening = false;
        }
      },
    );
    
    if (!available) {
      onError?.call('Speech recognition not available');
      return false;
    }
    
    _isListening = true;
    
    await _stt.listen(
      onResult: (result) {
        if (result.finalResult) {
          onResult(result.recognizedWords);
          _isListening = false;
        }
      },
      localeId: 'en_AU',
      listenMode: ListenMode.dictation,
      cancelOnError: true,
      partialResults: true,
    );
    
    return true;
  }
  
  Future<void> stopListening() async {
    if (_isListening) {
      await _stt.stop();
      _isListening = false;
    }
  }
  
  Future<void> speakThenListen({
    required String prompt,
    required Function(String) onResult,
    Function(String)? onError,
  }) async {
    await speak(prompt);
    
    await Future.delayed(const Duration(milliseconds: 500));
    
    while (_isSpeaking) {
      await Future.delayed(const Duration(milliseconds: 100));
    }
    
    await Future.delayed(const Duration(milliseconds: 300));
    
    await startListening(onResult: onResult, onError: onError);
  }
}
