import 'package:hive_flutter/hive_flutter.dart';
import 'elevenlabs_service.dart';
import '../models/region.dart';

class PreferencesService {
  static final PreferencesService _instance = PreferencesService._internal();
  factory PreferencesService() => _instance;
  PreferencesService._internal();

  static const String _boxName = 'preferences';
  static const String _hasSeenWelcome = 'hasSeenWelcome';
  static const String _voicePersona = 'voicePersona';
  static const String _autoPlayVoice = 'autoPlayVoice';
  static const String _hasShownMicPermissionGuide = 'hasShownMicPermissionGuide';
  static const String _userName = 'userName';
  static const String _lastVisit = 'lastVisit';
  static const String _isDarkMode = 'isDarkMode';
  static const String _region = 'region';
  static const String _preferredGreeting = 'preferredGreeting';
  static const String _customGreeting = 'customGreeting';
  
  // Callback for theme changes
  static Function(bool)? onThemeChanged;
  
  bool _isInitialized = false;
  bool get isInitialized => _isInitialized;

  Box? _box;

  Future<void> initialize() async {
    _box = await Hive.openBox(_boxName);
    _isInitialized = true;
  }

  bool get hasSeenWelcome => _box?.get(_hasSeenWelcome, defaultValue: false) ?? false;
  
  Future<void> setHasSeenWelcome(bool value) async {
    await _box?.put(_hasSeenWelcome, value);
  }

  VoicePersona get voicePersona {
    final value = _box?.get(_voicePersona, defaultValue: 'aunty') ?? 'aunty';
    return value == 'uncle' ? VoicePersona.uncle : VoicePersona.aunty;
  }

  Future<void> setVoicePersona(VoicePersona persona) async {
    await _box?.put(_voicePersona, persona == VoicePersona.uncle ? 'uncle' : 'aunty');
  }

  bool get autoPlayVoice => _box?.get(_autoPlayVoice, defaultValue: true) ?? true;
  
  Future<void> setAutoPlayVoice(bool value) async {
    await _box?.put(_autoPlayVoice, value);
  }

  bool get hasShownMicPermissionGuide => _box?.get(_hasShownMicPermissionGuide, defaultValue: false) ?? false;
  
  Future<void> setHasShownMicPermissionGuide(bool value) async {
    await _box?.put(_hasShownMicPermissionGuide, value);
  }

  String? get userName => _box?.get(_userName);
  
  Future<void> setUserName(String? name) async {
    if (name != null && name.trim().isNotEmpty) {
      await _box?.put(_userName, name.trim());
    }
  }

  DateTime? get lastVisit {
    final timestamp = _box?.get(_lastVisit);
    if (timestamp != null) {
      return DateTime.fromMillisecondsSinceEpoch(timestamp);
    }
    return null;
  }
  
  Future<void> updateLastVisit() async {
    await _box?.put(_lastVisit, DateTime.now().millisecondsSinceEpoch);
  }

  bool get isReturningUser => hasSeenWelcome && lastVisit != null;

  String getWelcomeGreeting() {
    final name = userName;
    final hour = DateTime.now().hour;
    
    String timeGreeting;
    if (hour < 12) {
      timeGreeting = "Mornin'";
    } else if (hour < 17) {
      timeGreeting = "Afternoon";
    } else {
      timeGreeting = "Evenin'";
    }
    
    if (name != null && name.isNotEmpty) {
      return "$timeGreeting, $name!";
    } else {
      return "Welcome back!";
    }
  }

  bool get isDarkMode => _box?.get(_isDarkMode, defaultValue: true) ?? true;
  
  Future<void> setDarkMode(bool value) async {
    await _box?.put(_isDarkMode, value);
    onThemeChanged?.call(value);
  }

  AustralianRegion get region {
    final value = _box?.get(_region, defaultValue: 0) ?? 0;
    if (value >= 0 && value < AustralianRegion.values.length) {
      return AustralianRegion.values[value];
    }
    return AustralianRegion.notSet;
  }
  
  Future<void> setRegion(AustralianRegion region) async {
    await _box?.put(_region, region.index);
  }

  RegionInfo get regionInfo => getRegionInfo(region);

  // Preferred greeting options: 'none', 'sister', 'brother', 'sista', 'bro', 'cuz', 'sis', 'friend', 'custom'
  String get preferredGreeting => _box?.get(_preferredGreeting, defaultValue: 'none') ?? 'none';
  
  Future<void> setPreferredGreeting(String value) async {
    await _box?.put(_preferredGreeting, value);
  }

  String? get customGreeting => _box?.get(_customGreeting);
  
  Future<void> setCustomGreeting(String? value) async {
    if (value != null && value.trim().isNotEmpty) {
      await _box?.put(_customGreeting, value.trim());
    } else {
      await _box?.delete(_customGreeting);
    }
  }

  String getGreetingTitle() {
    final greeting = preferredGreeting;
    if (greeting == 'custom') {
      return customGreeting ?? '';
    } else if (greeting == 'none') {
      return '';
    }
    // Capitalize first letter
    return greeting[0].toUpperCase() + greeting.substring(1);
  }

  String getWelcomeGreetingWithRegion() {
    final name = userName;
    final hour = DateTime.now().hour;
    final info = regionInfo;
    final greetingTitle = getGreetingTitle();
    
    String timeGreeting;
    if (hour < 12) {
      timeGreeting = "Mornin'";
    } else if (hour < 17) {
      timeGreeting = "Afternoon";
    } else {
      timeGreeting = "Evenin'";
    }
    
    // Build personalized greeting
    if (name != null && name.isNotEmpty) {
      if (greetingTitle.isNotEmpty) {
        return "$timeGreeting, $greetingTitle $name!";
      }
      return "$timeGreeting, $name!";
    } else if (greetingTitle.isNotEmpty) {
      return "$timeGreeting, $greetingTitle!";
    } else if (region != AustralianRegion.notSet) {
      return "$timeGreeting, ${info.selfIdentifyingTerm} friend!";
    } else {
      return "Welcome back!";
    }
  }

  Future<void> resetAll() async {
    await _box?.clear();
  }
}
