import 'dart:convert';
import 'package:shared_preferences/shared_preferences.dart';

class OfflineCacheService {
  static final OfflineCacheService _instance = OfflineCacheService._internal();
  factory OfflineCacheService() => _instance;
  OfflineCacheService._internal();

  SharedPreferences? _prefs;

  Future<void> initialize() async {
    _prefs ??= await SharedPreferences.getInstance();
  }

  Future<void> cacheResponse(String key, Map<String, dynamic> data) async {
    await initialize();
    _prefs?.setString('cache_$key', jsonEncode(data));
    _prefs?.setInt('cache_${key}_time', DateTime.now().millisecondsSinceEpoch);
  }

  Future<Map<String, dynamic>?> getCachedResponse(String key, {int maxAgeHours = 24}) async {
    await initialize();
    final cached = _prefs?.getString('cache_$key');
    final cacheTime = _prefs?.getInt('cache_${key}_time');
    
    if (cached != null && cacheTime != null) {
      final age = DateTime.now().millisecondsSinceEpoch - cacheTime;
      if (age < maxAgeHours * 60 * 60 * 1000) {
        return jsonDecode(cached);
      }
    }
    return null;
  }

  static const Map<String, dynamic> emergencyContacts = {
    'crisis': [
      {'name': '13YARN', 'phone': '13 92 76', 'description': 'Aboriginal crisis support, 24/7', 'mobRun': true},
      {'name': 'Lifeline', 'phone': '13 11 14', 'description': '24/7 crisis support', 'mobRun': false},
      {'name': 'Kids Helpline', 'phone': '1800 55 1800', 'description': 'For young people under 25', 'mobRun': false},
      {'name': 'MensLine', 'phone': '1300 78 99 78', 'description': 'Support for men', 'mobRun': false},
    ],
    'food': [
      {'name': 'Aboriginal Hostels Limited', 'phone': '1800 010 298', 'description': 'Accommodation and meals for mob', 'mobRun': true},
      {'name': 'Foodbank', 'phone': '1800 032 633', 'description': 'Emergency food relief', 'mobRun': false},
      {'name': 'OzHarvest', 'phone': '1800 108 006', 'description': 'Free food rescue', 'mobRun': false},
      {'name': 'Salvos', 'phone': '13 72 58', 'description': 'Food, clothing, bills', 'mobRun': false},
      {'name': 'Vinnies', 'phone': '13 18 12', 'description': 'Food, clothing, furniture', 'mobRun': false},
    ],
    'centrelink': [
      {'name': 'Centrelink Indigenous Line', 'phone': '1800 136 380', 'description': 'Dedicated line for Aboriginal and Torres Strait Islander people', 'mobRun': false},
      {'name': 'Welfare Rights Centre', 'phone': '1800 226 028', 'description': 'Free Centrelink advocacy', 'mobRun': false},
      {'name': 'Aboriginal Legal Services', 'phone': '1800 732 273', 'description': 'Free legal help for mob', 'mobRun': true},
    ],
    'housing': [
      {'name': 'Aboriginal Housing', 'phone': '1800 727 555', 'description': 'Housing for Aboriginal people', 'mobRun': true},
      {'name': 'Tenants Union', 'phone': '1800 251 101', 'description': 'Tenancy rights advice', 'mobRun': false},
      {'name': 'Homelessness Hotline', 'phone': '1800 474 753', 'description': 'Emergency housing help', 'mobRun': false},
    ],
    'legal': [
      {'name': 'Aboriginal Legal Services', 'phone': '1800 732 273', 'description': 'Free legal help for mob', 'mobRun': true},
      {'name': 'Legal Aid', 'phone': '1300 888 529', 'description': 'Free legal assistance', 'mobRun': false},
      {'name': 'Community Legal Centres', 'phone': '1300 888 529', 'description': 'Local legal help', 'mobRun': false},
    ],
    'health': [
      {'name': 'Aboriginal Medical Service', 'phone': '1800 190 000', 'description': 'Culturally safe health care', 'mobRun': true},
      {'name': 'Nurse on Call', 'phone': '1300 60 60 24', 'description': '24/7 health advice', 'mobRun': false},
      {'name': 'Poisons Info', 'phone': '13 11 26', 'description': 'Poisoning emergencies', 'mobRun': false},
    ],
    'family': [
      {'name': '1800RESPECT', 'phone': '1800 737 732', 'description': 'Family violence support', 'mobRun': false},
      {'name': 'Kids Helpline', 'phone': '1800 55 1800', 'description': 'Support for young people', 'mobRun': false},
      {'name': 'Parentline', 'phone': '1300 30 1300', 'description': 'Parenting support', 'mobRun': false},
    ],
  };

  static const Map<String, List<Map<String, String>>> offlineRights = {
    'centrelink': [
      {'title': 'Get it in writing', 'content': 'You have the right to ask for any decision in writing. This protects you and helps with appeals.'},
      {'title': 'Internal review', 'content': 'You can request an internal review of any decision within 13 weeks. They must reconsider.'},
      {'title': 'AAT appeal', 'content': 'After internal review, you can appeal to the Administrative Appeals Tribunal. This is free.'},
      {'title': 'Have support present', 'content': 'You can have an advocate, support person, or interpreter at any meeting.'},
      {'title': 'Cultural obligations', 'content': 'Sorry business, cultural ceremonies, and caring for family are valid reasons for missing appointments.'},
      {'title': 'Special circumstances', 'content': 'Centrelink must consider special circumstances like disability, family violence, or remote location.'},
    ],
    'police': [
      {'title': 'Right to silence', 'content': 'You have the right to remain silent. You only need to give your name and address.'},
      {'title': 'Right to a lawyer', 'content': 'You can ask for a lawyer before answering questions. Aboriginal Legal Services: 1800 732 273.'},
      {'title': 'Right to an interpreter', 'content': 'If English is hard, you can ask for an interpreter.'},
      {'title': 'Right to contact someone', 'content': 'You can make a phone call to family, friend, or lawyer if detained.'},
      {'title': 'Custody notification', 'content': 'If arrested, police must notify the Aboriginal Legal Service in most states.'},
    ],
    'housing': [
      {'title': 'Notice periods', 'content': 'Landlords must give proper notice before eviction. Usually 60-90 days depending on reason.'},
      {'title': 'Repairs', 'content': 'Landlords must fix urgent repairs within 24-48 hours, other repairs within 14 days.'},
      {'title': 'Bond protection', 'content': 'Your bond is held by the government, not the landlord. You have rights to get it back.'},
      {'title': 'Discrimination', 'content': 'It is illegal to refuse housing based on race, disability, or having children.'},
      {'title': 'Entry notice', 'content': 'Landlords must give at least 24-48 hours notice before entering, except emergencies.'},
    ],
    'school': [
      {'title': 'Absences', 'content': 'Cultural ceremonies and sorry business are valid reasons for absences. Get a note from an Elder.'},
      {'title': 'Suspension', 'content': 'Schools must follow proper process before suspension. You can appeal.'},
      {'title': 'Bullying', 'content': 'Schools have a duty to prevent bullying and racism. Document incidents.'},
      {'title': 'Support services', 'content': 'Your child may be entitled to Aboriginal Education Workers and extra support.'},
    ],
    'employment': [
      {'title': 'Discrimination', 'content': 'Employers cannot discriminate based on race, disability, or caring responsibilities.'},
      {'title': 'Cultural leave', 'content': 'Some workplaces offer cultural leave for sorry business and ceremonies. Ask your employer.'},
      {'title': 'Unfair dismissal', 'content': 'If sacked unfairly, you may be able to make a claim within 21 days.'},
      {'title': 'Wage theft', 'content': 'If not paid correctly, contact Fair Work: 13 13 94.'},
    ],
  };

  static const Map<String, String> offlineScripts = {
    'centrelink_breach': "I'm calling about a compliance failure notice I received. I'd like to understand why I was breached and what my options are. Can you please explain the decision and tell me how to request an internal review?",
    'centrelink_payment_stopped': "My payment has stopped and I need to understand why. Can you please check my account and explain what's happened? I'd also like to know if I can get an urgent payment while this is sorted out.",
    'centrelink_debt': "I received a debt notice and I don't agree with it. Can you please explain how this debt was calculated? I'd like to request a formal review of this decision.",
    'centrelink_appointment': "I missed my appointment due to [cultural/family/health reasons]. I'd like to explain my circumstances and reschedule. I understand there may be a compliance failure but I have special circumstances.",
    'landlord_repairs': "I'm calling about repairs that haven't been fixed. I notified you on [date] about [issue]. Under the Residential Tenancies Act, urgent repairs must be fixed within 24-48 hours. I need this addressed today.",
    'landlord_entry': "I'm concerned about the notice I received about entering my property. Under the Act, you must give [24/48] hours notice. Please reschedule with proper notice.",
    'police_rights': "I understand you need my name and address, but I choose to remain silent for other questions. I'd like to contact Aboriginal Legal Services before answering anything else.",
    'school_bullying': "I'm calling about ongoing bullying my child is experiencing. I've documented [number] incidents since [date]. I'd like to meet with the principal to discuss the school's anti-bullying policy and what steps will be taken.",
    'employer_unfair': "I believe I've been treated unfairly at work because of [reason]. I'd like to discuss this and understand the grievance process. I may also contact Fair Work if we can't resolve it.",
  };

  static String getOfflineFallbackAdvice(String category) {
    switch (category) {
      case 'food':
        return "When you get a connection, I can give you personalized tips. For now: Call Foodbank on 1800 032 633, or the Salvos on 13 72 58. For mob, Aboriginal Hostels on 1800 010 298 can help with food and a place to stay. These services don't need ID or a referral - just call and explain your situation.";
      case 'centrelink':
        return "When you get a connection, I can give you step-by-step help. For now: The Centrelink Indigenous Line is 1800 136 380 - they're set up to help mob. Always ask for decisions in writing. You have the right to internal review within 13 weeks. Welfare Rights Centre (1800 226 028) gives free Centrelink help.";
      case 'housing':
        return "When you get a connection, I can give you specific advice. For now: Know that landlords must give proper notice and fix repairs promptly. Tenants Union (1800 251 101) gives free advice. For emergency housing, call 1800 474 753. Aboriginal Housing (1800 727 555) helps mob find safe homes.";
      case 'legal':
        return "When you get a connection, I can explain your situation. For now: Aboriginal Legal Services (1800 732 273) provides free legal help for mob. Remember: you have the right to silence and the right to a lawyer. Don't sign anything you don't understand.";
      case 'health':
        return "When you get a connection, I can help you prepare. For now: Aboriginal Medical Services provide culturally safe care - search for your local AMS. Nurse on Call (1300 60 60 24) gives 24/7 health advice. For emergencies, always call 000.";
      default:
        return "I'm having trouble connecting right now. Try: 13YARN (13 92 76) for Aboriginal support, Lifeline (13 11 14) for crisis help, or check back when you have better signal.";
    }
  }
}
