import 'dart:io';
import 'dart:convert';
import 'package:flutter/foundation.dart';
import 'package:http/http.dart' as http;
import 'package:flutter_dotenv/flutter_dotenv.dart';

class OCRService {
  static const String _baseUrl = 'https://vision.googleapis.com/v1/images:annotate';
  static const int _maxFileSizeBytes = 10 * 1024 * 1024;

  Future<String> extractTextFromImage(File imageFile) async {
    try {
      debugPrint('OCR: Starting text extraction for ${imageFile.path}');

      if (!await validateImage(imageFile)) {
        throw Exception('Invalid image file');
      }

      final bytes = await imageFile.readAsBytes();
      final base64Image = base64Encode(bytes);
      debugPrint('OCR: Image encoded (${bytes.length} bytes)');

      final apiKey = dotenv.env['GOOGLE_CLOUD_VISION_API_KEY'];
      if (apiKey == null || apiKey.isEmpty) {
        throw Exception('Google Cloud Vision API key not configured in .env');
      }

      final response = await http.post(
        Uri.parse('$_baseUrl?key=$apiKey'),
        headers: {'Content-Type': 'application/json'},
        body: jsonEncode({
          'requests': [
            {
              'image': {'content': base64Image},
              'features': [
                {
                  'type': 'DOCUMENT_TEXT_DETECTION',
                  'maxResults': 1,
                }
              ],
            }
          ],
        }),
      ).timeout(
        const Duration(seconds: 30),
        onTimeout: () => throw Exception('OCR request timed out'),
      );

      if (response.statusCode != 200) {
        debugPrint('OCR Error: ${response.statusCode}');
        debugPrint('Response: ${response.body}');
        throw Exception('OCR API failed with status ${response.statusCode}');
      }

      final data = jsonDecode(response.body);
      final responses = data['responses'] as List?;
      
      if (responses == null || responses.isEmpty) {
        debugPrint('OCR: Empty response from API');
        return '';
      }

      final textAnnotations = responses[0]['textAnnotations'];
      
      if (textAnnotations == null || textAnnotations.isEmpty) {
        debugPrint('OCR: No text found in image');
        return '';
      }

      final extractedText = textAnnotations[0]['description'] as String;
      
      debugPrint('OCR: Extracted ${extractedText.length} characters');
      return extractedText.trim();
      
    } catch (e, stackTrace) {
      debugPrint('OCR Error: $e');
      debugPrint('Stack trace: $stackTrace');
      rethrow;
    }
  }

  Future<bool> validateImage(File imageFile) async {
    try {
      if (!await imageFile.exists()) {
        debugPrint('OCR: Image file does not exist');
        return false;
      }

      final fileSize = await imageFile.length();
      if (fileSize > _maxFileSizeBytes) {
        debugPrint('OCR: Image too large ($fileSize bytes, max $_maxFileSizeBytes)');
        return false;
      }

      final extension = imageFile.path.toLowerCase().split('.').last;
      final validExtensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp'];
      if (!validExtensions.contains(extension)) {
        debugPrint('OCR: Invalid image format ($extension)');
        return false;
      }

      return true;
    } catch (e) {
      debugPrint('OCR: Validation error: $e');
      return false;
    }
  }
}
