import 'package:flutter/foundation.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'dart:convert';

class NotificationService {
  static final NotificationService _instance = NotificationService._internal();
  factory NotificationService() => _instance;
  NotificationService._internal();

  SharedPreferences? _prefs;
  bool _initialized = false;

  Future<void> initialize() async {
    if (_initialized) return;
    _prefs = await SharedPreferences.getInstance();
    _initialized = true;
    debugPrint('NotificationService: Initialized (local storage mode)');
  }

  Future<void> scheduleMedicationReminder({
    required int id,
    required String medicationName,
    required int hour,
    required int minute,
    required List<int> days,
  }) async {
    await initialize();

    final reminder = MedicationReminder(
      id: id,
      medicationName: medicationName,
      hour: hour,
      minute: minute,
      days: days,
      isActive: true,
      createdAt: DateTime.now(),
    );

    await _saveReminder('med_$id', reminder.toJson());
    debugPrint('NotificationService: Saved medication reminder for $medicationName');
  }

  Future<void> scheduleAppointmentReminder({
    required int id,
    required String appointmentTitle,
    required DateTime appointmentTime,
    int reminderMinutesBefore = 60,
  }) async {
    await initialize();

    final reminder = AppointmentReminder(
      id: id,
      title: appointmentTitle,
      appointmentTime: appointmentTime,
      reminderMinutesBefore: reminderMinutesBefore,
      isActive: true,
    );

    await _saveReminder('apt_$id', reminder.toJson());
    debugPrint('NotificationService: Saved appointment reminder for $appointmentTitle');
  }

  Future<void> cancelReminder(String type, int id) async {
    await initialize();
    await _prefs?.remove('reminder_${type}_$id');
  }

  Future<void> cancelAllReminders() async {
    await initialize();
    final keys = _prefs?.getKeys().where((k) => k.startsWith('reminder_')) ?? [];
    for (final key in keys) {
      await _prefs?.remove(key);
    }
  }

  Future<List<MedicationReminder>> getMedicationReminders() async {
    await initialize();
    final reminders = <MedicationReminder>[];
    final keys = _prefs?.getKeys().where((k) => k.startsWith('reminder_med_')) ?? [];
    
    for (final key in keys) {
      final json = _prefs?.getString(key);
      if (json != null) {
        try {
          reminders.add(MedicationReminder.fromJson(jsonDecode(json)));
        } catch (e) {
          debugPrint('NotificationService: Error parsing reminder $key: $e');
        }
      }
    }
    
    reminders.sort((a, b) {
      final aMinutes = a.hour * 60 + a.minute;
      final bMinutes = b.hour * 60 + b.minute;
      return aMinutes.compareTo(bMinutes);
    });
    
    return reminders;
  }

  Future<List<AppointmentReminder>> getAppointmentReminders() async {
    await initialize();
    final reminders = <AppointmentReminder>[];
    final keys = _prefs?.getKeys().where((k) => k.startsWith('reminder_apt_')) ?? [];
    
    for (final key in keys) {
      final json = _prefs?.getString(key);
      if (json != null) {
        try {
          final reminder = AppointmentReminder.fromJson(jsonDecode(json));
          if (reminder.appointmentTime.isAfter(DateTime.now())) {
            reminders.add(reminder);
          }
        } catch (e) {
          debugPrint('NotificationService: Error parsing reminder $key: $e');
        }
      }
    }
    
    reminders.sort((a, b) => a.appointmentTime.compareTo(b.appointmentTime));
    
    return reminders;
  }

  Future<void> _saveReminder(String key, Map<String, dynamic> data) async {
    await _prefs?.setString('reminder_$key', jsonEncode(data));
  }

  String formatTime(int hour, int minute) {
    final period = hour >= 12 ? 'pm' : 'am';
    final displayHour = hour > 12 ? hour - 12 : (hour == 0 ? 12 : hour);
    final displayMinute = minute.toString().padLeft(2, '0');
    return '$displayHour:$displayMinute $period';
  }

  String formatDays(List<int> days) {
    const dayNames = ['', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'];
    if (days.length == 7) return 'Every day';
    if (days.length == 5 && !days.contains(6) && !days.contains(7)) return 'Weekdays';
    if (days.length == 2 && days.contains(6) && days.contains(7)) return 'Weekends';
    return days.map((d) => dayNames[d]).join(', ');
  }
}

class MedicationReminder {
  final int id;
  final String medicationName;
  final int hour;
  final int minute;
  final List<int> days;
  final bool isActive;
  final DateTime createdAt;

  MedicationReminder({
    required this.id,
    required this.medicationName,
    required this.hour,
    required this.minute,
    required this.days,
    required this.isActive,
    required this.createdAt,
  });

  Map<String, dynamic> toJson() => {
    'id': id,
    'medicationName': medicationName,
    'hour': hour,
    'minute': minute,
    'days': days,
    'isActive': isActive,
    'createdAt': createdAt.toIso8601String(),
  };

  factory MedicationReminder.fromJson(Map<String, dynamic> json) => MedicationReminder(
    id: json['id'],
    medicationName: json['medicationName'],
    hour: json['hour'],
    minute: json['minute'],
    days: List<int>.from(json['days']),
    isActive: json['isActive'] ?? true,
    createdAt: DateTime.tryParse(json['createdAt'] ?? '') ?? DateTime.now(),
  );
}

class AppointmentReminder {
  final int id;
  final String title;
  final DateTime appointmentTime;
  final int reminderMinutesBefore;
  final bool isActive;

  AppointmentReminder({
    required this.id,
    required this.title,
    required this.appointmentTime,
    required this.reminderMinutesBefore,
    required this.isActive,
  });

  Map<String, dynamic> toJson() => {
    'id': id,
    'title': title,
    'appointmentTime': appointmentTime.toIso8601String(),
    'reminderMinutesBefore': reminderMinutesBefore,
    'isActive': isActive,
  };

  factory AppointmentReminder.fromJson(Map<String, dynamic> json) => AppointmentReminder(
    id: json['id'],
    title: json['title'],
    appointmentTime: DateTime.parse(json['appointmentTime']),
    reminderMinutesBefore: json['reminderMinutesBefore'],
    isActive: json['isActive'] ?? true,
  );
}
