import 'dart:io';
import 'dart:typed_data';
import 'package:flutter/foundation.dart';
import 'package:just_audio/just_audio.dart';
import 'package:speech_to_text/speech_to_text.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:path_provider/path_provider.dart';
import 'api_client.dart';

enum VoicePersona { aunty, uncle }

class ElevenLabsService {
  static final ElevenLabsService _instance = ElevenLabsService._internal();
  factory ElevenLabsService() => _instance;
  
  final AudioPlayer _audioPlayer = AudioPlayer();
  final SpeechToText _stt = SpeechToText();
  final ApiClient _apiClient = ApiClient();
  
  bool _isListening = false;
  bool _isSpeaking = false;
  bool _isInitialized = false;
  VoicePersona _persona = VoicePersona.aunty;
  
  bool get isListening => _isListening;
  bool get isSpeaking => _isSpeaking;
  bool get isInitialized => _isInitialized;
  VoicePersona get persona => _persona;
  
  ElevenLabsService._internal();
  
  Future<void> initialize() async {
    if (_isInitialized) return;
    
    try {
      _audioPlayer.playerStateStream.listen((state) {
        if (state.processingState == ProcessingState.completed) {
          _isSpeaking = false;
        }
      });
      _isInitialized = true;
      debugPrint('ElevenLabsService initialized successfully');
    } catch (e) {
      debugPrint('ElevenLabsService initialization error: $e');
    }
  }
  
  void setPersona(VoicePersona persona) {
    _persona = persona;
    debugPrint('Voice persona set to: ${persona.name}');
  }
  
  Future<bool> requestMicrophonePermission() async {
    debugPrint('Requesting microphone permission...');
    
    // Check current status first
    var status = await Permission.microphone.status;
    debugPrint('Current microphone permission status: $status');
    
    if (status.isGranted) {
      return true;
    }
    
    // If permanently denied, just return false - don't open settings automatically
    if (status.isPermanentlyDenied) {
      debugPrint('Microphone permission permanently denied');
      return false;
    }
    
    // Request permission
    status = await Permission.microphone.request();
    debugPrint('After request, microphone permission status: $status');
    
    return status.isGranted;
  }
  
  Future<bool> checkMicrophonePermission() async {
    final status = await Permission.microphone.status;
    return status.isGranted;
  }
  
  Future<void> speak(String text) async {
    debugPrint('speak() called with text: "${text.substring(0, text.length > 50 ? 50 : text.length)}..."');
    
    if (_isSpeaking) {
      debugPrint('Already speaking, stopping first...');
      await stop();
    }
    
    if (text.isEmpty) {
      debugPrint('Empty text, skipping');
      return;
    }
    
    _isSpeaking = true;
    
    try {
      debugPrint('Making TTS API request to /api/tts...');
      final response = await _apiClient.postRaw('/api/tts', data: {
        'text': text,
        'voice': _persona == VoicePersona.aunty ? 'female' : 'male',
      });
      
      debugPrint('TTS response: status=${response.statusCode}, hasData=${response.data != null}, dataLength=${response.data?.length ?? 0}');
      
      if (response.statusCode == 200 && response.data != null && response.data!.isNotEmpty) {
        final tempDir = await getTemporaryDirectory();
        final tempFile = File('${tempDir.path}/tts_audio_${DateTime.now().millisecondsSinceEpoch}.mp3');
        
        // Convert List<int> to Uint8List properly
        final Uint8List bytes = Uint8List.fromList(response.data!);
        await tempFile.writeAsBytes(bytes);
        
        debugPrint('Audio saved to: ${tempFile.path}, size: ${bytes.length} bytes');
        
        await _audioPlayer.setFilePath(tempFile.path);
        await _audioPlayer.play();
        debugPrint('Audio playback started');
      } else {
        debugPrint('TTS API error: status=${response.statusCode}, no data or empty response');
        _isSpeaking = false;
      }
    } catch (e, stackTrace) {
      debugPrint('TTS error: $e');
      debugPrint('Stack trace: $stackTrace');
      _isSpeaking = false;
    }
  }
  
  Future<void> stop() async {
    await _audioPlayer.stop();
    _isSpeaking = false;
  }
  
  Future<bool> startListening({
    required Function(String) onResult,
    Function(String)? onError,
  }) async {
    debugPrint('startListening() called');
    
    if (_isListening) {
      debugPrint('Already listening, returning false');
      return false;
    }
    
    final hasPermission = await requestMicrophonePermission();
    if (!hasPermission) {
      debugPrint('Microphone permission not granted');
      onError?.call('Microphone access needed. Check your settings and try again.');
      return false;
    }
    
    debugPrint('Microphone permission granted, initializing speech recognition...');
    
    final available = await _stt.initialize(
      onError: (error) {
        debugPrint('STT Error: ${error.errorMsg}');
        onError?.call(error.errorMsg);
        _isListening = false;
      },
      onStatus: (status) {
        debugPrint('STT Status: $status');
        if (status == 'done' || status == 'notListening') {
          _isListening = false;
        }
      },
    );
    
    if (!available) {
      debugPrint('Speech recognition not available on this device');
      onError?.call('Speech recognition not available on this device');
      return false;
    }
    
    debugPrint('Speech recognition initialized, starting to listen...');
    _isListening = true;
    
    await _stt.listen(
      onResult: (result) {
        debugPrint('STT result: finalResult=${result.finalResult}, words="${result.recognizedWords}"');
        if (result.finalResult) {
          onResult(result.recognizedWords);
          _isListening = false;
        }
      },
      localeId: 'en_AU',
      listenMode: ListenMode.dictation,
      cancelOnError: true,
      partialResults: true,
    );
    
    debugPrint('Listening started successfully');
    return true;
  }
  
  Future<void> stopListening() async {
    if (_isListening) {
      await _stt.stop();
      _isListening = false;
    }
  }
  
  Future<void> speakThenListen({
    required String prompt,
    required Function(String) onResult,
    Function(String)? onError,
  }) async {
    await speak(prompt);
    
    await Future.delayed(const Duration(milliseconds: 500));
    
    while (_isSpeaking) {
      await Future.delayed(const Duration(milliseconds: 100));
    }
    
    await Future.delayed(const Duration(seconds: 2));
    
    await startListening(onResult: onResult, onError: onError);
  }
}
