import 'dart:io';
import 'package:flutter/foundation.dart';
import 'package:image_picker/image_picker.dart';

class DocumentScannerService {
  static final DocumentScannerService _instance = DocumentScannerService._internal();
  factory DocumentScannerService() => _instance;
  DocumentScannerService._internal();

  final ImagePicker _picker = ImagePicker();

  Future<DocumentScanResult> captureDocument() async {
    try {
      final XFile? image = await _picker.pickImage(
        source: ImageSource.camera,
        maxWidth: 2048,
        maxHeight: 2048,
        imageQuality: 85,
      );

      if (image == null) {
        return DocumentScanResult(
          success: false,
          message: "No photo taken. Tap the camera button to try again.",
        );
      }

      return DocumentScanResult(
        success: true,
        imagePath: image.path,
        message: "Photo captured. On mobile, use your phone's built-in text recognition to copy the text, then paste it here. Or read it out loud using the microphone.",
        instructions: _getInstructions(),
      );
    } catch (e) {
      debugPrint('DocumentScanner: Error capturing image: $e');
      return DocumentScanResult(
        success: false,
        message: "Couldn't access the camera. Try the microphone to read it out instead.",
      );
    }
  }

  Future<DocumentScanResult> pickDocument() async {
    try {
      final XFile? image = await _picker.pickImage(
        source: ImageSource.gallery,
        maxWidth: 2048,
        maxHeight: 2048,
        imageQuality: 85,
      );

      if (image == null) {
        return DocumentScanResult(
          success: false,
          message: "No photo selected. Tap to choose a photo of your letter.",
        );
      }

      return DocumentScanResult(
        success: true,
        imagePath: image.path,
        message: "Photo loaded. Use your phone's built-in text feature to copy the text, then paste it here.",
        instructions: _getInstructions(),
      );
    } catch (e) {
      debugPrint('DocumentScanner: Error picking image: $e');
      return DocumentScanResult(
        success: false,
        message: "Couldn't load that photo. Try a different one or use the microphone to read it out.",
      );
    }
  }

  List<String> _getInstructions() {
    return [
      "1. Open your Photos app and find the letter photo",
      "2. Tap and hold on the text in the photo",
      "3. Select 'Copy All' or drag to select the text you need",
      "4. Come back here and paste it in the box",
      "5. Tap 'Explain This' and I'll break it down for you",
    ];
  }
}

class DocumentScanResult {
  final bool success;
  final String? imagePath;
  final String message;
  final List<String>? instructions;

  DocumentScanResult({
    required this.success,
    this.imagePath,
    required this.message,
    this.instructions,
  });
}
