import 'dart:convert';
import 'package:flutter/foundation.dart';
import 'package:flutter_dotenv/flutter_dotenv.dart';
import 'package:http/http.dart' as http;
import '../../features/letter_helper/models/document_type.dart';
import '../../features/letter_helper/models/document_analysis.dart';

class DocumentAnalysisService {
  static const String _openAiUrl = 'https://api.openai.com/v1/chat/completions';

  Future<DocumentAnalysis> analyzeDocument(String documentText) async {
    try {
      debugPrint('DocumentAnalysis: Starting analysis of ${documentText.length} chars');

      final apiKey = dotenv.env['OPENAI_API_KEY'];
      if (apiKey == null || apiKey.isEmpty) {
        throw Exception('OpenAI API key not configured');
      }

      final prompt = _buildAnalysisPrompt(documentText);

      final response = await http.post(
        Uri.parse(_openAiUrl),
        headers: {
          'Content-Type': 'application/json',
          'Authorization': 'Bearer $apiKey',
        },
        body: jsonEncode({
          'model': 'gpt-4o',
          'messages': [
            {
              'role': 'system',
              'content': '''You are a document analysis assistant for Indigenous Australians. 
Your job is to translate complex government and legal documents into plain, friendly Australian English.
Always be respectful, clear, and helpful. Use simple words. Avoid jargon.
Identify key dates, amounts, and required actions clearly.
Return your analysis as valid JSON only, no markdown.'''
            },
            {
              'role': 'user',
              'content': prompt
            }
          ],
          'temperature': 0.3,
          'max_tokens': 2000,
        }),
      ).timeout(
        const Duration(seconds: 60),
        onTimeout: () => throw Exception('Analysis request timed out'),
      );

      if (response.statusCode != 200) {
        debugPrint('Analysis Error: ${response.statusCode}');
        debugPrint('Response: ${response.body}');
        throw Exception('Analysis API failed with status ${response.statusCode}');
      }

      final data = jsonDecode(response.body);
      final content = data['choices'][0]['message']['content'] as String;
      
      debugPrint('DocumentAnalysis: Received response');
      
      return _parseAnalysisResponse(content, documentText);
      
    } catch (e, stackTrace) {
      debugPrint('DocumentAnalysis Error: $e');
      debugPrint('Stack trace: $stackTrace');
      rethrow;
    }
  }

  String _buildAnalysisPrompt(String documentText) {
    return '''
Analyze this document and return a JSON object with the following structure:

{
  "type": "centrelink" | "legal" | "housing" | "health" | "employment" | "financial" | "other",
  "urgency": "urgent" | "important" | "normal",
  "plainLanguageSummary": "A clear, friendly explanation in 2-3 sentences of what this document is about and what it means for the person",
  "keyInfo": [
    {
      "category": "Dates" | "Money" | "What Happened" | "What They Want",
      "label": "Short label",
      "value": "The actual value or info",
      "explanation": "Plain English explanation if needed"
    }
  ],
  "actionSteps": [
    {
      "stepNumber": 1,
      "title": "Short action title",
      "description": "Clear explanation of what to do",
      "timing": "TODAY" | "THIS WEEK" | "BY [DATE]" | "AFTER STEP X",
      "phoneNumber": "optional phone number if calling is needed"
    }
  ],
  "deadline": "YYYY-MM-DD if there's a deadline, otherwise null",
  "warnings": ["Any important warnings or things to watch out for"]
}

DOCUMENT TEXT:
$documentText

Return ONLY the JSON object, no other text or markdown.
''';
  }

  DocumentAnalysis _parseAnalysisResponse(String content, String originalText) {
    try {
      String cleanContent = content.trim();
      if (cleanContent.startsWith('```json')) {
        cleanContent = cleanContent.substring(7);
      }
      if (cleanContent.startsWith('```')) {
        cleanContent = cleanContent.substring(3);
      }
      if (cleanContent.endsWith('```')) {
        cleanContent = cleanContent.substring(0, cleanContent.length - 3);
      }
      cleanContent = cleanContent.trim();

      final json = jsonDecode(cleanContent) as Map<String, dynamic>;

      final typeStr = json['type'] as String? ?? 'other';
      final type = DocumentType.values.firstWhere(
        (e) => e.name == typeStr,
        orElse: () => DocumentType.other,
      );

      final urgencyStr = json['urgency'] as String? ?? 'normal';
      final urgency = UrgencyLevel.values.firstWhere(
        (e) => e.name == urgencyStr,
        orElse: () => UrgencyLevel.normal,
      );

      final keyInfoList = (json['keyInfo'] as List<dynamic>? ?? [])
          .map((item) => KeyInformation(
                category: item['category'] as String? ?? '',
                label: item['label'] as String? ?? '',
                value: item['value'] as String? ?? '',
                explanation: item['explanation'] as String?,
              ))
          .toList();

      final actionStepsList = (json['actionSteps'] as List<dynamic>? ?? [])
          .map((item) => ActionStep(
                stepNumber: item['stepNumber'] as int? ?? 0,
                title: item['title'] as String? ?? '',
                description: item['description'] as String? ?? '',
                timing: item['timing'] as String? ?? '',
                phoneNumber: item['phoneNumber'] as String?,
                callCoachScenarioId: item['callCoachScenarioId'] as String?,
              ))
          .toList();

      DateTime? deadline;
      if (json['deadline'] != null && json['deadline'] != 'null') {
        try {
          deadline = DateTime.parse(json['deadline'] as String);
        } catch (_) {}
      }

      final warnings = (json['warnings'] as List<dynamic>? ?? [])
          .map((w) => w.toString())
          .toList();

      return DocumentAnalysis(
        id: DateTime.now().millisecondsSinceEpoch.toString(),
        type: type,
        urgency: urgency,
        plainLanguageSummary: json['plainLanguageSummary'] as String? ?? 'Unable to summarize this document.',
        keyInfo: keyInfoList,
        actionSteps: actionStepsList,
        deadline: deadline,
        originalText: originalText,
        warnings: warnings,
        createdAt: DateTime.now(),
      );
    } catch (e) {
      debugPrint('Parse error: $e');
      return DocumentAnalysis(
        id: DateTime.now().millisecondsSinceEpoch.toString(),
        type: DocumentType.unknown,
        urgency: UrgencyLevel.normal,
        plainLanguageSummary: 'Sorry, I had trouble understanding this document. Try taking another photo or typing the text.',
        keyInfo: [],
        actionSteps: [],
        originalText: originalText,
        warnings: ['Could not fully analyze this document'],
        createdAt: DateTime.now(),
      );
    }
  }
}
