import 'aboriginal_english_service.dart';

class CrisisDetectionService {
  static final CrisisDetectionService _instance = CrisisDetectionService._internal();
  factory CrisisDetectionService() => _instance;
  CrisisDetectionService._internal();

  final AboriginalEnglishService _aeService = AboriginalEnglishService();

  CrisisResponse checkMessage(String message) {
    if (message.isEmpty) {
      return CrisisResponse(
        shouldIntervene: false,
        isCritical: false,
        message: null,
        suggestedAction: CrisisAction.none,
        crisisContacts: [],
      );
    }

    final assessment = _aeService.assessForCrisis(message);
    
    if (assessment.level == CrisisLevel.high) {
      return CrisisResponse(
        shouldIntervene: true,
        isCritical: true,
        message: _buildCriticalResponse(assessment.detected ?? 'distress'),
        suggestedAction: CrisisAction.offer13yarn,
        crisisContacts: _getCriticalContacts(),
      );
    }
    
    if (assessment.level == CrisisLevel.moderate) {
      return CrisisResponse(
        shouldIntervene: true,
        isCritical: false,
        message: assessment.recommendation ?? "It sounds like things are tough. I'm here to help, and so is 13YARN if you need someone to yarn to.",
        suggestedAction: CrisisAction.gentleReminder,
        crisisContacts: _getSupportContacts(),
      );
    }
    
    return CrisisResponse(
      shouldIntervene: false,
      isCritical: false,
      message: null,
      suggestedAction: CrisisAction.none,
      crisisContacts: [],
    );
  }

  String _buildCriticalResponse(String trigger) {
    final responses = {
      'suicide': "I hear you, and I'm glad you're talking about this. You matter, and there's help. 13YARN (13 92 76) are Aboriginal people who understand - they're available 24/7. Would you like me to show you how to call them?",
      'self harm': "Thank you for trusting me with this. You deserve support. 13YARN (13 92 76) are mob who get it - they're there day and night. Can I help you reach them?",
      'domestic violence': "Your safety matters most right now. 1800RESPECT (1800 737 732) and 13YARN (13 92 76) both have people who can help. Are you safe right now?",
      'family violence': "No one should have to go through this alone. 13YARN (13 92 76) understands and can help. 1800RESPECT (1800 737 732) is also there 24/7. Are you somewhere safe?",
      'unsafe': "I want to make sure you're okay. If you're in danger right now, call 000. Otherwise, 13YARN (13 92 76) are there anytime. Can you tell me more about what's happening?",
      'default': "What you're going through sounds really hard. 13YARN (13 92 76) are Aboriginal people who understand and are available 24/7. Would you like their number or some other support?",
    };
    
    final triggerLower = trigger.toLowerCase();
    for (final key in responses.keys) {
      if (triggerLower.contains(key)) {
        return responses[key]!;
      }
    }
    return responses['default']!;
  }

  List<CrisisContact> _getCriticalContacts() {
    return [
      CrisisContact(
        name: '13YARN',
        phone: '13 92 76',
        description: 'Aboriginal crisis support - 24/7',
        isMobRun: true,
        isPrimary: true,
      ),
      CrisisContact(
        name: 'Lifeline',
        phone: '13 11 14',
        description: 'Crisis support - 24/7',
        isMobRun: false,
        isPrimary: false,
      ),
      CrisisContact(
        name: '1800RESPECT',
        phone: '1800 737 732',
        description: 'Family violence - 24/7',
        isMobRun: false,
        isPrimary: false,
      ),
      CrisisContact(
        name: 'Emergency',
        phone: '000',
        description: 'Police, Ambulance, Fire',
        isMobRun: false,
        isPrimary: false,
      ),
    ];
  }

  List<CrisisContact> _getSupportContacts() {
    return [
      CrisisContact(
        name: '13YARN',
        phone: '13 92 76',
        description: 'Aboriginal support - 24/7',
        isMobRun: true,
        isPrimary: true,
      ),
      CrisisContact(
        name: 'Beyond Blue',
        phone: '1300 22 4636',
        description: 'Anxiety and depression',
        isMobRun: false,
        isPrimary: false,
      ),
    ];
  }
}

enum CrisisAction { none, gentleReminder, offer13yarn, immediateReferral }

class CrisisResponse {
  final bool shouldIntervene;
  final bool isCritical;
  final String? message;
  final CrisisAction suggestedAction;
  final List<CrisisContact> crisisContacts;

  CrisisResponse({
    required this.shouldIntervene,
    required this.isCritical,
    this.message,
    required this.suggestedAction,
    required this.crisisContacts,
  });
}

class CrisisContact {
  final String name;
  final String phone;
  final String description;
  final bool isMobRun;
  final bool isPrimary;

  CrisisContact({
    required this.name,
    required this.phone,
    required this.description,
    required this.isMobRun,
    required this.isPrimary,
  });
}
