import 'package:flutter/foundation.dart';
import 'package:dio/dio.dart';
import 'package:flutter_dotenv/flutter_dotenv.dart';

class ApiClient {
  static final ApiClient _instance = ApiClient._internal();
  factory ApiClient() => _instance;
  
  late final Dio _dio;
  
  ApiClient._internal() {
    final baseUrl = dotenv.env['API_BASE_URL'] ?? 'http://localhost:5000';
    
    _dio = Dio(BaseOptions(
      baseUrl: baseUrl,
      connectTimeout: const Duration(seconds: 30),
      receiveTimeout: const Duration(seconds: 30),
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
    ));
    
    _dio.interceptors.add(LogInterceptor(
      requestBody: true,
      responseBody: true,
    ));
    
    _dio.interceptors.add(InterceptorsWrapper(
      onError: (error, handler) {
        if (error.type == DioExceptionType.connectionTimeout ||
            error.type == DioExceptionType.receiveTimeout) {
          return handler.reject(error);
        }
        return handler.next(error);
      },
    ));
  }
  
  Dio get dio => _dio;
  
  /// Check if backend is reachable
  Future<bool> checkBackendHealth() async {
    try {
      final response = await _dio.get(
        '/api/health',
        options: Options(
          sendTimeout: const Duration(seconds: 3),
          receiveTimeout: const Duration(seconds: 3),
        ),
      );
      return response.statusCode == 200;
    } catch (e) {
      debugPrint('Backend health check failed: $e');
      return false;
    }
  }

  /// Safe POST with fallback message
  Future<Map<String, dynamic>> postSafe(
    String path,
    Map<String, dynamic> data, {
    Map<String, dynamic>? fallbackResponse,
  }) async {
    try {
      return await post(path, data);
    } catch (e) {
      debugPrint('API call failed ($path): $e');
      if (fallbackResponse != null) {
        return fallbackResponse;
      }
      // Default fallback with culturally appropriate language
      return {
        'response': "I'm having trouble connecting right now, but I'm here to help. "
            "You can still use the app - just type what you need and I'll do my best to assist."
      };
    }
  }
  
  Future<Map<String, dynamic>> get(
    String path, {
    Map<String, dynamic>? queryParameters,
    Options? options,
  }) async {
    final response = await _dio.get<Map<String, dynamic>>(
      path,
      queryParameters: queryParameters,
      options: options,
    );
    return response.data ?? {};
  }
  
  Future<Map<String, dynamic>> post(
    String path,
    Map<String, dynamic> data, {
    Map<String, dynamic>? queryParameters,
    Options? options,
  }) async {
    final response = await _dio.post<Map<String, dynamic>>(
      path,
      data: data,
      queryParameters: queryParameters,
      options: options,
    );
    return response.data ?? {};
  }
  
  Future<Response<List<int>>> postRaw(
    String path, {
    dynamic data,
    Map<String, dynamic>? queryParameters,
  }) async {
    return _dio.post<List<int>>(
      path,
      data: data,
      queryParameters: queryParameters,
      options: Options(
        responseType: ResponseType.bytes,
        contentType: 'application/json',
        headers: {
          'Accept': '*/*',
        },
      ),
    );
  }
  
  Future<Map<String, dynamic>> delete(
    String path, {
    Map<String, dynamic>? data,
    Options? options,
  }) async {
    final response = await _dio.delete<Map<String, dynamic>>(
      path,
      data: data,
      options: options,
    );
    return response.data ?? {};
  }
}
