class AboriginalEnglishService {
  static final AboriginalEnglishService _instance = AboriginalEnglishService._internal();
  factory AboriginalEnglishService() => _instance;
  AboriginalEnglishService._internal();

  static const Map<String, String> commonMisrecognitions = {
    'deadly': 'deadly',
    'deadly Lee': 'deadly',
    'dead Lee': 'deadly', 
    'deadly E': 'deadly',
    'mob': 'mob',
    'mop': 'mob',
    'more B': 'mob',
    'yarn': 'yarn',
    'yawn': 'yarn',
    'young': 'yarn',
    'yarning': 'yarning',
    'yawning': 'yarning',
    'country': 'Country',
    'cuntry': 'Country',
    'bro': 'bro',
    'brother': 'brother',
    'sis': 'sis',
    'sister': 'sister',
    'aunty': 'Aunty',
    'auntie': 'Aunty',
    'anti': 'Aunty',
    'uncle': 'Uncle',
    'unkle': 'Uncle',
    'elder': 'Elder',
    'older': 'Elder',
    'koori': 'Koori',
    'curry': 'Koori',
    'cory': 'Koori',
    'murri': 'Murri',
    'murray': 'Murri',
    'mary': 'Murri',
    'nunga': 'Nunga',
    'nung a': 'Nunga',
    'noongar': 'Noongar',
    'noon gar': 'Noongar',
    'palawa': 'Palawa',
    'pallawa': 'Palawa',
    'yolngu': 'Yolngu',
    'yol ngu': 'Yolngu',
    'centre link': 'Centrelink',
    'center link': 'Centrelink',
    'sent a link': 'Centrelink',
    'central ink': 'Centrelink',
    'center think': 'Centrelink',
    'gammon': 'gammon',
    'gamman': 'gammon',
    'game on': 'gammon',
    'gubba': 'gubba',
    'gaba': 'gubba',
    'goober': 'gubba',
    'sorry business': 'sorry business',
    'sorry busy': 'sorry business',
    'sorry biz': 'sorry business',
    'dreaming': 'Dreaming',
    'dream in': 'Dreaming',
    'lore': 'Lore',
    'law': 'Lore',
    'shame': 'shame',
    'shamed': 'shame',
    'shame job': 'shame job',
    'gonna': 'gonna',
    'going to': 'gonna',
    'wanna': 'wanna',
    'want to': 'wanna',
    'reckon': 'reckon',
    'recken': 'reckon',
    'recon': 'reckon',
    'arvo': 'arvo',
    'arfternoon': 'arvo',
    'cuppa': 'cuppa',
    'cup a': 'cuppa',
    'tucker': 'tucker',
    'talker': 'tucker',
    'taker': 'tucker',
    'yabber': 'yabber',
    'yapping': 'yabber',
    'humbugging': 'humbugging',
    'hum bugging': 'humbugging',
    'home bugging': 'humbugging',
    'flash': 'flash',
    'flashy': 'flash',
    'proper way': 'proper way',
    'proper way E': 'proper way',
    'true god': 'true god',
    'true God': 'true god',
    'blackfella': 'blackfella',
    'black fella': 'blackfella',
    'black fellow': 'blackfella',
    'whitefella': 'whitefella',
    'white fella': 'whitefella',
    'white fellow': 'whitefella',
    'mission': 'mission',
    'michelin': 'mission',
    'community': 'community',
    'communitiy': 'community',
    'outstation': 'outstation',
    'out station': 'outstation',
    'homeland': 'homeland',
    'home land': 'homeland',
    'bush': 'bush',
    'boosh': 'bush',
    'town camp': 'town camp',
    'towncamp': 'town camp',
    'remote': 'remote',
    'remot': 'remote',
  };

  static const Map<String, String> dialectVariations = {
    'bala': 'brother',
    'tidda': 'sister',
    'budoo': 'child',
    'boodoo': 'child',
    'jarjum': 'children',
    'gubbament': 'government',
    'gummint': 'government',
    'guvment': 'government',
    'binj': 'angry/stomach',
    'binjy': 'angry/stomach',
    'boomerang': 'boomerang',
    'bunji': 'friend',
    'bungy': 'friend',
    'cuz': 'cousin/friend',
    'couz': 'cousin/friend',
    'gunya': 'shelter/house',
    'gunyah': 'shelter/house',
    'goomee': 'father',
    'goomeewa': 'grandfather',
    'nanna': 'grandmother',
    'nannie': 'grandmother',
    'womba': 'crazy',
    'womba womba': 'very crazy',
  };

  static const List<String> crisisKeywords = [
    'kill myself',
    'kill me',
    'want to die',
    'wanna die',
    'end it',
    'end my life',
    'suicide',
    'suicidal',
    'not worth living',
    'better off dead',
    'no point living',
    'hurt myself',
    'self harm',
    'cutting myself',
    'overdose',
    'no one cares',
    'nobody cares',
    'all alone',
    'cant go on',
    "can't go on",
    'give up',
    'giving up',
    'hopeless',
    'no hope',
    'desperate',
    'scared',
    'terrified',
    'being hurt',
    'being hit',
    'hitting me',
    'hurting me',
    'domestic violence',
    'family violence',
    'abusing me',
    'abuse',
    'unsafe',
    "don't feel safe",
    'not safe',
    'dangerous situation',
    'threatening me',
    'threat',
    'emergency',
  ];

  static const List<String> distressIndicators = [
    'stressed',
    'anxious',
    'worried',
    'struggling',
    'hard time',
    'difficult',
    'overwhelmed',
    'too much',
    'cant cope',
    "can't cope",
    'breaking down',
    'falling apart',
    'lost',
    'confused',
    'dont know what to do',
    "don't know what to do",
    'need help',
    'need someone',
    'lonely',
    'isolated',
    'no support',
    'family problems',
    'money problems',
    'broke',
    'hungry',
    'no food',
    'homeless',
    'no place to stay',
    'kicked out',
    'evicted',
    'breached',
    'cut off',
    'payment stopped',
    'no money',
    'sick',
    'unwell',
    'pain',
    'hurting',
  ];

  String normalizeInput(String input) {
    if (input.isEmpty) return input;
    
    String normalized = input.trim();
    
    commonMisrecognitions.forEach((misrecognition, correction) {
      try {
        final pattern = RegExp(
          r'\b' + RegExp.escape(misrecognition) + r'\b', 
          caseSensitive: false
        );
        normalized = normalized.replaceAllMapped(pattern, (match) => correction);
      } catch (e) {
        normalized = normalized.replaceAll(misrecognition, correction);
      }
    });
    
    return normalized;
  }

  String processForDisplay(String input) {
    if (input.isEmpty) return input;
    
    String processed = input;
    
    commonMisrecognitions.forEach((misrecognition, correction) {
      try {
        final pattern = RegExp(
          r'\b' + RegExp.escape(misrecognition) + r'\b', 
          caseSensitive: false
        );
        processed = processed.replaceAllMapped(pattern, (match) => correction);
      } catch (e) {
        processed = processed.replaceAll(misrecognition, correction);
      }
    });
    
    return processed;
  }

  CrisisAssessment assessForCrisis(String input) {
    if (input.isEmpty) {
      return CrisisAssessment(
        level: CrisisLevel.none,
        detected: null,
        recommendation: null,
      );
    }
    
    final lowered = input.toLowerCase();
    
    for (final keyword in crisisKeywords) {
      if (lowered.contains(keyword.toLowerCase())) {
        return CrisisAssessment(
          level: CrisisLevel.high,
          detected: keyword,
          recommendation: "This sounds really serious. Would it be okay if I connected you with 13YARN? They're Aboriginal people who understand what you're going through. Call 13 92 76 anytime.",
        );
      }
    }
    
    for (final indicator in distressIndicators) {
      if (lowered.contains(indicator.toLowerCase())) {
        return CrisisAssessment(
          level: CrisisLevel.moderate,
          detected: indicator,
          recommendation: "It sounds like you're going through a tough time. Remember, 13YARN (13 92 76) is always there if you need someone to yarn to who gets it.",
        );
      }
    }
    
    return CrisisAssessment(
      level: CrisisLevel.none,
      detected: null,
      recommendation: null,
    );
  }

  String getContextPrompt() {
    return '''
You are speaking with an Aboriginal Australian. Understand and use these terms naturally:
- "mob" = family, community, or Aboriginal people in general
- "deadly" = great, awesome, excellent
- "yarn" = chat, talk, story
- "Country" = traditional Aboriginal lands (capital C, spiritual connection)
- "sorry business" = mourning, funeral ceremonies
- "shame" = embarrassment, often used as "shame job"
- "Aunty/Uncle" = respected Elder or community member (not necessarily related)
- "brother/sis/cuz" = extended family or close friends
- "gammon" = fake, pretend, or joking
- "Centrelink" = Australian welfare agency

Be warm, patient, and practical. Use "mate" naturally. Acknowledge cultural obligations as valid. Never be preachy or judgmental. Focus on solutions and empowerment.
''';
  }
}

enum CrisisLevel { none, moderate, high }

class CrisisAssessment {
  final CrisisLevel level;
  final String? detected;
  final String? recommendation;

  CrisisAssessment({
    required this.level,
    this.detected,
    this.recommendation,
  });
}
