# Flutter iOS Build Guide for TestFlight Distribution

A comprehensive guide to avoid common Flutter iOS build errors, based on real troubleshooting experience.

---

## Debugging Methodology - CRITICAL

**BEFORE adding new code to fix an error:**
1. **Check when the error was first reported** - Compare against the Error/Fix Log below
2. **Identify recent code changes** - What build introduced the problem? Check git commits
3. **Reverse-engineer the cause** - Did a recent "fix" break something else?
4. **Prefer reverting/adjusting recent changes** over adding new workaround code
5. **Only add new code if the error existed before recent changes**

This prevents code bloat and cascading fixes that introduce more bugs.

**How to use the error log:**
- When a new error appears, check if a recent fix might have caused it
- Look at "Files Changed" - if the same file was just modified, the fix may have broken something
- Mark regressions clearly (when a fix caused a new problem)

---

## Pre-Flight Checklist (Before Every Build)

**Version & Issues:**
- [ ] Build number bumped in pubspec.yaml?
- [ ] All reported issues fixed?
- [ ] No regressions from recent fixes?

**Testing:**
- [ ] Test key user flows end-to-end
- [ ] Multi-language works (if applicable)?
- [ ] Portrait AND landscape tested?

**Code Quality (iOS-specific):**
- [ ] No @freezed without code generation? (use plain Dart classes)
- [ ] No Flutter 3.27+ only APIs? (use `withOpacity()` not `withValues()`)
- [ ] All imports reference existing files?
- [ ] iOS 13.0+ target set in Podfile with post_install hook?
- [ ] Run `flutter analyze` to catch errors before building?

**Build:**
- [ ] Old build artifacts deleted?
- [ ] Team ID and Bundle ID configured?
- [ ] All environment variables in .env?

---

## Common Build Errors and Fixes

| Issue | Cause | Solution |
|-------|-------|----------|
| `kernel_snapshot_program failed` | Dart compilation error | Check syntax, remove @freezed, use withOpacity() not withValues() |
| `IPHONEOS_DEPLOYMENT_TARGET 9.0` | Pods using old iOS version | Add post_install hook to Podfile setting iOS 13.0 |
| Pod install fails | Missing flutter pub get | Delete ios/Podfile.lock, run `flutter pub get`, then `cd ios && pod install` |
| Code signing errors | Team ID not configured | Configure DEVELOPMENT_TEAM in Xcode or via xcodeproj |
| Missing .freezed.dart | Using freezed without build_runner | Convert to plain Dart classes (recommended) |
| Build rejected by TestFlight | Build number already used | Always increment build number - never reuse or go backwards |
| Beta App Review delays | Using External Testing | Use Internal Testing for immediate distribution |

---

## Code Patterns to Avoid

### Code Generation Pitfalls

```dart
// BAD - requires code generation
@freezed
class MyModel with _$MyModel {
  factory MyModel({required String id}) = _MyModel;
}

// GOOD - no code generation needed
class MyModel {
  final String id;
  MyModel({required this.id});
}
```

### Flutter Version Compatibility

```dart
// BAD - Flutter 3.27+ only
color.withValues(alpha: 0.5)

// GOOD - works in all versions
color.withOpacity(0.5)
```

---

## Video Playback Issues (For Video Apps)

| Issue | Cause | Solution |
|-------|-------|----------|
| Video freezing at end | `isCompleted` not firing for HLS streams | Add position-based fallback: trigger completion when within 500ms of end |
| Video stalling near end | Buffering stall not detected | Add stall detection: if position unchanged for 2-3 seconds at 85%+, force completion |
| Black screen at video end | Completion detection failing | Check both `isCompleted` flag AND position-based checks |
| Audio continues after video | Audio service not stopped | Call audio stop in completion handler |

### Video Display

| Issue | Cause | Solution |
|-------|-------|----------|
| Black bars on 16:9 video | Using BoxFit.contain for all videos | Use BoxFit.cover for wide videos (aspect ratio >= 1.5) on landscape devices |
| 4:3 video getting cropped | BoxFit.cover threshold too low | Only use cover for truly wide videos (16:9 = 1.78, threshold >= 1.5) |
| Portrait video cropped in landscape | Wrong BoxFit | Always use BoxFit.contain for portrait/square videos (aspect <= 1.3) |

### Video Completion Detection (Robust Pattern)
```
Primary: Check controller.value.isCompleted
Fallback 1: Position within 500ms of duration
Fallback 2: Not playing + within 2s of end
Fallback 3: 99%+ complete
Stall detection: Position unchanged for 2+ seconds at 85%+
Buffering stall: Buffering at 80%+ for 3+ seconds
```

### BoxFit Logic for Video
```
Portrait/square video (aspect <= 1.3): BoxFit.contain (show full video)
Wide landscape (aspect >= 1.5) + landscape device: BoxFit.cover (fill screen)
All other cases: BoxFit.contain
```

---

## Multi-Language Issues

| Issue | Cause | Solution |
|-------|-------|----------|
| Description not showing | Using wrong getter for localized content | Use language-aware getters like `getDescriptionForLanguage(lang)` |
| App crashes on older data | Localized getter returns null | Always null-coalesce: `getText() ?? ''` |
| Wrong language video plays | Not preloading language variants | Preload all language video URLs, not just default |
| Only default language preloaded | Preload logic incomplete | Loop through `availableLanguages` and preload each variant |

### Null-Safe Localized Content
```dart
// Always guard
getLocalizedText() ?? ''

// Check before display
if ((getText() ?? '').isNotEmpty)
```

---

## Regression Examples (Learn From These)

1. **Fixed 16:9 bleed → Broke 4:3 framing**
   - Problem: Set BoxFit.cover threshold too low (1.0)
   - Fix: Raised threshold to 1.5 (only affects truly wide videos)

2. **Improved completion detection → Bypassed question scheduling**
   - Problem: Removed unanswered question check from completion guard
   - Fix: Reinstated the guard - completion should wait for scheduled questions

3. **Added video preloading → Only preloaded default language**
   - Problem: Forgot to loop through language variants
   - Fix: Preload all variants from availableLanguages list

---

## iOS Deployment Target Configuration

Must be set in BOTH Podfile AND Xcode project settings:

```ruby
# Podfile
platform :ios, '13.0'

post_install do |installer|
  installer.pods_project.targets.each do |target|
    target.build_configurations.each do |config|
      config.build_settings['IPHONEOS_DEPLOYMENT_TARGET'] = '13.0'
    end
  end
end
```

---

## Recommended Bootstrap Script

```bash
#!/bin/bash
set -e

TEAM_ID="YOUR_TEAM_ID"
BUNDLE_ID="com.yourcompany.yourapp"

echo "=== iOS Build ==="

# Step 1: Create iOS project if missing
if [ ! -d "ios" ]; then
    flutter create . --platforms=ios --org com.yourcompany
fi

# Step 2: Write correct Podfile
cat > ios/Podfile << 'PODFILE'
platform :ios, '13.0'

ENV['COCOAPODS_DISABLE_STATS'] = 'true'

project 'Runner', {
  'Debug' => :debug,
  'Profile' => :release,
  'Release' => :release,
}

def flutter_root
  generated_xcode_build_settings_path = File.expand_path(File.join('..', 'Flutter', 'Generated.xcconfig'), __FILE__)
  unless File.exist?(generated_xcode_build_settings_path)
    raise "#{generated_xcode_build_settings_path} must exist"
  end
  File.foreach(generated_xcode_build_settings_path) do |line|
    matches = line.match(/FLUTTER_ROOT\=(.*)/)
    return matches[1].strip if matches
  end
  raise "FLUTTER_ROOT not found"
end

require File.expand_path(File.join('packages', 'flutter_tools', 'bin', 'podhelper'), flutter_root)

flutter_ios_podfile_setup

target 'Runner' do
  use_frameworks!
  use_modular_headers!
  flutter_install_all_ios_pods File.dirname(File.realpath(__FILE__))
end

post_install do |installer|
  installer.pods_project.targets.each do |target|
    flutter_additional_ios_build_settings(target)
    target.build_configurations.each do |config|
      config.build_settings['IPHONEOS_DEPLOYMENT_TARGET'] = '13.0'
    end
  end
end
PODFILE

# Step 3: Configure code signing
ruby << RUBY
require 'xcodeproj'
project = Xcodeproj::Project.open('ios/Runner.xcodeproj')
project.targets.each do |target|
  next unless target.name == 'Runner'
  target.build_configurations.each do |config|
    config.build_settings['DEVELOPMENT_TEAM'] = '$TEAM_ID'
    config.build_settings['CODE_SIGN_STYLE'] = 'Automatic'
    config.build_settings['PRODUCT_BUNDLE_IDENTIFIER'] = '$BUNDLE_ID'
    config.build_settings['IPHONEOS_DEPLOYMENT_TARGET'] = '13.0'
  end
end
project.save
RUBY

# Step 4: Install pods
cd ios && pod install && cd ..

# Step 5: Build
flutter clean
flutter pub get
flutter build ipa --release --export-options-plist=tooling/ExportOptions.plist

echo "=== Build Complete ==="
```

---

## ExportOptions.plist Template

```xml
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
    <key>method</key>
    <string>app-store-connect</string>
    <key>destination</key>
    <string>upload</string>
    <key>signingStyle</key>
    <string>automatic</string>
    <key>teamID</key>
    <string>YOUR_TEAM_ID</string>
</dict>
</plist>
```

---

## Quick Diagnostic Commands

```bash
# Check Flutter version
flutter --version

# Check for Dart analysis errors
flutter analyze

# Clean everything
flutter clean && rm -rf ios/Pods ios/Podfile.lock

# Reinstall dependencies
flutter pub get && cd ios && pod install && cd ..

# Full rebuild
flutter build ios

# Build IPA for TestFlight
flutter build ipa --release && open build/ios/archive/Runner.xcarchive
```

---

## Error/Fix Log Template

Copy this into your project and fill it in as you work:

| Build | Date | Error Reported | Fix Applied | Files Changed |
|-------|------|----------------|-------------|---------------|
| 1 | | | | |
| 2 | | | | |
| 3 | | | | |

---

## Best Practices

1. **Never do partial builds** - Always use the full bootstrap script
2. **Delete ios/ folder** when switching machines or after major changes
3. **Increment build number** for every TestFlight upload
4. **Use plain Dart classes** instead of freezed to avoid code generation issues
5. **Stick to stable Flutter APIs** - avoid using cutting-edge features
6. **Test on minimum iOS version** you're targeting
7. **Prefer reverting/adjusting recent changes** over adding new workaround code
8. **Track regressions** - when a fix breaks something else, document it

---

## Key Files Reference

**iOS Build:**
- ios/Podfile - iOS version target, pod configuration
- ios/Runner.xcodeproj - Xcode project settings, code signing
- pubspec.yaml - version/build number, dependencies

**Video Apps:**
- video_tool.dart - playback, completion detection, display
- sound_service.dart - audio management
- video_preload_service.dart - caching

**Survey Apps:**
- survey_dialog.dart - Begin Poll screen, descriptions
- pre_survey_onboarding.dart - onboarding flow, preloading
